<?php

namespace App\Services\PurchaseOrder\PurchaseOrderBuilder\Builders;

use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\Exceptions\LargePurchaseOrderException;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\PurchaseOrderBuilder;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\PurchaseOrderBuilderLine;
use Carbon\Carbon;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Collection;

/**
 * Class SalesBasedPurchaseOrderBuilder.
 */
class SalesBasedPurchaseOrderBuilder extends PurchaseOrderBuilder
{
    const ROUNDING_METHOD_HALF_UP = 'round_half_up';

    const ROUNDING_METHOD_HALF_DOWN = 'round_half_down';

    const ROUNDING_METHOD_ALWAYS_ROUND_DOWN = 'always_round_down';

    const ROUNDING_METHOD_ALWAYS_ROUND_UP = 'always_round_up';

    const ROUNDING_METHODS = [
        self::ROUNDING_METHOD_HALF_UP,
        self::ROUNDING_METHOD_ALWAYS_ROUND_DOWN,
        self::ROUNDING_METHOD_HALF_UP,
        self::ROUNDING_METHOD_HALF_DOWN,
        self::ROUNDING_METHOD_ALWAYS_ROUND_UP,
    ];

    const BUILDER_TYPE_TARGET_STOCK_LEVEL = 'target_stock_level';

    /**
     * @var Carbon
     */
    protected $salesHistoryStartDate;

    /**
     * @var Carbon
     */
    protected $salesHistoryEndDate;

    /**
     * @var int
     */
    protected $targetDaysOfStock = 60;

    /**
     * @var array
     */
    protected $salesHistoryFilters = [];

    /**
     * @var bool
     */
    protected $useLeadtime = false;

    /**
     * @var bool
     */
    protected $useMOQ = false;

    /**
     * @var mixed|string
     */
    protected $roundingMethod;

    /**
     * SalesBasedPurchaseOrderBuilder constructor.
     *
     *
     * @throws BindingResolutionException
     */
    public function __construct(
        Supplier $supplier,
        array $options
    ) {
        $this->salesHistoryStartDate = isset($options['sales_start_date']) ? Carbon::parse($options['sales_start_date']) : null;
        $this->salesHistoryEndDate = isset($options['sales_end_date']) ? Carbon::parse($options['sales_end_date']) : null;
        $this->targetDaysOfStock = $options['target_stock_days'] ?? 60;
        $this->salesHistoryFilters = $options['sales_history_filters'] ?? [];
        $this->useLeadtime = $options['use_leadtime'] ?? false;
        $this->useMOQ = $options['use_moq'] ?? false;
        $this->roundingMethod = $options['rounding_method'] ?? static::ROUNDING_METHOD_HALF_UP;

        parent::__construct(
            $supplier,
            $options['destination_warehouse_id'] ?? null,
            $options['product_filters'] ?? []
        );
    }

    public function getSalesHistoryFilters(): array
    {
        return $this->salesHistoryFilters;
    }

    protected function buildOrder(): array
    {
        $salesData = $this->purchaseOrders->getAverageDailySalesWithInventory(
            supplierId: $this->supplier->id,
            warehouseId: $this->destinationWarehouseId,
            historyStartDate: $this->salesHistoryStartDate,
            historyEndDate: $this->salesHistoryEndDate,
            productFilters: $this->getProductFilters(),
            salesHistoryFilters: $this->getSalesHistoryFilters()
        );

        $results = [];

        $supplierProducts = SupplierProduct::with(['supplierPricingTiers', 'product'])
            ->where('supplier_id', $this->supplier->id)
            ->whereIn('product_id', $salesData->pluck('product_id')->toArray())
            ->get()->all();

        $productIds = array_column($supplierProducts, 'product_id');
        $supplierProducts = array_combine($productIds, $supplierProducts);

        $sampleDays = $this->getSampleDays();

        foreach ($salesData as $forSku) {
            $dailyAverageConsumption = $forSku->quantity;
            if ($dailyAverageConsumption <= 0) {
                continue;
            }

            /** @var SupplierProduct $supplierProduct */
            $supplierProduct = $supplierProducts[$forSku->product_id];

            $leadtime = $this->useLeadtime ? ($supplierProduct->product_leadtime ?? 0) : 0;

            $inventoryAvailable = $forSku->inventory_available;

            $backordered = max(0, -($inventoryAvailable ?: 0));

            $incoming = $forSku->inventory_inbound;
            $inTransit = $forSku->inventory_in_transit;

            $calculatedQuantity = (($this->targetDaysOfStock + $leadtime) * $dailyAverageConsumption) - max(0, $inventoryAvailable) - $incoming - $inTransit + $backordered;

            $neededQuantity = $this->handleRounding($calculatedQuantity);

            if ($neededQuantity <= 0 || ($this->useMOQ && $neededQuantity < $supplierProduct->product_minimum_order_quantity)) {
                continue;
            }

            $results[] = new PurchaseOrderBuilderLine(
                supplierProduct: $supplierProduct,
                quantity: $neededQuantity,
                options: [
                    'sample_days' => $sampleDays,
                    'num_sales' => round($dailyAverageConsumption * $sampleDays),
                    'target_days_of_stock' => $this->targetDaysOfStock,
                    'leadtime' => $this->useLeadtime ? $leadtime : 0,
                    'daily_average_consumption' => $dailyAverageConsumption,
                    'num_backordered' => $backordered,
                    'num_ordered' => $incoming,
                    'num_stock' => $inventoryAvailable,
                    'rounding_method' => $this->roundingMethod ?? self::ROUNDING_METHOD_HALF_UP,
                    'quantity_calculated' => $calculatedQuantity,
                    'quantity_needed' => $neededQuantity,
                    'num_inbound' => $forSku->inventory_inbound,
                    'num_in_transit' => $forSku->inventory_in_transit,
                    'sales_order_numbers' => explode(',', $forSku->sales_order_numbers)
                ]
            );
        }

        return $results;
    }

    protected function getSampleDays(): ?int
    {
        if (! $this->salesHistoryStartDate || ! $this->salesHistoryEndDate) {
            return null;
        }

        return $this->salesHistoryStartDate->diffInDays($this->salesHistoryEndDate);
    }

    protected function handleRounding(float $value): int
    {
        switch ($this->roundingMethod) {
            case self::ROUNDING_METHOD_ALWAYS_ROUND_UP:
                return ceil($value);
            case self::ROUNDING_METHOD_ALWAYS_ROUND_DOWN:
                return floor($value);
            case self::ROUNDING_METHOD_HALF_DOWN:
                return round($value, 0, PHP_ROUND_HALF_DOWN);
            default:
                return round($value, 0, PHP_ROUND_HALF_UP);
        }
    }

    public function build(): Collection
    {
        $results = collect($this->buildOrder());

        if (($count = $results->count()) > self::MAX_PRODUCTS_COUNT) {
            throw new LargePurchaseOrderException($count);
        }

        return $results;
    }
}
