<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 12/22/20
 * Time: 3:05 PM.
 */

namespace App\Services\Product;

use App\Models\Product;
use App\Models\ProductImage;
use App\Models\ProductPricing;
use App\Models\ProductToCategory;
use Illuminate\Support\Facades\DB;

/**
 * Class ProductDuplicator.
 */
class ProductDuplicator
{
    const DUPLICATE_ATTRIBUTES = 'attributes';

    const DUPLICATE_CATEGORIES = 'categories';

    const DUPLICATE_IMAGES = 'images';

    const DUPLICATE_PRICING = 'pricing';

    const DUPLICATE_SUPPLIERS = 'suppliers';

    public function duplicate(Product $product, array $overrides = [], array $excludedRelations = []): Product
    {
        return DB::transaction(function () use ($product, $overrides, $excludedRelations) {
            $duplicated = $product->replicate($overrides);
            $duplicated->fill($overrides);
            $duplicated->save();

            return $this->duplicateRelations($duplicated, $product, $excludedRelations);
        });
    }

    protected function duplicateRelations(Product $duplicate, Product $original, array $excludedRelations = []): Product
    {
        if (! in_array(self::DUPLICATE_ATTRIBUTES, $excludedRelations)) {
            $this->copyAttributes($duplicate, $original);
        }

        if (! in_array(self::DUPLICATE_CATEGORIES, $excludedRelations)) {
            $this->copyCategories($duplicate, $original);
        }

        if (! in_array(self::DUPLICATE_IMAGES, $excludedRelations)) {
            $this->copyImages($duplicate, $original);
        }

        if (! in_array(self::DUPLICATE_PRICING, $excludedRelations)) {
            $this->copyPricing($duplicate, $original);
        }

        if (! in_array(self::DUPLICATE_SUPPLIERS, $excludedRelations)) {
            $this->copySuppliers($duplicate, $original);
        }

        $duplicate->refresh();

        return $duplicate;
    }

    public function copyAttributes(Product $duplicate, Product $original): void
    {
        $duplicate->productAttributeValues()->createMany(
            $original->productAttributeValues->toArray()
        );
    }

    public function copyCategories(Product $duplicate, Product $original): void
    {
        $categories = ProductToCategory::with([])->where('product_id', $original->id)->get();
        $categories->each(function (ProductToCategory $productToCategory) use ($duplicate) {
            ProductToCategory::with([])->create([
                'category_id' => $productToCategory->category_id,
                'product_id' => $duplicate->id,
                'is_primary' => $productToCategory->is_primary,
            ]);
        });
    }

    public function copyImages(Product $duplicate, Product $original): void
    {
        $images = ProductImage::with([])->where('product_id', $original->id)->get();
        $images->each(function (ProductImage $image) use ($duplicate) {
            $data = $image->getAttributes();
            $data['product_id'] = $duplicate->id;
            ProductImage::with([])->create($data);
        });
    }

    public function copyPricing(Product $duplicate, Product $original): void
    {
        $originalPricing = ProductPricing::with([])->where('product_id', $original->id)->get();
        $originalPricing->each(function (ProductPricing $pricing) use ($duplicate) {
            ProductPricing::with([])->create([
                'product_pricing_tier_id' => $pricing->product_pricing_tier_id,
                'product_id' => $duplicate->id,
                'price' => $pricing->price,
            ]);
        });
    }

    /**
     * @throws \Exception
     */
    public function copySuppliers(Product $duplicate, Product $original): void
    {
        throw new \Exception('This functionality is not implemented');
    }
}
