<?php

namespace App\Services\Product;

use App\Exceptions\KitWithMovementsCantBeChangedException;
use App\Exceptions\ProductBundleException;
use App\Helpers;
use App\Models\Attribute;
use App\Models\AttributeGroup;
use App\Models\Product;
use App\Models\ProductAttribute;
use App\Models\ProductBrand;
use App\Models\ProductCategory;
use App\Models\ProductComponent;
use App\Models\ProductImage;
use App\Models\ProductPricingTier;
use App\Models\Supplier;
use App\Models\SupplierPricingTier;
use App\Models\SupplierProduct;
use App\Observers\AddPackingSlipQueueObserver;
use App\Repositories\SupplierInventoryRepository;
use App\Response;
use Exception;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Arr;

class CreateUpdateProductService extends ProductService
{
    /**
     * Set product brand by brand name.
     */
    public function setBrandName(?string $brandName, bool $save = false): bool
    {
        if (! is_null($brandName)) {
            $productBrand = ProductBrand::with([])->firstOrCreate(['name' => $brandName]);

            $this->product->brand_id = $productBrand->id;
        }

        if ($save) {
            return $this->product->save();
        }

        return true;
    }

    /**
     * Set pricing tiers to the product.
     */
    public function setPricing(?array $prices, bool $detaching = true): ?array
    {
        if (is_null($prices)) {
            return null;
        }

        $productPricing = [];
        foreach ($prices as $pricing) {
            // get product pricing tier id
            $pricingTierId = $pricing['product_pricing_tier_id'] ??
                       ProductPricingTier::with([])->firstOrCreate(['name' => $pricing['product_pricing_tier_name']])->id;

            if (! empty($pricing['operation'])) {
                $detaching = false;

                if ($pricing['operation'] == Helpers::OPERATION_UPDATE_CREATE) {
                    $productPricing[$pricingTierId] = ['price' => $pricing['price']];
                } elseif ($pricing['operation'] == Helpers::OPERATION_DELETE) {
                    $this->product->productPricingTiers()->detach($pricingTierId);
                }
            } else {
                $productPricing[$pricingTierId] = ['price' => $pricing['price']];
            }
        }

        return $this->product->productPricingTiers()->sync($productPricing, $detaching);
    }

    /**
     * Set supplier products.
     */
    public function setSupplierProducts(?array $supplierProducts, bool $detaching = true): ?array
    {
        if (is_null($supplierProducts)) {
            return null;
        }

        if (count($supplierNames = array_column($supplierProducts, 'supplier_name'))) {
            $suppliersByName = Supplier::with([])->whereIn('name', $supplierNames)->get(['id', 'name']);
        }

        $suppliers = [];
        $supplierPricing = [];

        foreach ($supplierProducts as $supplierProduct) {
            $supplierId = $supplierProduct['supplier_id'] ?? $suppliersByName->firstWhere('name', $supplierProduct['supplier_name'])->id;
            if (! empty($supplierProduct['pricing'])) {
                $supplierProductPricing = collect($supplierProduct['pricing'])->mapWithKeys(function ($pricing, $index) use (&$supplierProduct) {
                    $key = $pricing['supplier_pricing_tier_id'] ??
                 SupplierPricingTier::with([])->firstOrCreate(['name' => $pricing['supplier_pricing_tier_name']])->id;

                    $supplierProduct['pricing'][$index]['supplier_pricing_tier_id'] = $key;

                    return [$key => ['price' => $pricing['price'] ?? 0]];
                })->toArray();
            }

            if (! empty($supplierProduct['operation'])) {
                $detaching = false;

                if ($supplierProduct['operation'] == Helpers::OPERATION_UPDATE_CREATE) {
                    $suppliers[$supplierId] = Arr::only($supplierProduct, ( new SupplierProduct([]) )->getFillable());
                } elseif ($supplierProduct['operation'] == Helpers::OPERATION_DELETE) {
                    $this->product->suppliers()->detach($supplierId);
                }
            } else {
                $suppliers[$supplierId] = Arr::only($supplierProduct, ( new SupplierProduct([]) )->getFillable());
            }

            $supplierPricing[] = [
                'sync' => $supplierProductPricing ?? [],
                'operation' => $supplierProduct['pricing'] ?? [],
                'supplier_id' => $supplierId,
            ];
        }

        $supplierInventoryRepo = new SupplierInventoryRepository();
        // sync suppliers and supplier product's pricing/inventory
        $synced = $this->product->suppliers()->sync($suppliers, $detaching);

        foreach ($this->product->load('supplierProducts')->supplierProducts as $supplierProduct) {
            // sync pricing
            $operations = collect($supplierPricing)->where('supplier_id', $supplierProduct->supplier_id)->toArray();
            foreach ($operations as $operation) {
                $supplierProduct->setPricing($detaching ? $operation['sync'] : $operation['operation'], $detaching);
            }

            // sync inventory
            $supplierInventoryRepo->initializeInventoryForSupplierProduct($supplierProduct);
        }

        // Clear supplier product's inventory for removed supplier products
        foreach ($synced['detached'] as $supplierId) {
            $supplierInventoryRepo->clearInventoryForSupplierProduct($supplierId, $this->product->id);
        }

        return $synced;
    }

    /**
     * Set bundle components.
     *
     * @throws ProductBundleException
     */
    public function setBundleComponents(?array $components): ?bool
    {
        if (is_null($components)) {
            return null;
        }

        foreach ($components as $component_data) {
            if ($product = Product::find($component_data['id'])) {
                if ($product->type != 'standard') {
                    throw new ProductBundleException("{$product->sku} is not a standard product and cannot be bundled.");
                }
            }

            $this->upsertComponent($component_data);
        }

        $existing_components = ProductComponent::where(['parent_product_id' => $this->product->id])->get();

        if ($existing_components) {
            foreach ($existing_components as $existing_component) {
                if (! in_array($existing_component->component_product_id, array_column($components, 'id'))) {
                    $existing_component->delete();
                }
            }
        }

        return true;
    }

    /**
     * Set kit components.
     *
     * @throws ProductBundleException
     * @throws KitWithMovementsCantBeChangedException
     */
    public function setKitComponents(?array $components): ?bool
    {
        if (is_null($components)) {
            return null;
        }

        if ($this->product->inventoryMovements()->count() > 0) {
            throw new KitWithMovementsCantBeChangedException("{$this->product->sku} has inventory movements and cannot be changed.  You will have to create a new kit or delete the inventory history of this item.");
        }

        foreach ($components as $componentData) {
            if ($product = Product::find($componentData['id'])) {
                if ($product->type !== Product::TYPE_STANDARD) {
                    throw new ProductBundleException("{$product->sku} is not a standard product and cannot be bundled.");
                }
            }

            $this->upsertComponent($componentData);
        }

        $existingComponents = ProductComponent::where('parent_product_id', $this->product->id)->get();

        if ($existingComponents) {
            foreach ($existingComponents as $existingComponent) {
                if (! in_array($existingComponent->component_product_id, array_column($components, 'id'))) {
                    $existingComponent->delete();
                }
            }
        }

        return true;
    }

    private function upsertComponent(array $componentData): void
    {
        // Select existing component, which prevents duplicate components from being added to the same product
        $component = ProductComponent::where([
            'parent_product_id' => $this->product->id,
            'component_product_id' => $componentData['id'],
        ])->first();

        if (! $component) {
            $component = new ProductComponent();
            $component->parent_product_id = $this->product->id;
            $component->component_product_id = $componentData['id'];
        }

        $component->quantity = $componentData['quantity'];
        $component->save();
    }

    /**
     * Assign product categories.
     */
    public function setCategories(?array $categories, bool $detaching = true): ?array
    {
        if (is_null($categories)) {
            return null;
        }

        $categoryIds = [];
        foreach ($categories as $category) {
            if (empty($category['category_id'])) {
                $categoryPath = str_replace(' ', '', $category['category_path']);
                $c = ProductCategory::with([])->whereRaw("REPLACE(`path`, ' ', '') = '{$categoryPath}'")->firstOrFail();

                $category['category_id'] = $c->id;
                unset($category['category_path']);
            }

            if (! empty($category['operation'])) {
                $detaching = false;

                if ($category['operation'] == Helpers::OPERATION_UPDATE_CREATE) {
                    $categoryIds[$category['category_id']] = ['is_primary' => $category['is_primary'] ?? false];
                } elseif ($category['operation'] == Helpers::OPERATION_DELETE) {
                    $this->product->categories()->detach($category['category_id']);
                }
            } else {
                $categoryIds[$category['category_id']] = ['is_primary' => $category['is_primary'] ?? false];
            }

            // Create product attributes for category attributes
            $cat = ProductCategory::with(['attributeGroups', 'attributes'])->findOrFail($category['category_id']);

            $cat->attributeGroups->each(function (AttributeGroup $group) {
                $productAttributes = $group->attributes()
                    ->whereDoesntHave('productAttributes')
                    ->get()
                    ->map(function (Attribute $attribute) use ($group) {
                        return [
                            'attribute_id' => $attribute->id,
                            'attribute_group_id' => $group->id,
                            'product_id' => $this->product->id,
                            'value' => null,
                        ];
                    })->toArray();

                foreach ($productAttributes as $productAttribute) {
                    ProductAttribute::with([])->create($productAttribute);
                }
            });

            $productAttributes = $cat->attributes()->whereDoesntHave('productAttributes', function (Builder $builder) {
                return $builder->where('product_id', $this->product->id);
            })->get()->map(function (Attribute $attribute) {
                return [
                    'attribute_id' => $attribute->id,
                    'attribute_group_id' => null,
                    'product_id' => $this->product->id,
                    'value' => null,
                ];
            })->toArray();

            foreach ($productAttributes as $productAttribute) {
                ProductAttribute::with([])->create($productAttribute);
            }
        }

        return $this->product->categories()->sync($categoryIds, $detaching);
    }

    /**
     * Assign attribute groups.
     */
    public function setAttributeGroups(?array $attributeGroups, string $operation = Helpers::OPERATION_SET): array|int|null
    {
        if (is_null($attributeGroups) || empty($attributeGroups)) {
            return null;
        }

        if ($operation == Helpers::OPERATION_SET) {
            return $this->product->attributeGroups()->sync($attributeGroups, true);
        }

        if ($operation == Helpers::OPERATION_APPEND) {
            return $this->product->attributeGroups()->sync($attributeGroups, false);
        }

        if ($operation == Helpers::OPERATION_DELETE) {
            return $this->product->attributeGroups()->detach($attributeGroups);
        }
    }

    /**
     * Set attributes.
     */
    public function setProductAttributes(?array $productAttributes, bool $detaching = true): ?array
    {
        if (is_null($productAttributes)) {
            return null;
        }

        $syncedAttributes = [];
        foreach ($productAttributes as $attribute) {
            $attributeId = $attribute['id'] ??
                     Attribute::with([])->firstOrCreate(['name' => $attribute['name']])->id;

            if (! empty($attribute['operation'])) {
                $detaching = false;

                if ($attribute['operation'] == Helpers::OPERATION_UPDATE_CREATE) {
                    $syncedAttributes[$attributeId] = [
                        'value' => $attribute['value'] ?? null,
                        'attribute_group_id' => $attribute['attribute_group_id'] ?? null,
                    ];
                } elseif ($attribute['operation'] == Helpers::OPERATION_DELETE) {
                    $this->product->productAttributes()->detach($attributeId);
                }
            } else {
                $syncedAttributes[$attributeId] = [
                    'value' => $attribute['value'] ?? null,
                    'attribute_group_id' => $attribute['attribute_group_id'] ?? null,
                ];
            }
        }

        // For product variants, ensure that shared attributes of the parent
        // are not deleted (SKU-2388).
        if ($this->product->is_variation && $this->product->parent->shared_children_attributes) {
            $updatingAttributeIds = array_keys($syncedAttributes);
            foreach ($this->product->parent->shared_children_attributes as $sharedAttributeId) {
                if (! in_array($sharedAttributeId, $updatingAttributeIds)) {
                    // A shared attribute isn't in the updating attributes,
                    // we add it back in
                    /** @var ProductAttribute $productAttribute */
                    $productAttribute = $this->product->productAttributeValues()->where('attribute_id', $sharedAttributeId)->first();
                    $syncedAttributes[$sharedAttributeId] = [
                        'value' => $productAttribute ? $productAttribute->value : null,
                        'attribute_group_id' => $productAttribute ? $productAttribute->attribute_group_id : null,
                    ];

                    $attribute = Attribute::with([])->findOrFail($sharedAttributeId);

                    Response::instance()->addWarning('Attribute '.$attribute->name.' is assigned to the Matrix product, unable to remove.', Response::CODE_MATRIX_ATTRIBUTE_NO_DELETE, 'SharedMatrixAttributeNotDeletable');
                }
            }
        }

        $sync = $this->product->productAttributes()->sync($syncedAttributes, $detaching);

        // reprint packing slips
        if (Attribute::query()->whereIn('id', array_keys($syncedAttributes))->where('available_for_templates')->exists()) {
            AddPackingSlipQueueObserver::productUpdated($this->product);
        }

        return $sync;
    }

    /**
     * Add images.
     *
     *
     * @throws FileNotFoundException
     * @throws Exception
     */
    public function addImages(?array $images): ?bool
    {
        if (is_null($images)) {
            return null;
        }

        if (count($images) == 0) {
            $this->product->images()->delete();

            return true;
        }

        $productImagesIds = [];
        $detaching = true;
        foreach ($images as $index => $image) {
            if (! empty($image['url'])) {
                // store image
                $imageUrl = Helpers::getImageUrl($image['url'], boolval($image['download'] ?? false));

                // add warning
                if ($imageUrl === false) {
                    $primaryImage = boolval($image['is_primary'] ?? false);
                    $message = $primaryImage ?
            __('messages.product.invalid_main_image_url', ['url' => $image['url']]) :
            __('messages.product.invalid_image_url', ['url' => $image['url'], 'index' => " (Index #$index)"]);

                    $data = ['index' => $index, 'url' => $image['url'], 'main_image' => $primaryImage];
                    if (! empty($image['id'])) {
                        $data['id'] = $image['id'];
                    }

                    Response::instance()->addWarning($message, Response::CODE_UNRESOLVABLE_IMAGE, "images.$index.url", $data);
                }
            }

            if (! empty($image['operation'])) {
                $detaching = false;
            }

            if (! empty($image['id'])) {
                $productImage = ProductImage::with([])->findOrFail($image['id']);

                if (empty($image['operation'])) {
                    $productImage->updateData($image, $imageUrl ?? null);
                } else {
                    if ($image['operation'] == Helpers::OPERATION_UPDATE_CREATE) {
                        $productImage->updateData($image, $imageUrl ?? null);
                    } elseif ($image['operation'] == Helpers::OPERATION_DELETE) {
                        $productImage->delete();
                    }
                }
            } else {
                $this->product->images()->upsert([
                    'product_id' => $this->product->id,
                    'url' => ($imageUrl ?? false) ?: $image['url'],
                    'name' => $image['name'] ?? null,
                    'sort_order' => $image['sort_order'] ?? null,
                    'is_primary' => $image['is_primary'] ?? false,
                    'resolvable' => boolval($imageUrl ?? false),
                ], ['name', 'sort_order', 'is_primary', 'resolvable']);
                $productImage = $this->product->images()->firstWhere(['url' => ($imageUrl ?? false) ?: $image['url']]);
            }

            $productImagesIds[] = $productImage->id;
        }

        // sync product images
        if ($detaching) {
            $deletedImages = ProductImage::with([])->where('product_id', $this->product->id)->whereNotIn('id', $productImagesIds)->get();
            $deletedImages->map->delete();
        }

        // If only 1 image remains, we make it the primary image
        if ($this->product->images()->count() === 1) {
            $this->product->images()->update(['is_primary' => true]);
        }

        return true;
    }

    /**
     * Set variations.
     *
     *
     * @throws FileNotFoundException
     */
    public function setVariations(?array $variations, bool $sync = false): ?bool
    {
        if (is_null($variations)) {
            return null;
        }

        foreach ($variations as $variation) {
            $this->addVariation($variation);
        }

        // Synchronise variations
        if ($sync) {
            // Break link for products not included
            // in the variations lists
            $ids = collect($variations)->map(function ($variation) {
                return $variation['id'] ?? null;
            })->filter(function ($variation) {
                return ! is_null($variation);
            })->unique()->toArray();

            $skus = collect($variations)->map(function ($variation) {
                return $variation['sku'] ?? null;
            })->filter(function ($variation) {
                return ! is_null($variation);
            })->unique()->toArray();

            $this->product->variations()
                ->whereNotIn('sku', $skus)
                ->whereNotIn('id', $ids)
                ->update(['parent_id' => null]);
        }

        return true;
    }

    /**
     * Add variation.
     *
     *
     * @throws FileNotFoundException
     */
    public function addVariation(array $variation)
    {
        /**
         * Get or Create variation  product by "id" or "sku"
         * If product variation exists, we will fill only attributes that send.
         * If new product, we will fill attributes from parent and variation.
         */
        if (! empty($variation['id'])) {
            $productVariation = Product::with([])->findOrFail($variation['id']);
            $productVariation->fill($variation);
        } else {
            $productVariation = Product::with([])->firstOrNew(['sku' => $variation['sku']], $variation);

            if (! $productVariation->exists) {
                $productVariationValues = array_merge(Arr::only($this->product->attributesToArray(), [
                    'brand_id',
                    'type',
                    'weight',
                    'weight_unit',
                    'length',
                    'width',
                    'height',
                    'dimension_unit',
                ]), $variation);

                $productVariation->fill($productVariationValues);
            }
        }

        $productVariation->parent_id = $this->product->id;
        $productVariation->save();

        // Get Syncing attributes
        $productVariation->setProductAttributes($variation['attributes'] ?? [], $productVariation->wasRecentlyCreated);
        //$productVariation->addImages($variation['images'] ?? []);
    }
}
