<?php

namespace App\Services\Magento;

use App\Integrations\Magento;
use App\Models\IntegrationInstance;
use App\Models\Magento\Order;
use App\Models\SalesOrderFulfillmentLine;
use Illuminate\Support\Str;

class SubmitTrackingInfo extends \App\Services\SalesOrderFulfillment\SubmitTrackingInfo
{
    /**
     * {@inheritDoc}
     */
    public function submit(): array
    {
        if (config('app.env') !== 'production') {
            //            return ['success' => true, 'message' => 'The fulfillment submitted successfully (just on production)'];
        }

        /** @var Order $magentoOrder */
        $magentoOrder = Order::with([])->where('sales_order_id', $this->salesOrderFulfillment->sales_order_id)->firstOrFail();

        // if it's mapped just mark as submitted
        if (collect($magentoOrder->fulfillments_map)->firstWhere('sku_fulfillment_id', $this->salesOrderFulfillment->id)) {
            $this->salesOrderFulfillment->markAsSubmittedToSalesChannel(now());

            return ['success' => true, 'message' => 'The fulfillment is already submitted'];
        }

        // the order is canceled
        if ($magentoOrder->json_object['status'] == 'canceled') {
            $this->salesOrderFulfillment->markAsSubmittedToSalesChannel(now());

            return ['success' => true, 'message' => 'The magento order is canceled (marked as submitted)'];
        }

        //TODO: bundle item
        if (collect($magentoOrder->json_object['items'])->where('product_type', 'bundle')->isNotEmpty()) {
            return ['success' => false, 'message' => 'Submit tracking for bundle items is still not supported'];
        }

        $integrationInstance = $this->salesOrderFulfillment->salesOrder->salesChannel->integrationInstance;
        // shopify fulfillment request
        $options = [
            'notify' => config('app.env') === 'production',
            'tracks' => [
                [
                    'carrier_code' => $this->salesOrderFulfillment->fulfilledShippingMethod ? $this->salesOrderFulfillment->fulfilledShippingMethod->shippingCarrier->name : $this->salesOrderFulfillment->fulfilled_shipping_method,
                    'title' => $this->salesOrderFulfillment->fulfilledShippingMethod ? $this->salesOrderFulfillment->fulfilledShippingMethod->name : $this->salesOrderFulfillment->fulfilled_shipping_method,
                    'track_number' => $this->salesOrderFulfillment->tracking_number,
                ],
            ],
            'items' => $this->salesOrderFulfillment->salesOrderFulfillmentLines->map(function (SalesOrderFulfillmentLine $fulfillmentLine) use ($magentoOrder) {
                // we send only the fulfillable quantity from the magento order payload
                $magentoLine = collect($magentoOrder->json_object['items'])->firstWhere('item_id', $fulfillmentLine->salesOrderLine->sales_channel_line_id);
                // the line does not found in the magento or the line is unfulfillable(already fulfilled or canceled)
                $fulfillableQuantity = $this->getLineFulfillableQuantity($magentoLine);
                if (! $magentoLine || $fulfillableQuantity == 0) {
                    return null;
                }

                $quantity = $fulfillmentLine->quantity;
                // if some quantity may have been fulfilled or refunded
                if ($quantity > $fulfillableQuantity) {
                    $quantity = $fulfillableQuantity;
                }

                return [
                    'order_item_id' => $magentoLine['parent_item_id'] ?? $magentoLine['item_id'],
                    'qty' => $quantity,
                ];
            })->filter()->values()->toArray(),
        ];

        $magento = new Magento($integrationInstance);

        // there are no lines for fulfilling that means that the lines are refunded or already fulfilled
        if (empty($options['items'])) {
            $shipmentLineIds = $this->salesOrderFulfillment->salesOrderFulfillmentLines->pluck('salesOrderLine.sales_channel_line_id');
            $shipmentLineIds = $shipmentLineIds->map(function ($lineId) use ($magentoOrder) {
                $magentoLine = collect($magentoOrder->json_object['items'])->firstWhere('item_id', $lineId);

                return array_filter([$magentoLine['item_id'], $magentoLine['parent_item_id'] ?? null]);
            })->flatten()->values()->toArray();
            // check if the lines are fulfilled
            $orderShipments = collect($this->getMagentoShipments($integrationInstance, $magentoOrder->json_object['entity_id']));
            // the shipment that has lines that we are trying to fulfill and it is not mapped before
            $magentoShipmentId = $orderShipments->filter(fn ($shipment) => collect($shipment['items'])->whereIn('order_item_id', $shipmentLineIds)->isNotEmpty())
                ->whereNotIn('entity_id', $magentoOrder->fulfillments_map->pluck('magento_shipment_id'))
                ->first()['entity_id'] ?? null;

            if ($magentoShipmentId) {
                $this->salesOrderFulfillment->markAsSubmittedToSalesChannel(now());
                $magentoOrder->mapFulfillmentToSkuFulfillment($magentoShipmentId, $this->salesOrderFulfillment->id);

                return ['success' => true, 'message' => 'the lines have already been fulfilled (marked as submitted)'];
            }

            return ['success' => false, 'message' => 'the fulfillment does not have any fulfillable line and the magento order does not have a shipment with these lines and the lines are not refunded'];
        }
        // send the shipment to Magento
        try {
            $magentoShipmentId = $magento->shipOrder($magentoOrder->json_object['entity_id'], $options);
        } catch (\Throwable $exception) {
            // some lines or all lines have been already fulfilled
            if (Str::contains($exception->getMessage(), "You can't create a shipment without products")) {
                // trying to get the magento shipments that contains these lines
                $orderShipments = collect($this->getMagentoShipments($integrationInstance, $magentoOrder->json_object['entity_id']));
                // the shipment that has lines that we are trying to fulfill and it is not mapped before
                $magentoShipmentId = $orderShipments->filter(fn ($shipment) => collect($shipment['items'])->whereIn('order_item_id', array_column($options['items'], 'order_item_id'))->isNotEmpty())
                    ->whereNotIn('entity_id', $magentoOrder->fulfillments_map->pluck('magento_shipment_id'))
                    ->first()['entity_id'] ?? null;
                if (! $magentoShipmentId) {
                    return ['success' => false, 'message' => 'the lines marked as fulfilled but the magento order does not have a shipment with these lines'];
                }
            } else {
                return ['success' => false, 'message' => $exception->getMessage()];
            }
        }
        // the fulfillment submitted successfully (or it was already submitted)
        $this->salesOrderFulfillment->markAsSubmittedToSalesChannel(now());
        $magentoOrder->mapFulfillmentToSkuFulfillment($magentoShipmentId, $this->salesOrderFulfillment->id);

        return ['success' => true, 'message' => 'The fulfillment submitted successfully'];
    }

    /**
     * Get Magento shipments to the given order
     */
    private function getMagentoShipments(IntegrationInstance $integrationInstance, int $magentoOrderId): array
    {
        $options['pageSize'] = 300; // maximum
        $options['filterGroups'][]['filters'][0] = [
            'field' => 'order_id',
            'conditionType' => 'eq',
            'value' => $magentoOrderId,
        ];

        return (new Magento($integrationInstance))->getShipments($options)->current()['items'];
    }

    private function getLineFulfillableQuantity(array $magentoLine)
    {
        return max($magentoLine['qty_ordered'] - $magentoLine['qty_canceled'] - $magentoLine['qty_shipped'], 0);
    }
}
