<?php

namespace App\Services\Magento;

use App\Integrations\Listings\ProductMapper;
use App\Jobs\Magento\MapSalesOrderLines;
use App\Models\IntegrationInstance;
use App\Models\Magento\Product;
use App\Models\ProductListing;
use App\Models\TaskStatus\TaskStatus;
use App\Response;
use App\Services\StockTake\OpenStockTakeException;
use Illuminate\Support\Facades\DB;
use InvalidArgumentException;
use JetBrains\PhpStorm\ArrayShape;

/**
 * Class MagentoListingMapper.
 */
class MagentoListingMapper implements ProductMapper
{
    /**
     * @var IntegrationInstance
     */
    protected $integrationInstance;

    /**
     * MagentoListingMapper constructor.
     */
    public function __construct(IntegrationInstance $integrationInstance)
    {
        $this->integrationInstance = $integrationInstance;
    }

    /**
     * @return string[]
     */
    public function validationRules(): array
    {
        return [
            'mapping' => 'required|array|min:1',
            'mapping.*.variant_id' => 'required_without:mapping.*.document_id|exists:magento_products,variant_id',
            'mapping.*.document_id' => 'required_without:mapping.*.variant_id|exists:magento_products,id',
            'mapping.*.product_id' => 'exists:products,id|nullable',
        ];
    }

    /**s
     *
     * @param array $mappings
     * @param TaskStatus|null $taskStatus
     *
     * @return array
     * @throws \Throwable
     */
    #[ArrayShape(['mapped' => 'int', 'unmapped' => 'int', 'errors' => 'array'])]
    public function mapListings(array $mappings, ?TaskStatus $taskStatus = null)
    {
        $variantIds = collect($mappings)->pluck('variant_id')
            ->filter(function ($row) {
                return (bool) $row;
            })
            ->map(function ($id) {
                return (int) $id;
            })
            ->toArray();

        $documentIds = collect($mappings)->pluck('document_id')
            ->filter(function ($row) {
                return (bool) $row;
            })->toArray();

        $magentoProducts = Product::with([])
            ->whereIn('variant_id', $variantIds)
            ->orWhereIn('id', $documentIds)
            ->get();

        $mappings = array_map(function ($mapping) use ($magentoProducts) {
            if (isset($mapping['variant_id'])) {
                return $mapping;
            }

            $match = $magentoProducts->firstWhere('id', $mapping['document_id']);

            if (! $match) {
                throw new InvalidArgumentException('Invalid document id: '.$mapping['document_id']);
            }
            $mapping['variant_id'] = $match->variant_id;

            return $mapping;
        }, $mappings);

        $mapped = 0;
        $unmapped = 0;
        $errors = [];
        foreach ($mappings as $mapping) {
            if (empty($mapping['product_id'])) {
                $this->unmapById($mapping['variant_id']);
                $unmapped++;

                continue;
            }

            $productDocument = $magentoProducts->firstWhere('variant_id', (int) $mapping['variant_id']);

            if (! $productDocument) {
                // Document does not exist, we skip (Kalvin)
                $errors[$mapping['variant_id']] = ["Skipping variant_id: {$mapping['variant_id']}, document doesn't exist.", Response::CODE_NOT_FOUND, $mapping['variant_id']];
                if ($taskStatus) {
                    $taskStatus->addErrorMessage("Skipping variant_id: {$mapping['variant_id']}, document doesn't exist.");
                }

                continue;
            }

            try {
                DB::transaction(function () use ($productDocument, $mapping) {
                    $this->mapListing($mapping, $productDocument);
                });
                $mapped++;
            } catch (OpenStockTakeException $openStockTakeException) {
                $errors[$mapping['variant_id']] = [$openStockTakeException->getMessage(), Response::CODE_OPEN_STOCK_TAKE, $mapping['variant_id']];
            }
        }

        return ['mapped' => $mapped, 'unmapped' => $unmapped, 'errors' => $errors];
    }

    /**
     * @throws \Exception
     */
    private function unmapById($id)
    {
        $productListing = ProductListing::with([])
            ->where('sales_channel_listing_id', $id)
            ->where('sales_channel_id', $this->integrationInstance->salesChannel->id)
            ->first();

        if ($productListing) {
            $this->unmapListings($productListing);
        }
    }

    private function mapListing(array $mapping, Product $product)
    {
        /** @var ProductListing $productListing */
        $productListing = ProductListing::query()->firstOrNew([
            'sales_channel_id' => $this->integrationInstance->salesChannel->id,
            'sales_channel_listing_id' => $product->variant_id,
        ]);

        // sales_channel_listing_id based on integration instance
        $productListing->product_id = $mapping['product_id'];
        $productListing->document_id = $product->id;
        $productListing->document_type = Product::class;
        $productListing->listing_sku = $product['sku'];
        $productListing->title = $product['name'];

        $productListing->save();

        $product->product = $productListing->id;
        $product->mapped_at = now();
        $product->update();

        $productListingStatus = $productListing->wasRecentlyCreated ? MapSalesOrderLines::STATUS_NEW : MapSalesOrderLines::STATUS_UPDATED;
        dispatch_sync(( new MapSalesOrderLines($this->integrationInstance, $productListing, $productListingStatus) )->onConnection('sync'));
    }

    public function unmapListings(ProductListing $productListing)
    {
        $productListing->delete();

        //        dispatch_sync(( new MapSalesOrderLines($this->integrationInstance, $productListing, MapSalesOrderLines::STATUS_DELETED) )->onConnection('sync'));
    }
}
