<?php

namespace App\Services\InventoryManagement;

use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Services\StockTake\OpenStockTakeException;
use Illuminate\Database\Eloquent\Builder;

/**
 * Class SalesOrderFifoExtractor
 */
class SalesOrderFifoExtractor extends InventoryManager
{
    /**
     * Attempts to release fifo layers
     * used by sales orders and put the
     * affected sales orders in the backorder
     * queue to the tune of the given quantity.
     */
    public function takeFifoFromSalesOrders(
        int $quantity,
        InventoryEvent $event,
        InventoryReductionCause $reductionCause = InventoryReductionCause::INCREASED_NEGATIVE_EVENT,
        array $excludedSources = []
    ): array {
        $appliedQuantity = 0;
        $gatheredLayers = [];

        SalesOrder::with(['salesOrderLines', 'salesOrderLines.inventoryMovements'])
            ->where('order_status', SalesOrder::STATUS_OPEN)
            ->whereHas('salesOrderLines', function (Builder $builder) use ($event, $excludedSources) {
                // take quantity from other lines
                if ($event instanceof SalesOrderLine) {
                    $builder->where('id', '!=', $event->id);
                }

                return $builder->where('product_id', $this->product->id)
                    ->where('warehouse_id', $this->warehouseId)
                    ->whereNotIn('id', $excludedSources)
                    ->whereHas('fifoLayers', function (Builder $builder) {
                        return $builder->where('product_id', $this->product->id);
                    });
            })->each(function (SalesOrder $salesOrder) use (&$appliedQuantity, &$gatheredLayers, $quantity, $reductionCause, $event) {

                // We end whenever we're done applying all the needed quantity.
                if ($appliedQuantity >= $quantity) {
                    return;
                }

                $layers = $this->applyQuantityToSalesOrder(
                    $salesOrder,
                    $quantity - $appliedQuantity,
                    $reductionCause,
                    $event
                );
                $appliedQuantity += collect($layers)->sum('quantity');

                array_push($gatheredLayers, ...$layers);
            });

        return $gatheredLayers;
    }

    private function applyQuantityToSalesOrder(
        SalesOrder $salesOrder,
        int $quantity,
        InventoryReductionCause $reductionCause,
        InventoryEvent $event,
    ): array
    {
        $usedQuantity = 0;
        $layers = [];

        $salesOrder->salesOrderLines()
            ->with(['backorderQueue', 'salesOrderFulfillmentLines'])
            ->where('product_id', $this->product->id)
            ->where('warehouse_id', $this->warehouseId)
            ->each(function (SalesOrderLine $orderLine) use (&$usedQuantity, &$layers, $quantity, $reductionCause, $event) {
                if ($usedQuantity >= $quantity) {
                    return;
                }

                if (! ($releaseableQuantity = $orderLine->inventoryMovements()
                    ->where(
                        'inventory_status',
                        InventoryMovement::INVENTORY_STATUS_RESERVED
                    )->sum('quantity'))) {
                    // The order line is already fulfilled and can't release quantities.
                    return;
                }

                /**
                 * The quantity releaseable for this line must not include backordered quantity for the line
                 * and any already fulfilled quantity.
                 * Neither must it exceed the overall quantity remaining to be released.
                 */
                $releaseableQuantity = max(0, min($releaseableQuantity - $orderLine->active_backordered_quantity - $orderLine->fulfilled_quantity, ($quantity - $usedQuantity)));
                if ($releaseableQuantity == 0) {
                    return; // There are no more quantities to release for this sales order line.
                }

                /**
                 * We have a quantity to release for the line,
                 * we need to iteratively swap layers (fifo -> backorder queue)
                 * for the total quantity releasable
                 */
                $processedFifoLayerIds = [];
                do {
                    /** @var SalesOrderLineLayer $lineLayer */
                    $lineLayer = SalesOrderLineLayer::with([])
                        ->where('sales_order_line_id', $orderLine->id)
                        ->where('layer_type', FifoLayer::class)
                        ->whereNotIn('layer_id', $processedFifoLayerIds)
                        ->when(!empty($this->applicableFifoLayersForNegativeEvents), function (Builder $builder) {
                            return $builder->whereIn('layer_id', $this->applicableFifoLayersForNegativeEvents);
                        })
                        ->first();

                    if ($lineLayer && $lineLayer->quantity < $releaseableQuantity) {
                        /**
                         * The sales order line is using a fifo layer but the
                         * quantity isn't enough to cover the entire quantity to
                         * be released. We release the entire usage on the fifo layer
                         * and put the sales order line into the backorder queue.
                         */

                        /** @var FifoLayer $fifoLayer */
                        // Get used fifo and release the quantity used.
                        $fifoLayer = $lineLayer->layer;
                        $fifoLayer->fulfilled_quantity = max(0, $fifoLayer->fulfilled_quantity - $lineLayer->quantity);
                        $fifoLayer->save();
                        // Keep track of the released fifo to be sent to the consumer.
                        $layers[] = [
                            'layer' => $fifoLayer,
                            'layer_type' => FifoLayer::class,
                            'quantity' => $lineLayer->quantity,
                            'avg_cost' => $fifoLayer->avg_cost,
                        ];

                        // We create the backorder queue
                        $backorderQueue = $this->createBackorderQueueForQuantity(
                            $lineLayer->quantity,
                            $orderLine,
                            $reductionCause,
                            $event
                        );

                        /**
                         * Next, we update the inventory movements to use the backorder layer
                         * not the fifo layer.
                         */
                        $this->moveOrderLineMovementsToBackorder($orderLine, $fifoLayer, $backorderQueue, null, $reductionCause);

                        // Update the releasable quantity
                        $releaseableQuantity -= $lineLayer->quantity;

                        // Update the used quantity
                        $usedQuantity += $lineLayer->quantity;

                        /**
                         * Since the fifo quantity has been put back and a backorder has been created
                         * for the line, we no longer need the line layer
                         */
                        SalesOrderLineLayer::with([])->where('id', $lineLayer->id)->delete();
                    } else {
                        /**
                         * In this case, either there is no line layer or the
                         * quantity on the layer is enough to cover the entire
                         * quantity needed. Either way, a backorder queue is needed
                         * for the entire releasable quantity.
                         */
                        // We create the backorder queue
                        $backorderQueue = $this->createBackorderQueueForQuantity(
                            $releaseableQuantity,
                            $orderLine,
                            $reductionCause,
                            $event
                        );

                        if ($lineLayer) {
                            /**
                             * Here, the line layer has a fifo quantity that is
                             * enough to cover the releasable quantity. We
                             * simply release the fifo and map the movements to
                             * the backorder queue.
                             */

                            /** @var FifoLayer $fifoLayer */
                            $fifoLayer = $lineLayer->layer;
                            $fifoLayer->fulfilled_quantity = max(0, $fifoLayer->fulfilled_quantity - $releaseableQuantity);
                            $fifoLayer->save();

                            /**
                             * Again, we keep track of the layer
                             * used for potential use in inventory movements.
                             */
                            $layers[] = [
                                'layer' => $fifoLayer,
                                'layer_type' => FifoLayer::class,
                                'quantity' => $releaseableQuantity,
                                'avg_cost' => $fifoLayer->avg_cost,
                            ];

                            /**
                             * Next, we move the inventory movement from the fifo layer
                             * to the line layer..
                             */
                            $this->moveOrderLineMovementsToBackorder(
                                $orderLine,
                                $lineLayer->layer,
                                $backorderQueue,
                                $releaseableQuantity,
                                $reductionCause
                            );

                            /**
                             * We update the line layer. Thus, we delete it if all the quantity is moved or
                             * update it's quantity if the move was a partial move.
                             */
                            if ($lineLayer->quantity == $releaseableQuantity) {
                                /**
                                 * All the quantity is released, no need for the line layer
                                 * at this point.
                                 */
                                SalesOrderLineLayer::with([])->where('id', $lineLayer->id)->delete();
                            } else {
                                /**
                                 * The quantity on the line layer is more than what's needed
                                 * for the release, we simply adjust the line layer quantity
                                 * to reflect the balance.
                                 */
                                $lineLayer->quantity -= $releaseableQuantity;
                                $lineLayer->save();
                            }
                        }

                        // Update used quantity
                        $usedQuantity += $releaseableQuantity;

                        // All releaseable quantity is released.
                        $releaseableQuantity = 0;
                    }

                    // We mark the fifo layer as processed.
                    if (isset($fifoLayer)) {
                        $processedFifoLayerIds[] = $fifoLayer->id;
                    }
                } while ($releaseableQuantity > 0);
            });

        return $layers;
    }

    /**
     * @throws OpenStockTakeException
     */
    private function moveOrderLineMovementsToBackorder(
        SalesOrderLine $orderLine,
        FifoLayer $fifoLayer,
        BackorderQueue $backorderQueue,
        ?int $quantityToMove = null,
        InventoryReductionCause $reductionCause = InventoryReductionCause::INCREASED_NEGATIVE_EVENT
    ) {
        if (is_null($quantityToMove)) {
            /**
             * No quantity is specified, we'll take it to
             * mean that we need to move the entire quantity
             * from the fifo layer to the backorder queue.
             */
            $orderLine->inventoryMovements()
                ->where('layer_type', FifoLayer::class)
                ->where('layer_id', $fifoLayer->id)
                ->where(function (Builder $builder) {
                    return $builder->where(function (Builder $builder) {
                        return $builder->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                            ->where('quantity', '>', 0);
                    })
                        ->orWhere(function (Builder $builder) {
                            return $builder->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                                ->where('quantity', '<', 0);
                        });
                })
                ->update(['layer_type' => BackorderQueue::class, 'layer_id' => $backorderQueue->id]);
            $quantityToMove = abs($orderLine->inventoryMovements()
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->where('layer_type', BackorderQueue::class)
                ->where('layer_id', $backorderQueue->id)
                ->sum('quantity'));
        } else {
            /**
             * We balance the reserved and active inventory
             */
            /** @var InventoryMovement $reservedMovement */
            $reservedMovement = $orderLine->inventoryMovements()
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                ->where('quantity', '>', 0)
                ->where('layer_type', FifoLayer::class)
                ->where('layer_id', $fifoLayer->id)
                ->firstOrFail();

            /** @var InventoryMovement $activeMovement */
            $activeMovement = $orderLine->inventoryMovements()
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->where('quantity', '<', 0)
                ->where('layer_type', FifoLayer::class)
                ->where('layer_id', $fifoLayer->id)
                ->firstOrFail();

            $movementQuantity = $reservedMovement->quantity;
            $reservedMovement->quantity = max(0, $movementQuantity - $quantityToMove);
            if ($reservedMovement->quantity == 0) {
                /**
                 * Looks like the entire quantity is being moved, we simply map
                 * the movement to the backorder queue for the original quantity
                 */
                $reservedMovement->quantity = $movementQuantity;
                $reservedMovement->backorder_queue = $backorderQueue->id;
                $reservedMovement->save();
                $activeMovement->quantity = -$movementQuantity;
            } else {
                /**
                 * We update the reserved and active movements
                 */
                $reservedMovement->save();

                $activeMovement->update(['quantity' => -$reservedMovement->quantity]);

                /**
                 * Finally, we create the split movements for the backorder queue
                 */
                $reservedMovement = $reservedMovement->replicate();
                $reservedMovement->quantity = $quantityToMove;
                $reservedMovement->backorder_queue = $backorderQueue->id;
                $reservedMovement->save();

                $activeMovement = $activeMovement->replicate();
                $activeMovement->quantity = -$quantityToMove;
            }
            $activeMovement->backorder_queue = $backorderQueue->id;
            $activeMovement->save();

            $this->combineMovementsByQuantity(
                $orderLine->id,
                SalesOrderLine::class,
                $backorderQueue->id,
                BackorderQueue::class,
            );
            // Since we just moved units from fifo to backorder queue,
            // we ensure that any prior releases and coverages are handled
            // for increases in negative event quantity.
            // Note that for positive event, these are handled when creating
            // the backorder queue.
            if($reductionCause === InventoryReductionCause::INCREASED_NEGATIVE_EVENT){
                $backorderQueue->released_quantity = max(0, $backorderQueue->released_quantity - $quantityToMove);
                $backorderQueue->save();

                (new BackorderManager)->syncCoveragesAndReleasesAfterPositiveEventReduction(
                    $backorderQueue,
                    $quantityToMove,
                    $fifoLayer->link
                );
            }
        }
    }

    protected function combineMovementsByQuantity($linkId, $linkType, $layerId, $layerType)
    {
        InventoryMovement::with([])
            ->where('layer_id', $layerId)
            ->where('layer_type', $layerType)
            ->where('link_id', $linkId)
            ->where('link_type', $linkType)
            ->get()
            ->groupBy('inventory_status')
            ->each(function ($movements) {
                if (count($movements) <= 1) {
                    return;
                }

                /**
                 * We simply retain the first record and set its quantity to
                 * the sum of all the quantities.
                 */
                $first = $movements->first();
                $first->quantity = $movements->sum('quantity');
                $first->save();

                InventoryMovement::with([])
                    ->whereIn('id', array_diff($movements->pluck('id')->toArray(), [$first->id]))
                    ->delete();
            });
    }
}
