<?php

namespace App\Services\InventoryManagement;

use App;
use App\Actions\InventoryHealth\FifoLayerInventoryMovementRelocator;
use App\Exceptions\InsufficientStockException;
use App\Jobs\ProcessNewInventory;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Repositories\InventoryMovementRepository;
use App\Services\Product\ReleaseBackorderQueues;
use App\Services\StockTake\OpenStockTakeException;
use Carbon\Carbon;
use Exception;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use InvalidArgumentException;
use Throwable;

/**
 * Class InventoryManager
 */
class InventoryManager
{
    protected Product $product;

    protected int $warehouseId;

    protected InventoryMovementRepository $inventoryMovementRepository;

    private BackorderManager $backorderManager;

    /**
     * A restriction on what fifo layers can be used for negative events.
     *
     * @var array|FifoLayer[]
     */
    protected array $applicableFifoLayersForNegativeEvents = [];

    protected FifoLayerInventoryMovementRelocator $fifoLayerInventoryMovementRelocator;

    /**
     * InventoryManager constructor.
     */
    public function __construct(
        int $warehouseId,
        Product $product,
        array $applicableFifoLayersForNegativeEvents = []
    )
    {
        $this->product = $product;
        $this->warehouseId = $warehouseId;
        $this->applicableFifoLayersForNegativeEvents = $applicableFifoLayersForNegativeEvents;
        $this->backorderManager = app(BackorderManager::class);
        $this->inventoryMovementRepository = app(InventoryMovementRepository::class);
        $this->fifoLayerInventoryMovementRelocator = app(FifoLayerInventoryMovementRelocator::class);
    }

    public static function with(
        int $warehouseId,
        Product $product,
        array $applicableFifoLayersForNegativeEvents = []
    ): self
    {
        return new static($warehouseId, $product, $applicableFifoLayersForNegativeEvents);
    }

    /**
     * Adds the given quantity to stock from the positive inventory event.
     *
     * @param  float|null  $unitCost null: use the default value of the event
     *
     * @throws Exception
     */
    public function addToStock(
        $quantity, PositiveInventoryEvent $event,
        bool $syncStockNow = true,
        bool $autoApplyStock = true,
        ?float $unitCost = null,
    ): void {
        /**
         * We create the fifo layer, the inventory movements
         * and attempt to release any backorder queues.
         */
        if (empty($unitCost)) {
            // Use waterfall approach for unit cost if not set.
            $unitCost = $this->product->getUnitCostAtWarehouse($this->warehouseId);
        }
        $event->createInventoryMovements(
            $quantity,
            $fifoLayer = $event->createFifoLayer($quantity, $unitCost, $this->product->id)
        );

        $this->afterInventoryEvent($fifoLayer, $fifoLayer->avg_cost);

        if ($syncStockNow) {
            if ($autoApplyStock) {
                $this->onStockAvailable($fifoLayer, $event, $fifoLayer->original_quantity);
            }
        } else {
            dispatch(
                new ProcessNewInventory($fifoLayer, $event)
            );
        }
    }

    /**
     * @param  array|FifoLayer  $fifo
     */
    public function afterInventoryEvent($fifo = null, ?float $additionalCost = 0)
    {
        /**
         * We make sure that fifo fulfilled quantity
         * is in sync with inventory movement quantities.
         */
        //        $this->checkFifoUsageWithMovements($fifo);
        /*
         * TODO: This is extremely inefficient.  There are likely multiple places this is being called.
         */
        (new UpdateProductsInventoryAndAvgCost([$this->product->id]))->handle();
    }

    public function getTallies(string $inventoryStatus): array
    {
        return $this->inventoryMovementRepository->getTallies(
            $this->warehouseId,
            $inventoryStatus,
            $this->product
        );
    }

    /**
     * @param  array|FifoLayer|null  $fifo
     */
    protected function checkFifoUsageWithMovements($fifo = null)
    {
        if (! $fifo) {
            return;
        }

        if ($fifo instanceof FifoLayer) {
            $this->checkSingleFifoUsageWithMovements($fifo);
        } else {
            foreach ($fifo as $fifoLayer) {
                $this->checkSingleFifoUsageWithMovements($fifoLayer);
            }
        }
    }

    protected function checkSingleFifoUsageWithMovements(FifoLayer $fifoLayer)
    {
        /**
         * The fulfilled quantity as measured by negative active
         * movements on the fifo layer must match the fulfilled
         * quantity cache on the fifo layer.
         */
        $fifoLayer->load('inventoryMovements')->refresh();
        $movementUsage = abs($fifoLayer->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '<', 0)
            ->sum('quantity'));
        if ($fifoLayer->fulfilled_quantity != $movementUsage) {
            throw new InvalidFifoLayerCacheException($fifoLayer, $movementUsage);
        }
    }

    /**
     * Attempts to remove all stock added by the given
     * positive inventory event from inventory.
     */
    public function removeAllStockFrom(PositiveInventoryEvent $event, string $inventoryDeficiencyActionType = InventoryDeficiencyActionType::ACTION_TYPE_EXCEPTION): void
    {
        if (($movement = $event->getOriginatingMovement()) && ($fifoLayer = $event->getFifoLayer())) {
            $this->reducePositiveEventQty($fifoLayer->fulfilled_quantity, $event, $inventoryDeficiencyActionType);
        }

        // Clean up
        $movement?->delete();
        if (isset($fifoLayer) && $fifoLayer) {
            $fifoLayer->delete();
        }

        $this->afterInventoryEvent();
    }

    /**
     * Increases the quantity of a positive inventory event.
     */
    public function increasePositiveEventQty(int $quantity, PositiveInventoryEvent $event, bool $autoApplyStock = true): void
    {
        DB::transaction(function () use ($quantity, $event, $autoApplyStock) {
            /**
             * We increase the original fifo layer quantity,
             * update the originating movement quantity and attempt
             * to release backorder queues by the increased quantity.
             */
            if ($movement = $event->getOriginatingMovement()) {
                $movement->quantity += $quantity;
                $movement->save();
            }

            if ($fifoLayer = $event->getFifoLayer()) {
                $fifoLayer->original_quantity += $quantity;
                $fifoLayer->save();

                if ($autoApplyStock) {
                    $this->onStockAvailable($fifoLayer, $event, $quantity);
                }
            }

            $this->afterInventoryEvent($fifoLayer, $fifoLayer->avg_cost);
        });
    }

    /**
     * Reduces the positive inventory event by the given quantity.
     */
    public function reducePositiveEventQty(int $quantity, PositiveInventoryEvent $event, $inventoryDeficiencyActionType = InventoryDeficiencyActionType::ACTION_TYPE_EXCEPTION): void
    {
        if ($quantity <= 0) {
            return;
        }

        $fifoLayer = $event->getFifoLayer();

        DB::transaction(function () use ($event, $quantity, $fifoLayer, $inventoryDeficiencyActionType) {
            /**
             * We attempt to charge any used stock on the fifo layer
             * to other layers (fifo or create backorder queues for
             * events that support backorder queues).
             *
             * Inventory movements must be carefully handled to ensure
             * that the applicable layers for the used fifo layer quantity
             * are mapped to the inventory movements.
             *
             * Note that active inventory movements with negative quantity
             * are the ones that draw from inventory. We call these deducting
             * inventory movements.
             */
            $originatingMovement = $event->getOriginatingMovement();
            $transactions = $fifoLayer->inventoryMovements()
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->where('quantity', '<', 0)
                ->get();

            if ($transactions->isEmpty()) {
                if ($fifoLayer->fulfilled_quantity > 0) {
                    /**
                     * This is a weird situation, the fifo has used
                     * quantity but no negative movements, we error out
                     * for debugging
                     */
                    throw new InvalidArgumentException("Fifo Layer: {$fifoLayer->id} is used but without negative movements.");
                } else {
                    /**
                     * The fifo layer has no transactions and not used,
                     * we simply reduce the original quantity on the fifo
                     * and the originating movement.
                     */
                    $fifoLayer->original_quantity = max(0, $fifoLayer->original_quantity - $quantity);
                    $fifoLayer->save();
                    $originatingMovement->quantity = max(0, $originatingMovement->quantity - $quantity);
                    $originatingMovement->save();
                }
            } else {
                /**
                 * For each deducting movement, we attempt to charge their quantities
                 * to other layers. This ensures that deleting the fifo layer for the
                 * positive event doesn't create discrepancy in inventory.
                 */
                /** @var InventoryMovement $movement */
                foreach ($transactions as $movement) {
                    /**
                     * We attempt to pass back the quantity on
                     * the transaction movement and map
                     * the applicable layers used. Note that multiple
                     * layers may be required in consuming the quantity.
                     *
                     * Also, since the originating fifo layer is what's
                     * being removed, it cannot be used to cater for the stock
                     * we need to charge back.
                     */
                    $applicableQuantity = min(abs($movement->quantity), $quantity);
                    $appliedLayers = $this->reduceInventory($applicableQuantity, $event, InventoryReductionCause::REDUCED_POSITIVE_EVENT, true, null, [$fifoLayer->id], $inventoryDeficiencyActionType);
                    $this->reassignLayers($movement, $appliedLayers);
                    $quantity = max(0, $quantity - $applicableQuantity);

                    $fifoLayer->original_quantity = max(0, $fifoLayer->original_quantity - $applicableQuantity);
                    $fifoLayer->fulfilled_quantity = max(0, $fifoLayer->fulfilled_quantity - $applicableQuantity);
                    $fifoLayer->save();
                    $originatingMovement->quantity = max(0, $originatingMovement->quantity - $applicableQuantity);
                    $originatingMovement->save();

                    if ($quantity == 0) {
                        break;
                    }
                }
            }
        });

        $this->afterInventoryEvent($fifoLayer);
    }

    /**
     * Takes from stock for the negative movement and returns the
     * layers involved in the process.
     *
     * @throws InsufficientStockException
     * @throws Throwable
     */
    public function takeFromStock(
        int $quantity,
        NegativeInventoryEvent $event,
        bool $fifoOnly = true,
        ?Model $backorderLink = null,
        string $inventoryDeficiencyAction = InventoryDeficiencyActionType::ACTION_TYPE_EXCEPTION,
        ?Carbon $dateOverride = null,
        array $excludedSources = []
    ): void {

        /**
         * We charge the quantity to stock and delegate to the event
         * to handle inventory movements.
         */
        $event->createInventoryMovementsForLayers(
            $layers = $this->reduceInventory(
                quantity: $quantity,
                event: $event,
                reductionCause: InventoryReductionCause::INCREASED_NEGATIVE_EVENT,
                fifoOnly: $fifoOnly,
                backorderLink: $backorderLink,
                inventoryDeficiencyActionType: $inventoryDeficiencyAction,
                excludedSources: $excludedSources
            ),
            $dateOverride
        );

        $this->afterInventoryEvent(
            $this->getFifosFromLayers($layers)
        );
    }

    protected function getFifosFromLayers(array $layers): Arrayable
    {
        return collect($layers)->where('layer_type', FifoLayer::class)
            ->map(function ($layerInfo) {
                return $layerInfo['layer'];
            });
    }

    /**
     * Completely reverses a negative inventory event.
     *
     *
     * @throws Exception
     */
    public function reverseNegativeEvent(NegativeInventoryEvent $event, bool $autoApplyStock = true): void
    {
        $this->decreaseNegativeEventQty(
            abs(collect($event->getReductionActiveMovements()->toArray())->sum('quantity')),
            $event,
            $autoApplyStock
        );
    }

    /**
     * Increases a negative event quantity
     *
     *
     * @throws InsufficientStockException|Throwable
     */
    public function increaseNegativeEventQty(
        int $quantity,
        NegativeInventoryEvent $event,
        bool $fifoOnly = true,
        ?Model $backorderLink = null,
        array $excludeFifoLayers = []
    ): void {
        $event->createInventoryMovementsForLayers(
            $layers = $this->reduceInventory($quantity, $event, InventoryReductionCause::INCREASED_NEGATIVE_EVENT, $fifoOnly, $backorderLink, $excludeFifoLayers)
        );

        $this->afterInventoryEvent(
            $this->getFifosFromLayers($layers)
        );
    }

    /**
     * Handles the decrease in negative event quantity.
     *
     *
     * @throws Exception
     */
    public function decreaseNegativeEventQty(int $quantity, NegativeInventoryEvent $event, bool $autoApplyStock = true): void
    {
        /** @var InventoryMovement $movement */
        foreach ($event->getReductionActiveMovements() as $movement) {
            if ($movement->quantity >= 0) {
                continue;
            }
            $applicableQuantity = min(abs($movement->quantity), $quantity);
            $layer = $movement->fifo_layer ?: $movement->backorder_queue;
            // Update the quantity on the inventory movement(s)
            $event->reduceQtyWithSiblings($applicableQuantity, $movement);
            // Reduce layer usage.
            $this->reduceLayerUsage($layer, $applicableQuantity, $event);

            if ($layer instanceof FifoLayer && $autoApplyStock) {
                $this->onStockAvailable($layer, $event, $quantity);
            }

            $quantity = max(0, $quantity - $applicableQuantity);
            if ($quantity == 0) {
                break;
            }
        }

        $this->afterInventoryEvent();
    }

    /**
     * @param $layer
     * @param  int  $quantity
     * @param  NegativeInventoryEvent  $event
     * @return void
     * @throws App\Exceptions\OversubscribedFifoLayerException
     * @throws Throwable
     */
    protected function reduceLayerUsage($layer, int $quantity, NegativeInventoryEvent $event): void
    {
        if ($layer instanceof FifoLayer) {
            $layer->fulfilled_quantity = max(0, $layer->fulfilled_quantity - $quantity);
            $layer->save();

            // If the event has a backorder history, we reduce the quantity of the history.
            if($event instanceof SalesOrderLine && ($queue = $event->backorderQueue)){

                /** @var PositiveInventoryEvent $positiveEvent */
                $positiveEvent = $layer->link;

                $this->reduceBackorderQueueQuantity($queue, $quantity);

                // We handle coverages and releases here as the client code may
                // not know that the reduction was done on a Fifo Layer.
                // Unlike below where $layer is BackorderQueue, it will be obvious from context.
                (new BackorderManager)->syncCoveragesAndReleasesAfterPositiveEventReduction(
                    $queue,
                    $quantity,
                    $positiveEvent
                );
            }

        } elseif ($layer instanceof BackorderQueue) {
            /**
             * Reduce the backordered quantity and move any extra coverages
             * to other backorder queues. Note that handling of backorder
             * coverages and releases are not core inventory actions and are
             * delegated to client code.
             */
            $this->reduceBackorderQueueQuantity($layer, $quantity);
        }
    }

    private function reduceBackorderQueueQuantity(BackorderQueue $queue, int $quantity): void{
        $queue->backordered_quantity = max(0, $queue->backordered_quantity - $quantity);
        if ($queue->backordered_quantity == 0) {
            $queue->delete();
        } else {
            $queue->save();
        }
    }

    /**
     * This method attempts to reduce the inventory by
     * the provided quantity and returns the layers (fifo or backorder)
     * used in the reduction process with their corresponding quantities applied.
     * Note that when $fifoOnly is true, it means the reduction can only be applied
     * to fifo layers and not backorder queues. In such a situation, we may attempt to
     * strip fifo quantities from existing sales orders if necessary to handle the reduction
     * and put the affected sales orders in the backorder queue.
     *
     * @throws InsufficientStockException
     * @throws Exception
     * @throws Throwable
     */
    public function reduceInventory(
        int $quantity,
        NegativeInventoryEvent|PositiveInventoryEvent $event,
        InventoryReductionCause $reductionCause,
        bool $fifoOnly = true,
        ?Model $backorderLink = null,
        array $excludeFifoLayers = [],
        string $inventoryDeficiencyActionType = InventoryDeficiencyActionType::ACTION_TYPE_EXCEPTION,
        array $excludedSources = []
    ): array {
        if ($quantity <= 0) {
            return [];
        }

        $layers = [];

        DB::transaction(function () use (
            $quantity,
            $event,
            $reductionCause,
            $fifoOnly,
            $backorderLink,
            $excludeFifoLayers,
            $inventoryDeficiencyActionType,
            &$layers,
            $excludedSources
        ) {
            do {

                /** @var FifoLayer $activeFifoLayer */
                $activeFifoLayer = $this->product->activeFifoLayers()
                    ->where('warehouse_id', $this->warehouseId)
                    ->whereNotIn('id', $excludeFifoLayers)
                    ->when(!empty($this->applicableFifoLayersForNegativeEvents), function ($query) {
                        $query->whereIn('id', $this->applicableFifoLayersForNegativeEvents);
                    })
                    ->first();

                if ($activeFifoLayer) {
                    $quantityApplied = $this->chargeQuantityToFifoLayer($activeFifoLayer, $quantity);

                    $layers[] = [
                        'layer' => $activeFifoLayer,
                        'layer_type' => FifoLayer::class,
                        'quantity' => $quantityApplied,
                        'avg_cost' => $activeFifoLayer->avg_cost,
                    ];

                    // Update the quantity
                    $quantity = max(0, $quantity - $quantityApplied);
                } else {
                    if ($this->applicableFifoLayersForNegativeEvents) {
                        if (count($this->applicableFifoLayersForNegativeEvents) > 1) {
                            throw new InvalidArgumentException('Multiple applicable fifo layers for negative events not supported.');
                        }
                        // The applicableFifoLayer is not active
                        $applicableFifoLayer = FifoLayer::findOrFail($this->applicableFifoLayersForNegativeEvents[0]);
                        // We need to clear out quantity in the applicableFifoLayer then run the do-while loop again
                        $fifoLayerAllocationData = ($this->fifoLayerInventoryMovementRelocator)(abs($quantity), $applicableFifoLayer);
                        $backorderQuantity = $fifoLayerAllocationData->remainingQuantityToAllocate;

                        if ($backorderQuantity > 0) {
                            dd("Backorder of $backorderQuantity needed");
                        } else {
                            continue;
                        }
                    }
                    /**
                     * The product doesn't have any more active fifo layers to account
                     * for the quantity needed. If other layers are acceptable (such as backorder queues),
                     * we resort to those instead. However, if only fifo layers are needed, we
                     * attempt to strip some from sales orders and move those orders to the backorder
                     * queue. If it's still not enough, we abort and disallow the reduction in stock.
                     */
                    if ($fifoOnly) {
                        $gatheredLayers = $this->takeFifoFromOtherSources(
                            $quantity,
                            $event,
                            $reductionCause,
                            $excludedSources
                        );
                        /**
                         * We need to ensure that only fifo layers
                         * are received from the other sources.
                         */
                        $this->ensureLayersAreFifo($gatheredLayers);

                        // Add the gathered layers
                        array_push($layers, ...$gatheredLayers);

                        /**
                         * Charge the quantities to the gathered
                         * fifo layers.
                         */
                        foreach ($gatheredLayers as $layerInfo) {
                            $this->chargeQuantityToFifoLayer($layerInfo['layer'], $layerInfo['quantity']);
                        }
                        $quantity = max(0, $quantity - collect($gatheredLayers)->sum('quantity'));
                        customlog('SKU-6135', $event->getProductId().': FIFO only', [
                            'quantity_remaining' => $quantity,
                            'used_quantity' => collect($gatheredLayers)->sum('quantity'),
                        ]);
                    } elseif ($backorderLink) {
                        customlog('SKU-6135', $event->getProductId().': Backorder queue to be created for '.$quantity.' units');
                        /**
                         * Since other layers are acceptable, we simply create a backorder queue
                         * for the remaining quantity.
                         */
                        $backorderQueue = $this->createBackorderQueueForQuantity($quantity, $backorderLink, $reductionCause, $event);
                        customlog('SKU-6135', $event->getProductId().': Backorder queue created for '.$quantity.' units', $backorderQueue->toArray());
                        $layers[] = [
                            'layer' => $backorderQueue,
                            'layer_type' => BackorderQueue::class,
                            'quantity' => $quantity,
                        ];
                        $quantity = max(0, $quantity - $backorderQueue->backordered_quantity);
                        customlog('SKU-6135', $event->getProductId().': Quantity reduced', [
                            'quantity_remaining' => $quantity,
                            'backordered_quantity' => $backorderQueue->backordered_quantity,
                        ]);
                    }
                    /**
                     * At this point, the entire quantity should be fully accounted for, otherwise,
                     * then there isn't enough quantity in stock to allow the reduction (and we can't create backorders),
                     * so we abort the operation.
                     */
                    if ($quantity != 0 && $inventoryDeficiencyActionType === InventoryDeficiencyActionType::ACTION_TYPE_EXCEPTION) {
                        customlog('SKU-6135', $event->getProductId().': Backorders failed to cover the quantity needed to be reduced.', [
                            'quantity_remaining' => $quantity,
                        ]);
                        $exception = new InsufficientStockException($this->product->id, "Insufficient stock for product: {$this->product->sku}");
                        $exception->setShortage($quantity);
                        throw $exception;
                    } elseif ($quantity != 0) {
                        $gatheredLayers = $this->handleInventoryDeficiencyAction($inventoryDeficiencyActionType, $quantity, $event);
                        foreach ($gatheredLayers as $layerInfo) {
                            $this->chargeQuantityToFifoLayer($layerInfo['layer'], $layerInfo['quantity']);
                        }
                        $quantity = max(0, $quantity - collect($gatheredLayers)->sum('quantity'));
                        array_push($layers, ...$gatheredLayers);
                    }
                }
            } while ($quantity > 0);
        });

        return $layers;
    }

    private function handleInventoryDeficiencyAction(string $action, int $quantity, NegativeInventoryEvent|PositiveInventoryEvent $event): array
    {
        // TODO:: Move these into dedicated handlers.
        $layers = [];
        switch ($action) {
            case InventoryDeficiencyActionType::ACTION_TYPE_POSITIVE_ADJUSTMENT:
                $adjustment = new InventoryAdjustment();
                $adjustment->quantity = $quantity;
                $adjustment->adjustment_date = Carbon::now('UTC');
                $adjustment->notes = 'sku.io integrity (inventory deficiency action).';
                $adjustment->warehouse_id = $this->warehouseId;
                $adjustment->link_id = $event->getId();
                $adjustment->link_type = $event->getLinkType();
                $this->product->inventoryAdjustments()->save($adjustment);

                if (App::environment() !== 'testing') {
                    slack('Inventory deficiency action (env '.App::environment().', '.config('app.url')."): {$action} for product: {$this->product->sku} ({$this->product->id}) with quantity: {$quantity} ({$event->getLinkType()}: {$event->getId()})\n\n".debug_pretty_string());
                }

                try {
                    self::with(
                        $adjustment->warehouse_id,
                        $adjustment->product
                    )->addToStock(
                        quantity: abs($adjustment->quantity),
                        event: $adjustment,
                        syncStockNow: true,
                        autoApplyStock: false
                    );
                    $layers[] = [
                        'layer' => $adjustment->getFifoLayer(),
                        'layer_type' => FifoLayer::class,
                        'quantity' => $adjustment->quantity,
                        'avg_cost' => $adjustment->getFifoLayer()->avg_cost,
                    ];
                } catch (Throwable) {
                }
        }

        return $layers;
    }

    /**
     * @throws InvalidArgumentException
     */
    private function ensureLayersAreFifo(array $gatheredLayers)
    {
        $nonFifoCount = collect($gatheredLayers)
            ->where('layer_type', '!=', FifoLayer::class)
            ->count();
        if ($nonFifoCount > 0) {
            throw new InvalidArgumentException('Non-Fifo layer encountered when only fifo is required.');
        }
    }

    /**
     * Takes stock (fifo quantity) from other sources.
     */
    protected function takeFifoFromOtherSources(
        int $quantity,
        InventoryEvent $event,
        InventoryReductionCause $reductionCause = InventoryReductionCause::INCREASED_NEGATIVE_EVENT,
        array $excludedSources = []
    ): array {
        /**
         * There may potentially be different sources to take fifo from (not sure of others yet)
         * but right now, we only take them from sales orders. We can handle other sources
         * in the future here.
         */
        $results = [];
        array_push($results, ...(
            new SalesOrderFifoExtractor($this->warehouseId, $this->product)
        )->takeFifoFromSalesOrders($quantity, $event, $reductionCause, $excludedSources));

        return $results;
    }

    protected function createBackorderQueueForQuantity(
        int $quantity,
        Model|SalesOrderLine $backorderLink,
        InventoryReductionCause $reductionCause,
        InventoryEvent $event,
    ): BackorderQueue {
        return $this->backorderManager->createBackorderQueue(
            quantity: $quantity,
            salesOrderLine: $backorderLink,
            reductionCause: $reductionCause,
            event: $event
        );
    }

    /**
     * Charges a max of the given quantity to the provided fifo layer
     * and returns the actual quantity charged.
     *
     *
     * @throws Exception
     */
    protected function chargeQuantityToFifoLayer(FifoLayer $fifoLayer, int $maxQuantity): int
    {
        $applicableQuantity = min($fifoLayer->available_quantity, $maxQuantity);
        $fifoLayer->fulfilled_quantity += $applicableQuantity;
        $fifoLayer->save();

        return $applicableQuantity;
    }

    /**
     * Reassigns inventory layers (fifo & backorder queue) to the negating movement
     * and its siblings.
     *
     *
     * @throws OpenStockTakeException
     */
    protected function reassignLayers(InventoryMovement $movement, array $layers): void
    {
        if (empty($layers)) {
            return;
        }

        /**
         * Here, we simply map the movement and its siblings to the provided
         * layers.
         */
        if (count($layers) == 1) {
            /**
             * It's a single layer, so we simply map.
             */
            InventoryMovement::with([])
                ->where('link_id', $movement->link_id)
                ->where('link_type', $movement->link_type)
                ->where('layer_id', $movement->layer_id)
                ->where('layer_type', $movement->layer_type)
                ->update([
                    'layer_id' => $layers[0]['layer']->id,
                    'layer_type' => $layers[0]['layer_type'],
                ]);

            $this->combineMovementsByQuantity(
                $movement->link_id,
                $movement->link_type,
                $layers[0]['layer']->id,
                $layers[0]['layer_type']
            );

            /**
             * For sales order lines, we also update the sales order line layers
             * if the applied layer is a Fifo Layer.
             */
            $this->handleSalesOrderLineLayers($movement, $layers[0]);
        } else {
            /**
             * More than 1 layer was applied, we need to split the movements
             * to map to each of the layers.
             */
            $movements = [$movement];
            $siblings = InventoryMovement::with([])
                ->where('link_id', $movement->link_id)
                ->where('link_type', $movement->link_type)
                ->where('layer_id', $movement->layer_id)
                ->where('layer_type', $movement->layer_type)
                ->where('id', '!=', $movement->id)
                ->get();

            array_push($movements, ...$siblings);
            $this->splitMovementsToLayers($movements, $layers);
        }
    }

    /**
     * Aggregates inventory movements based on matching link and layer
     */
    protected function combineMovementsByQuantity($linkId, $linkType, $layerId, $layerType)
    {
        InventoryMovement::with([])
            ->where('layer_id', $layerId)
            ->where('layer_type', $layerType)
            ->where('link_id', $linkId)
            ->where('link_type', $linkType)
            ->get()
            ->groupBy('inventory_status')
            ->each(function ($movements) {
                if (count($movements) <= 1) {
                    return;
                }

                /**
                 * We simply retain the first record and set it's quantity to
                 * the sum of all the quantities.
                 */
                $first = $movements->first();
                $first->quantity = $movements->sum('quantity');
                $first->save();

                InventoryMovement::with([])
                    ->whereIn('id', array_diff($movements->pluck('id')->toArray(), [$first->id]))
                    ->delete();
            });
    }

    /**
     * Splits the provided inventory movements for the given
     * layers.
     *
     *
     * @throws OpenStockTakeException
     */
    protected function splitMovementsToLayers(array $movements, array $layers)
    {
        /** @var InventoryMovement $movement */
        foreach ($movements as $movement) {
            foreach ($layers as $key => $layerInfo) {
                if ($key > 0) {
                    $movement = $movement->replicate();
                    $movement->save();
                }
                $movement->update([
                    'quantity' => ($movement->quantity < 0 ? -1 : 1) * $layerInfo['quantity'],
                    'layer_id' => $layerInfo['layer']->id,
                    'layer_type' => $layerInfo['layer_type'],
                ]);

                $this->combineMovementsByQuantity(
                    $movement->link_id,
                    $movement->link_type,
                    $layerInfo['layer']->id,
                    $layerInfo['layer_type']
                );

                $this->handleSalesOrderLineLayers($movement, $layerInfo);
            }
        }
    }

    /**
     * Handles sales order line layers for fifo layers.
     */
    protected function handleSalesOrderLineLayers(InventoryMovement $movement, $layerInfo)
    {
        /**
         * For sales order lines, we update the sales order line layers
         * if the applied layer is a Fifo Layer.
         */
        if ($movement->link_type == SalesOrderLine::class && $layerInfo['layer_type'] == FifoLayer::class) {
            SalesOrderLineLayer::with([])->where('sales_order_line_id', $movement->link_id)
                ->update([
                    'layer_id' => $layerInfo['layer']->id,
                    'layer_type' => $layerInfo['layer_type'],
                ]);
        }
    }

    /**
     * Handler for when new stock becomes available.
     *
     *
     * @throws Exception
     */
    public function onStockAvailable(
        FifoLayer $fifoLayer,
        InventoryEvent $event,
        ?int $quantity = null
    ) {
        /**
         * When stock becomes available, we perform
         * any tasks that need to use the new stock.
         * For now, we only release backorder queue.
         */
        if (is_null($quantity)) {
            $quantity = $fifoLayer->available_quantity;
        }

        /**
         * If the event is a negative inventory event (thus the quantity is reduced to create more stock),
         * we use the link id and type on the fifo layer.
         */
        if ($event instanceof PositiveInventoryEvent) {
            $linkId = $event->getId();
            $linkType = $event->getLinkType();
        } else {
            $linkId = $fifoLayer->link_id;
            $linkType = $fifoLayer->link_type;
        }

        ReleaseBackorderQueues::make($fifoLayer->product)
            ->execute($fifoLayer, $linkType, $linkId, $quantity);
    }

    public function clearAllInventoryMovements(InventoryEvent $event): void
    {
        $this->inventoryMovementRepository->deleteAllInventoryMovementsForEvent($event);
    }
}
