<?php

namespace App\Services\InventoryManagement;

use App\Collections\BackorderQueueCollection;
use App\Collections\FifoLayerCollection;
use App\Collections\InventoryMovementCollection;
use App\DTO\BackorderQueueDto;
use App\DTO\BackorderQueueReleaseDto;
use App\DTO\BulkImportResponseDto;
use App\DTO\FifoLayerDto;
use App\DTO\InventoryMovementDto;
use App\DTO\SalesOrderLineLayerDto;
use App\Exceptions\NegativeInventoryFulfilledSalesOrderLinesException;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\BackorderQueue;
use App\Models\BackorderQueueRelease;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Models\WarehouseTransferShipmentLine;
use App\Models\WarehouseTransferShipmentReceiptLine;
use App\Repositories\BackorderQueueRepository;
use App\Repositories\FifoLayerRepository;
use App\Repositories\InventoryMovementRepository;
use App\Repositories\SalesOrderLineLayerRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Str;
use Throwable;

class BulkInventoryManager
{
    private FifoLayerRepository $fifoLayerRepository;

    private InventoryMovementRepository $inventoryMovementRepository;

    private BackorderQueueRepository $backorderQueueRepository;

    private SalesOrderLineLayerRepository $salesOrderLineLayerRepository;

    public function __construct()
    {
        $this->fifoLayerRepository = app(FifoLayerRepository::class);
        $this->inventoryMovementRepository = app(InventoryMovementRepository::class);
        $this->backorderQueueRepository = app(BackorderQueueRepository::class);
        $this->salesOrderLineLayerRepository = app(SalesOrderLineLayerRepository::class);
    }

    /**
     * @throws NegativeInventoryFulfilledSalesOrderLinesException
     * @throws Throwable
     */
    public function bulkDeletePositiveInventoryEvents(
        Collection|EloquentCollection $positiveInventoryEventCollection,
    ): void {
        if ($positiveInventoryEventCollection->isEmpty()) {
            return;
        }

        $fifoLayerIdsBeingDeleted = $positiveInventoryEventCollection
            ->map(fn (PositiveInventoryEvent $positiveInventoryEvent) => $positiveInventoryEvent->getFifoLayer())
            ->filter()
            ->pluck('id')
            ->toArray();

        $productIds = $positiveInventoryEventCollection
            ->map(fn (PositiveInventoryEvent $positiveInventoryEvent) => $positiveInventoryEvent->getProductId())
            ->filter()
            ->unique()
            ->toArray();

        // Perform select queries to load collections into memory that will be needed to write upsert DTOs
        $activeMovementUsages = $this->inventoryMovementRepository->getActiveMovementUsagesForFifoLayerIds($fifoLayerIdsBeingDeleted);
        $reservedMovementUsages = $this->inventoryMovementRepository->getReservedMovementUsagesForFifoLayerIds($fifoLayerIdsBeingDeleted);
        $salesOrderLinesFromUsages = $this->inventoryMovementRepository->getSalesOrderLinesFromUsages($activeMovementUsages);
        $backorderQueueReleases = $this->backorderQueueRepository->getBackorderQueueReleasesForFifoLayerIds($fifoLayerIdsBeingDeleted);
        $availableFifoLayers = $this->getAvailableFifoLayersForMovements($activeMovementUsages, $fifoLayerIdsBeingDeleted);
        $fifoLayersBeingDeleted = $this->fifoLayerRepository->getFifoLayersForIds($activeMovementUsages->pluck('layer_id')->toArray());
        $historicalBackorderQueues = $this->backorderQueueRepository->getBackorderQueuesForSalesOrderLineIds($activeMovementUsages->where('link_type', SalesOrderLine::class)->pluck('link_id')->toArray());

        // Throws exception if any of the positive inventory events being deleted have fulfilled sales order lines
        $this->checkEligibilityForReallocation($fifoLayerIdsBeingDeleted);

        // Must be able to do the reallocation, deletion of the positive inventory event deletion of the positive inventory event's fifo layer within a transaction
        DB::transaction(function () use (
            $positiveInventoryEventCollection,
            $activeMovementUsages,
            $reservedMovementUsages,
            $salesOrderLinesFromUsages,
            $backorderQueueReleases,
            $availableFifoLayers,
            $fifoLayersBeingDeleted,
            $historicalBackorderQueues,
        ) {
            try {
                // Reallocation.  This is the most complex part of the process.
                $this->reallocateOnFifoDeletion(
                    $activeMovementUsages,
                    $reservedMovementUsages,
                    $salesOrderLinesFromUsages,
                    $backorderQueueReleases,
                    $availableFifoLayers,
                    $fifoLayersBeingDeleted,
                    $historicalBackorderQueues
                );

                // Delete fifo layer and movement of each positive inventory event
                $positiveInventoryEventCollection->each(function (PositiveInventoryEvent $positiveInventoryEvent) {
                    $positiveInventoryEvent->getFifoLayer()->delete();
                    // TODO: Build delete method and handle status update
                    $positiveInventoryEvent->delete();
                });
            } catch (Throwable $e) {
                DB::rollBack(0);
                throw $e;
            }
        });

        dispatch(new UpdateProductsInventoryAndAvgCost($productIds));
        dispatch(new SyncBackorderQueueCoveragesJob(null, null, $productIds));
    }

    /*
      * TODO: Consider re-architecting reservation movements to link to parent movements, not to layers
      * TODO: here are the usages of reserved status:
      * SalesOrderFulfillmentLine
      * SalesOrderLine
      * WarehouseTransferShipmentLine
      * WarehouseTransferLine
      *
      */

    /**
     * @throws NegativeInventoryFulfilledSalesOrderLinesException
     * @throws Throwable
     */
    public function reallocateOnFifoDeletion(
        EloquentCollection $activeMovementUsages,
        EloquentCollection $reservedMovementUsages,
        EloquentCollection $salesOrderLinesFromUsages,
        EloquentCollection $backorderQueueReleases,
        EloquentCollection $availableFifoLayers,
        EloquentCollection $fifoLayersBeingDeleted,
        EloquentCollection $historicalBackorderQueues
    ): void {
        $collections = $this->buildCollectionsForReallocation(
            $activeMovementUsages,
            $reservedMovementUsages,
            $salesOrderLinesFromUsages,
            $backorderQueueReleases,
            $availableFifoLayers,
            $fifoLayersBeingDeleted,
            $historicalBackorderQueues
        );

        // Delete Backorder Queue Releases
        $backorderQueueReleases->each->delete();

        // Delete all old inventory movements that belonged to the deleted fifo layer
        InventoryMovement::destroy($collections['inventoryMovementCollectionForDelete']);

        /*
        |--------------------------------------------------------------------------
        | Writing to the database with bulk queries
        |--------------------------------------------------------------------------
        */

        // Update Backorder Queue Cache
        batch()->update(new BackorderQueue(), $collections['backorderQueueCollectionForUpdate']->toArray(), 'id');

        // Create backorders then update new layer reference in movements
        BackorderQueue::insert($collections['backorderQueueCollectionForInsert']->toArray());

        // Create backorder queue releases for any released by the unallocated fifo layers
        BackorderQueueRelease::insert($collections['backorderQueueReleaseCollectionForInsert']->toArray());

        // Create sales order line layers for any allocated to the unallocated fifo layers
        SalesOrderLineLayer::insert($collections['salesOrderLineLayerCollectionForInsert']->toArray());

        // Now that new backorder queues are inserted, we can query them and use the data to prepare the new inventory movements
        $backorderQueueMappings = BackorderQueue::query()
            ->select('id', 'sales_order_line_id', 'backordered_quantity')
            ->whereNotIn('id', $historicalBackorderQueues->pluck('id'))
            ->get()
            ->toArray();

        $inventoryMovementCollectionForNewBackorders = $this->buildCollectionsForInventoryMovementsForNewBackorderQueues($backorderQueueMappings, $activeMovementUsages, $reservedMovementUsages);

        // Create new inventory movements for new and old backorders
        InventoryMovement::query()->insert(
            array_merge(
                $collections['inventoryMovementCollectionForInsert']->toArray(),
                $inventoryMovementCollectionForNewBackorders['inventoryMovementCollectionForInsert']->toArray(),
            )
        );

        // Save FIFO layer usage cache
        batch()->update(new FifoLayer(), $collections['fifoLayerCollectionForUpdate']->toArray(), 'id');
    }

    /**
     *
     * Note that $negativeInventoryEventCollection must be already sanitized with the events that are intended to be allocated
     * For example, for sales order lines, you'll want to make sure they don't belong to sales orders that are in draft or canceled.
     *
     * @throws BindingResolutionException
     */
    public function bulkAllocateNegativeInventoryEvents(
        Collection|EloquentCollection $negativeInventoryEventCollection,
    ): void
    {

        if ($negativeInventoryEventCollection->count() == 0) {
            return;
        }

        if ($negativeInventoryEventCollection instanceof EloquentCollection) {
            $negativeInventoryEventCollection = collect($negativeInventoryEventCollection)->sortBy(function ($model) {
                return $model->getEventDate();
            });
        }

        // First, collect all unique product_id and warehouse_id pairs
        $productWarehousePairs = $negativeInventoryEventCollection->map(function (NegativeInventoryEvent $event) {
            return ['product_id' => $event->getProductId(), 'warehouse_id' => $event->getWarehouseId()];
        })->unique();

        // Query available fifo layers
        $availableFifoLayers = $this->fifoLayerRepository->getAvailableFifoLayersForProductWarehousePairs($productWarehousePairs);

        // Build a collection of InventoryMovementDto for insertion
        $collections = $this->buildCollectionsForAllocation($negativeInventoryEventCollection, $availableFifoLayers);

        /*
        |--------------------------------------------------------------------------
        | Writing to the database with bulk queries
        |--------------------------------------------------------------------------
        */

        // TODO: Encapsulate in a transaction

        // Create Backorder Queues
        $this->backorderQueueRepository->saveBulk($collections['backorderQueueCollection']);

        // Build collections for backorder inventory movements insertion
        $collections['inventoryMovementCollection'] = $collections['inventoryMovementCollection']->merge($this->buildCollectionForBackorderInventoryMovements($collections['backorderQueueCollection'], $collections['salesOrderLineEventCollection']));
        $this->inventoryMovementRepository->saveBulk($collections['inventoryMovementCollection']);
        $this->fifoLayerRepository->saveBulk($collections['fifoLayerCollection']);
        $this->salesOrderLineLayerRepository->saveBulk($collections['salesOrderLineLayerCollection']);

        dispatch(new UpdateProductsInventoryAndAvgCost($negativeInventoryEventCollection->pluck('product_id')->unique()->toArray()))->onQueue('syncInventory');

    }

    private function buildCollectionForBackorderInventoryMovements(Collection $backorderQueues, Collection $salesOrderLineEventCollection): InventoryMovementCollection
    {
        $backorderQueueMappings = DB::table('backorder_queues')
            ->whereIn('sales_order_line_id', $backorderQueues->pluck('sales_order_line_id'))
            ->select('id', 'sales_order_line_id', 'backordered_quantity')->get()->toArray();

        $inventoryMovementCollection = new InventoryMovementCollection();

        foreach ($backorderQueueMappings as $backorderQueueMapping) {
            /** @var SalesOrderLine $salesOrderLine */
            $salesOrderLine = $salesOrderLineEventCollection
                ->where('id', $backorderQueueMapping->sales_order_line_id)
                ->firstOrFail();

            $inventoryMovementCollection->add(InventoryMovementDto::from([
                'layer_type' => BackorderQueue::class,
                'layer_id' => $backorderQueueMapping->id,
                'link_type' => SalesOrderLine::class,
                'link_id' => $salesOrderLine->id,
                'type' => $salesOrderLine->getType(),
                'inventory_status' => InventoryMovement::INVENTORY_STATUS_ACTIVE,
                'inventory_movement_date' => $salesOrderLine->getEventDate(),
                'quantity' => -$backorderQueueMapping->backordered_quantity,
                'product_id' => $salesOrderLine->getProductId(),
                'warehouse_id' => $salesOrderLine->getWarehouseId(),
                'reference' => $salesOrderLine->getReference(),
            ]));

            $inventoryMovementCollection->add(InventoryMovementDto::from([
                'layer_type' => BackorderQueue::class,
                'layer_id' => $backorderQueueMapping->id,
                'link_type' => SalesOrderLine::class,
                'link_id' => $salesOrderLine->id,
                'type' => $salesOrderLine->getType(),
                'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
                'inventory_movement_date' => $salesOrderLine->getEventDate(),
                'quantity' => $backorderQueueMapping->backordered_quantity,
                'product_id' => $salesOrderLine->getProductId(),
                'warehouse_id' => $salesOrderLine->getWarehouseId(),
                'reference' => $salesOrderLine->getReference(),
            ]));
        }

        return $inventoryMovementCollection;
    }

    private function buildCollectionsForAllocation(Collection $negativeInventoryEventCollection, Collection $availableFifoLayers): array
    {
        $inventoryMovementCollection = new InventoryMovementCollection();
        $backorderQueueCollection = new BackorderQueueCollection();
        $fifoLayerCollection = new FifoLayerCollection();
        $salesOrderLineLayerCollection = collect();
        $priority = [];
        $salesOrderLineEventCollection = new Collection();

        // STOCK ALLOCATION
        // For each negative event, allocate stock.  Start with the oldest available layer to newest, then use backorder queues
        $negativeInventoryEventCollection->each(/**
         * @throws Throwable
         */ function (NegativeInventoryEvent $event) use (
            &$inventoryMovementCollection,
            &$backorderQueueCollection,
            &$fifoLayerCollection,
            &$priority,
            &$salesOrderLineLayerCollection,
            &$salesOrderLineEventCollection,
            $availableFifoLayers
        ) {

            $unallocatedFifoLayers = $availableFifoLayers
                ->where('product_id', $event->getProductId())
                ->where('warehouse_id', $event->getWarehouseId())
                ->sortBy('fifo_layer_date');

            $remainingQuantityToAllocate = $event->getQuantity();

            do {
                // In stock
                if ($fifoLayer = $unallocatedFifoLayers->shift()) {
                    $remainingQuantityToAllocate = $this->allocateInventory(
                        $fifoLayer,
                        $event,
                        $remainingQuantityToAllocate,
                        $inventoryMovementCollection,
                        $fifoLayerCollection,
                        $salesOrderLineLayerCollection,
                    );
                } elseif ($event->getLinkType() === SalesOrderLine::class) {
                    // If not in stock and the event is a sales order line, we create a backorder queue
                    // We wil create the inventory movements for the backorder queue later once we get a response from the saveBulk()
                    $remainingQuantityToAllocate = $this->createBackorderQueue(
                        $event,
                        $remainingQuantityToAllocate,
                        $backorderQueueCollection,
                        $salesOrderLineEventCollection,
                        $priority
                    );
                }
            } while ($remainingQuantityToAllocate > 0);

            // TODO: Add test to make sure this works
            // Re-add unallocated layer to the top of the collection
            if ($fifoLayer?->available_quantity > 0) {
                $unallocatedFifoLayers->prepend($fifoLayer);
            }
        });

        return [
            'inventoryMovementCollection' => $inventoryMovementCollection,
            'backorderQueueCollection' => $backorderQueueCollection,
            'fifoLayerCollection' => $fifoLayerCollection,
            'salesOrderLineLayerCollection' => $salesOrderLineLayerCollection,
            'salesOrderLineEventCollection' => $salesOrderLineEventCollection,
        ];
    }

    /**
     * @throws Throwable
     */
    private function allocateInventory(
        FifoLayer $fifoLayer,
        NegativeInventoryEvent $event,
        float $remainingEventQuantityToAllocate,
        InventoryMovementCollection $inventoryMovementCollection,
        FifoLayerCollection $fifoLayerCollection,
        Collection $salesOrderLineLayerCollection
    ): float {

        // We get the actual quantity available on the fifo layer.
        // Note that some quantity may be used for other negative inventory events,
        // so we need to account for that.
        /** @var FifoLayer|null $currentlyTrackedFifoLayer */
        $currentlyTrackedFifoLayer = $fifoLayerCollection
            ->where('id', $fifoLayer->id)
            ->first();

        if(!$currentlyTrackedFifoLayer){
            // If the fifo layer is not in the collection, we can just use the default
            // quantities from the database.
            $fifoLayerQuantityAvailable = $fifoLayer->available_quantity;
            $totalFulfilledQuantity = $fifoLayer->fulfilled_quantity;
        }else{
            // If the fifo layer is in the collection, we need to use the
            // tracked quantities in the collection as those are the most up-to-date.
            $totalFulfilledQuantity = $currentlyTrackedFifoLayer->fulfilled_quantity;
            $fifoLayerQuantityAvailable = max(0, $fifoLayer->original_quantity - $totalFulfilledQuantity);
        }


        $allocationQuantity = min($fifoLayerQuantityAvailable, $remainingEventQuantityToAllocate);
        if($allocationQuantity <= 0){
            return $remainingEventQuantityToAllocate;
        }

        $remainingEventQuantityToAllocate -= $allocationQuantity;

        $movementDto = $this->buildInventoryMovementDtoFromEvent($fifoLayer, $event, -$allocationQuantity);
        $inventoryMovementCollection->add($movementDto);

        /*
         * Here we also create inventory movements for reservations if the event qualifies as needing one.
         * We do want to refactor this to avoid creating inventory movements for reservations, but that is a
         * different scope and requires system-wide consideration.
         */
        if (in_array($event->getLinkType(), [SalesOrderLine::class, WarehouseTransferShipmentLine::class, WarehouseTransferShipmentReceiptLine::class])) {
            $inventoryMovementCollection->add($this->buildInventoryMovementDtoFromEvent($fifoLayer, $event, $allocationQuantity, InventoryMovement::INVENTORY_STATUS_RESERVED));
        }

        throw_if(($totalFulfilledQuantity + $allocationQuantity) > $fifoLayer->original_quantity, 'Impossible to allocate more than original quantity to FIFO Layer '.$fifoLayer->id);

        if($currentlyTrackedFifoLayer){
            // Update the tracked fifo layer in the collection (fulfilled quantity only)
            $currentlyTrackedFifoLayer->fulfilled_quantity += $allocationQuantity;
        } else {
            $fifoLayerCollection->add($this->buildFifoLayerDto($fifoLayer, $allocationQuantity));
        }
        $salesOrderLineLayerCollection->add($this->buildSalesOrderLineLayerDto($fifoLayer, $event, $allocationQuantity));

        return $remainingEventQuantityToAllocate;
    }

    private function createBackorderQueue(
        NegativeInventoryEvent $event,
        float $remainingEventQuantityToAllocate,
        BackorderQueueCollection $backorderQueueCollection,
        Collection $salesOrderLineEventCollection,
        array &$priority
    ): float {
        /** @var SalesOrderLine $salesOrderLine */
        $salesOrderLine = $event;
        $salesOrderLineEventCollection->add($salesOrderLine);
        $priority[$event->getProductId()] = ($priority[$event->getProductId()] ?? 0) + 1;
        $backorderQueueCollection->add(BackorderQueueDto::from([
            'sales_order_line_id' => $salesOrderLine->id,
            'backorder_date' => $salesOrderLine->getEventDate(),
            'supplier_id' => $salesOrderLine->product->defaultSupplierProduct?->supplier_id,
            'priority' => $priority[$salesOrderLine->getProductId()],
            'backordered_quantity' => $remainingEventQuantityToAllocate,
            'released_quantity' => 0,
            'uid' => (string) Str::uuid(),
        ]));

        return 0;
    }

    private function buildInventoryMovementDtoFromEvent(FifoLayer $fifoLayer, NegativeInventoryEvent $event, float $quantity, string $inventoryStatus = InventoryMovement::INVENTORY_STATUS_ACTIVE): InventoryMovementDto
    {
        return InventoryMovementDto::from([
            'layer_type' => FifoLayer::class,
            'layer_id' => $fifoLayer->id,
            'type' => $event->getType(),
            'inventory_status' => $inventoryStatus,
            'inventory_movement_date' => $event->getEventDate(),
            'quantity' => $quantity,
            'product_id' => $event->getProductId(),
            'warehouse_id' => $event->getWarehouseId(),
            'link_type' => $event->getLinkType(),
            'link_id' => $event->getId(),
            'reference' => $event->getReference(),
        ]);
    }

    private function buildInventoryMovementDtoFromMovement($layer, InventoryMovement $movement, float $quantity): InventoryMovementDto
    {
        return InventoryMovementDto::from([
            'layer_type' => get_class($layer),
            'layer_id' => $layer->id,
            'type' => $movement->type,
            'inventory_status' => $movement->inventory_status,
            'inventory_movement_date' => $movement->inventory_movement_date,
            'quantity' => $quantity,
            'product_id' => $movement->product_id,
            'warehouse_id' => $movement->warehouse_id,
            'link_type' => $movement->link_type,
            'link_id' => $movement->link_id,
            'reference' => $movement->reference,
            'created_at' => Carbon::now(),
        ]);
    }

    private function buildFifoLayerDto(FifoLayer $fifoLayer, float $allocationQuantity): FifoLayerDto
    {
        return FifoLayerDto::from([
            'id' => $fifoLayer->id,
            'total_cost' => $fifoLayer->total_cost,
            'original_quantity' => $fifoLayer->original_quantity,
            'warehouse_id' => $fifoLayer->warehouse_id,
            'fifo_layer_date' => $fifoLayer->fifo_layer_date,
            'product_id' => $fifoLayer->product_id,
            'link_type' => $fifoLayer->link_type,
            'fulfilled_quantity' => $fifoLayer->fulfilled_quantity + $allocationQuantity,
            'link_id' => $fifoLayer->link_id,
            'updated_at' => Carbon::now(),
        ]);
    }

    private function buildSalesOrderLineLayerDto(FifoLayer $fifoLayer, SalesOrderLine $salesOrderLine, float $allocationQuantity): SalesOrderLineLayerDto
    {
        return SalesOrderLineLayerDto::from([
            'sales_order_line_id' => $salesOrderLine->id,
            'quantity' => $allocationQuantity,
            'layer_id' => $fifoLayer->id,
            'layer_type' => FifoLayer::class,
            'created_at' => Carbon::now(),
        ]);
    }

    private function buildBackorderQueueReleaseDto(FifoLayer $fifoLayer, BackorderQueue $backorderQueue, float $allocationQuantity): BackorderQueueReleaseDto
    {
        return BackorderQueueReleaseDto::from([
            'backorder_queue_id' => $backorderQueue->id,
            'released_quantity' => $allocationQuantity,
            'link_id' => $fifoLayer->link_id,
            'link_type' => get_class($fifoLayer->link),
            'reference' => $fifoLayer->link->getReference(),
            'created_at' => Carbon::now(),
        ]);
    }

    /**
     * @throws NegativeInventoryFulfilledSalesOrderLinesException
     */
    private function checkEligibilityForReallocation(array $fifoLayerIdsBeingDeleted): void
    {
        // Determine ineligibility for reallocation
        $fulfillmentMovementsForLayersBeingDeleted = InventoryMovement::query()
            ->whereIn('layer_id', $fifoLayerIdsBeingDeleted)
            ->where('layer_type', FifoLayer::class)
            ->where('link_type', SalesOrderFulfillmentLine::class);
        if ($fulfillmentMovementsForLayersBeingDeleted->exists()) {
            $fulfilledSalesOrderLinesData = [];
            SalesOrderLine::query()
                ->where('fulfilled_quantity', '>', 0)
                ->whereHas('inventoryMovements', function (Builder $query) use ($fifoLayerIdsBeingDeleted) {
                    $query->whereIn('layer_id', $fifoLayerIdsBeingDeleted);
                    $query->where('layer_type', FifoLayer::class);
                })
                ->each(function (SalesOrderLine $salesOrderLine) use (&$fulfilledSalesOrderLinesData) {
                    $fulfilledSalesOrderLinesData[] = [
                        'sku' => $salesOrderLine->product->sku,
                        'sales_order_number' => $salesOrderLine->salesOrder->sales_order_number,
                        'fulfilled_quantity' => $salesOrderLine->fulfilled_quantity,
                    ];
                });
            throw new NegativeInventoryFulfilledSalesOrderLinesException($fulfilledSalesOrderLinesData);
        }
    }

    private function getAvailableFifoLayersForMovements(Collection $movements, $fifoLayerIdsBeingDeleted)
    {
        // First, collect all unique product_id and warehouse_id pairs
        $productWarehousePairs = $movements->map(function (InventoryMovement $movement) {
            return ['product_id' => $movement->product_id, 'warehouse_id' => $movement->warehouse_id];
        })->unique();

        // Query available fifo layers
        return $this->fifoLayerRepository->getAvailableFifoLayersForProductWarehousePairs($productWarehousePairs, $fifoLayerIdsBeingDeleted);
    }

    /**
     * This function builds collections that will be used to:
     * - Insert new inventory movements
     * - Delete existing inventory movements
     * - Insert new backorder queues
     * - Update existing backorder queues (quantity backordered and quantity released)
     * - Update existing fifo layers (fulfilled quantity)
     *
     * @throws NegativeInventoryFulfilledSalesOrderLinesException
     * @throws Throwable
     */
    private function buildCollectionsForReallocation(
        Collection $activeMovementUsages,
        Collection $reservedMovementUsages,
        Collection $salesOrderLinesFromUsages,
        Collection $backorderQueueReleases,
        Collection $availableFifoLayers,
        Collection $fifoLayersBeingDeleted,
        Collection $historicalBackorderQueues
    ): array {
        $inventoryMovementCollectionForInsert = new InventoryMovementCollection();
        $inventoryMovementCollectionForDelete = new InventoryMovementCollection();
        $backorderQueueCollectionForInsert = new BackorderQueueCollection();
        $backorderQueueCollectionForUpdate = new BackorderQueueCollection();
        $fifoLayerCollectionForUpdate = new FifoLayerCollection();
        $salesOrderLineLayerCollectionForInsert = collect();
        $backorderQueueReleaseCollectionForInsert = collect();
        $priority = [];
        $backorderQueueReleasedQuantity = [];

        // STOCK ALLOCATION
        // For each usage movement, allocate stock.  Start with the oldest available layer to newest, then use backorder queues
        $activeMovementUsages->each(/**
         * @throws Throwable
         */ function (InventoryMovement $usageMovement) use (
            &$inventoryMovementCollectionForInsert,
            &$inventoryMovementCollectionForDelete,
            &$backorderQueueCollectionForInsert,
            &$backorderQueueCollectionForUpdate,
            &$fifoLayerCollectionForUpdate,
            &$salesOrderLineLayerCollectionForInsert,
            &$backorderQueueReleaseCollectionForInsert,
            &$priority,
            $availableFifoLayers,
            $reservedMovementUsages,
            $salesOrderLinesFromUsages,
            $fifoLayersBeingDeleted,
            $backorderQueueReleases,
            $historicalBackorderQueues,
            &$backorderQueueReleasedQuantity,
        ) {
            // The reserved counterpart movement
            /** @var InventoryMovement $usageReservationMovement */
            $usageReservationMovement = $reservedMovementUsages
                ->where('link_id', $usageMovement->link_id)
                ->where('link_type', $usageMovement->link_type)
                ->first();

            $unallocatedFifoLayers = $availableFifoLayers
                ->where('product_id', $usageMovement->product_id)
                ->where('warehouse_id', $usageMovement->warehouse_id)
                ->sortBy('fifo_layer_date');

            $remainingMovementQuantityToAllocate = -$usageMovement->quantity; // Negative since movement quantity is negative

            do {
                // In stock
                /** @var FifoLayer $fifoLayer */
                if ($fifoLayer = $unallocatedFifoLayers->shift()) {
                    $allocationQuantity = min($fifoLayer->available_quantity, $remainingMovementQuantityToAllocate);
                    $remainingMovementQuantityToAllocate -= $allocationQuantity;

                    $inventoryMovementCollectionForDelete->add($usageMovement->id);
                    $inventoryMovementCollectionForInsert->add($this->buildInventoryMovementDtoFromMovement($fifoLayer, $usageMovement, -$allocationQuantity));

                    /*
                     * Here we also create inventory movements for reservations if the event qualifies as needing one.
                     * We do want to refactor this to avoid creating inventory movements for reservations, but that is a
                     * different scope and requires system-wide consideration.
                     */
                    switch ($usageMovement->link_type) {
                        case SalesOrderLine::class:
                            if ($usageMovement->link->backorderQueue) {
                                $backorderQueueReleaseCollectionForInsert->add($this->buildBackorderQueueReleaseDto($fifoLayer, $usageMovement->link->backorderQueue, $allocationQuantity));
                            }
                        case WarehouseTransferShipmentLine::class:
                        case WarehouseTransferShipmentReceiptLine::class:
                            $inventoryMovementCollectionForInsert->add($this->buildInventoryMovementDtoFromMovement($fifoLayer, $usageReservationMovement, $allocationQuantity));
                            $inventoryMovementCollectionForDelete->add($usageReservationMovement->id);
                            $salesOrderLineLayerCollectionForInsert->add($this->buildSalesOrderLineLayerDto($fifoLayer, $usageMovement->link, $allocationQuantity));
                            break;
                    }

                    throw_if(($fifoLayer->fulfilled_quantity + $allocationQuantity) > $fifoLayer->original_quantity, 'Impossible to allocate more than original quantity to FIFO Layer '.$fifoLayer->id);
                    $fifoLayerCollectionForUpdate->add($this->buildFifoLayerDto($fifoLayer, $allocationQuantity));
                } elseif ($usageMovement->link_type == SalesOrderLine::class) {

                    // The following uses collections instead of eloquent relationships to avoid querying database from within this method

                    /** @var SalesOrderLine $salesOrderLine */
                    $salesOrderLine = $salesOrderLinesFromUsages->where('id', $usageMovement->link_id)->first();

                    // A single sales order line can only have a single backorder queue.
                    /** @var BackorderQueue $backorderQueue */
                    $backorderQueue = $historicalBackorderQueues->where('sales_order_line_id', $salesOrderLine->id)->first();

                    /** @var FifoLayer $fifoLayerBeingDeleted */
                    $fifoLayerBeingDeleted = $fifoLayersBeingDeleted->where('id', $usageMovement->layer_id)->first();

                    // The single backorder queue release associated to the specific usage being iterated through
                    $backorderQueueRelease = $backorderQueueReleases
                        ->where('link_type', $fifoLayerBeingDeleted->link_type)
                        ->where('link_id', $fifoLayerBeingDeleted->link_id)
                        ->where('backorder_queue_id', $backorderQueue?->id)
                        ->first();

                    // The backorder queue due to the deletion of the fifo layer, can...
                    // Scenario 1. Have quantity increase
                    // Scenario 2. Have released quantity decrease
                    // Scenario 3. Have quantity increase and released quantity decrease
                    // Scenario 4. Have the backorder quantity remain the same and the release quantity remain the same (new release added for the new fifo layer)
                    if ($backorderQueue) {
                        if ($backorderQueueRelease) {
                            // Ensures accuracy in case of multiple releases
                            $backorderQueueReleasedQuantity[$backorderQueue->id] = max(0,
                                isset($backorderQueueReleasedQuantity[$backorderQueue->id]) ?
                                    $backorderQueueReleasedQuantity[$backorderQueue->id] - $backorderQueueRelease->released_quantity :
                                    $backorderQueue->released_quantity - $backorderQueueRelease->released_quantity
                            );

                            if ($remainingMovementQuantityToAllocate > 0) {
                                // The backorder queue release due to deletion of the fifo layer, must be deleted completely
                                // There is a 1:1 relationship between a release and a fifo layer (via the positive inventory event
                                // that is being deleted).  This is taken care of within the reallocateOnFifoDeletion method, so we
                                // don't have to worry about it here.

                                // Update the released quantity, ensuring it doesn't go below zero, (Scenario 3)
                                $newReleasedQuantity = $backorderQueueReleasedQuantity[$backorderQueue->id];

                                $backorderQueueCollectionForUpdate = $backorderQueueCollectionForUpdate->reject(function ($value, $key) use ($backorderQueue) {
                                    return $value->id == $backorderQueue->id;
                                });
                                $backorderQueueCollectionForUpdate->add(BackorderQueueDto::from([
                                    'id' => $backorderQueue->id,
                                    'sales_order_line_id' => $backorderQueue->sales_order_line_id,
                                    'backorder_date' => $backorderQueue->backorder_date,
                                    'supplier_id' => $backorderQueue->supplier_id,
                                    'priority' => 1, // Since released backorder queue priority gets nulled out, we are making a decision here to set this as first priority (since we have to choose something)
                                    'backordered_quantity' => $backorderQueue->backordered_quantity,
                                    'released_quantity' => $newReleasedQuantity,
                                ]));

                                $inventoryMovementCollectionForDelete->add($usageMovement->id);
                                $inventoryMovementCollectionForDelete->add($usageReservationMovement->id);
                                $inventoryMovementCollectionForInsert->add($this->buildInventoryMovementDtoFromMovement($backorderQueue, $usageMovement, $usageMovement->quantity));
                                $inventoryMovementCollectionForInsert->add($this->buildInventoryMovementDtoFromMovement($backorderQueue, $usageReservationMovement, -$usageMovement->quantity));
                            }
                        // Else scenario 4, no change to backorder queue
                        } else {
                            // If a backorder exists, but there is no release, then the backorder needs an increase as a result of deletion of the fifo layer (Scenario 1)
                            $newBackorderedQuantity = $backorderQueue->backordered_quantity + $remainingMovementQuantityToAllocate;

                            $backorderQueueCollectionForUpdate->add(BackorderQueueDto::from([
                                'id' => $backorderQueue->id,
                                'sales_order_line_id' => $backorderQueue->sales_order_line_id,
                                'backorder_date' => $backorderQueue->backorder_date,
                                'supplier_id' => $backorderQueue->supplier_id,
                                'priority' => 1, // Since released backorder queue priority gets nulled out, we are making a decision here to set this as first priority (since we have to choose something)
                                'backordered_quantity' => $newBackorderedQuantity,
                                'released_quantity' => 0,
                                'updated_at' => Carbon::now(),
                            ]));

                            $inventoryMovementCollectionForDelete->add($usageMovement->id);
                            $inventoryMovementCollectionForDelete->add($usageReservationMovement->id);
                            $inventoryMovementCollectionForInsert->add($this->buildInventoryMovementDtoFromMovement($backorderQueue, $usageMovement, $usageMovement->quantity));
                            $inventoryMovementCollectionForInsert->add($this->buildInventoryMovementDtoFromMovement($backorderQueue, $usageReservationMovement, -$usageMovement->quantity));
                        }
                    } else {
                        // If no backorder queue exists, we create a new backorder queue and a new movement for the new quantity.  For new backorder queues, we must
                        // create the backorder queue first, then the movement gets created in reallocateOnFifoDeletion
                        $this->createBackorderQueueForMovement(
                            $salesOrderLine,
                            $remainingMovementQuantityToAllocate,
                            $backorderQueueCollectionForInsert,
                            $priority
                        );
                    }
                    // The remaining quantity to allocate is guaranteed to be covered by an existing backorder queue or a new one
                    $remainingMovementQuantityToAllocate = 0;
                } else {
                    throw new Exception('Usages of type '.$usageMovement->type.' for fifo layer '.$usageMovement->layer_id.' could not be reallocated to an available fifo layer.  To delete this positive inventory event you must delete the '.$usageMovement->link_type.' with id of '.$usageMovement->link_id.' first');
                }
            } while ($remainingMovementQuantityToAllocate > 0);

            // Re-add unallocated layer to the top of the collection
            if ($fifoLayer?->available_quantity > 0) {
                $unallocatedFifoLayers->prepend($fifoLayer);
            }
        });

        return [
            'inventoryMovementCollectionForInsert' => $inventoryMovementCollectionForInsert,
            'inventoryMovementCollectionForDelete' => $inventoryMovementCollectionForDelete->unique(),
            'salesOrderLineLayerCollectionForInsert' => $salesOrderLineLayerCollectionForInsert,
            'backorderQueueReleaseCollectionForInsert' => $backorderQueueReleaseCollectionForInsert,
            'backorderQueueCollectionForInsert' => $backorderQueueCollectionForInsert,
            'backorderQueueCollectionForUpdate' => $backorderQueueCollectionForUpdate,
            'fifoLayerCollectionForUpdate' => $fifoLayerCollectionForUpdate,
        ];
    }

    /**
     * @throws Exception
     */
    private function createBackorderQueueForMovement(
        SalesOrderLine $salesOrderLine,
        float $remainingMovementQuantityToAllocate,
        BackorderQueueCollection $backorderQueueCollectionForInsert,
        array &$priority
    ): void {
        $priority[$salesOrderLine->product_id] = ($priority[$salesOrderLine->product_id] ?? 0) + 1;

        $backorderQueueCollectionForInsert->add(BackorderQueueDto::from([
            'sales_order_line_id' => $salesOrderLine->id,
            'backorder_date' => $salesOrderLine->getEventDate(),
            'supplier_id' => $salesOrderLine->product->defaultSupplierProduct?->supplier_id,
            'priority' => $priority[$salesOrderLine->getProductId()],
            'backordered_quantity' => $remainingMovementQuantityToAllocate,
            'released_quantity' => 0,
            'uid' => Str::uuid()->toString(),
            'created_at' => Carbon::now(),
        ]));

    }

    /**
     * @throws Exception
     */
    private function createInventoryMovementInsertDtoFromExistingBackorderMovements(Collection $existingBackorderMovements, int $remainingMovementQuantityToAllocate, InventoryMovementCollection $inventoryMovementCollectionForInsert, ?int $layerId = null): void
    {
        if (count($existingBackorderMovements) % 2 != 0) {
            throw new Exception(count($existingBackorderMovements).' movements for sales order line id '.$existingBackorderMovements->first()->link_id.' (should be matching pair) exists.  Please check data integrity');
        }
        foreach ($existingBackorderMovements as $existingBackorderMovement) {
            $sign = $existingBackorderMovement->quantity > 0 ? 1 : -1;
            $inventoryMovementCollectionForInsert->add(InventoryMovementDto::from([
                'layer_type' => BackorderQueue::class,
                'layer_id' => $layerId ?? $existingBackorderMovement->layer_id,
                'type' => $existingBackorderMovement->type,
                'inventory_status' => $existingBackorderMovement->inventory_status,
                'inventory_movement_date' => $existingBackorderMovement->inventory_movement_date,
                'quantity' => $existingBackorderMovement->quantity + ($sign * $remainingMovementQuantityToAllocate),
                'product_id' => $existingBackorderMovement->product_id,
                'warehouse_id' => $existingBackorderMovement->warehouse_id,
                'link_type' => $existingBackorderMovement->link_type,
                'link_id' => $existingBackorderMovement->link_id,
                'reference' => $existingBackorderMovement->reference,
                'created_at' => Carbon::now(),
            ]));
        }
    }

    private function createInventoryMovementDtoFromBackorderQueueMapping(InventoryMovement $movement, array $backorderQueueMapping, bool $isUsage, bool $includeId): InventoryMovementDto
    {
        $dto = InventoryMovementDto::from([
            'layer_type' => BackorderQueue::class,
            'layer_id' => $backorderQueueMapping['id'],
            'link_id' => $movement->link_id,
            'link_type' => $movement->link_type,
            'type' => $movement->type,
            'inventory_status' => $movement->inventory_status,
            'inventory_movement_date' => $movement->inventory_movement_date,
            'quantity' => ($isUsage ? -1 : 1) * min($backorderQueueMapping['backordered_quantity'], abs($movement->quantity)),
            'product_id' => $movement->product_id,
            'warehouse_id' => $movement->warehouse_id,
            'reference' => $movement->reference,
            'created_at' => Carbon::now(),
        ]);

        if ($includeId) {
            $dto->id = $movement->id;
        }

        return $dto;
    }

    private function buildCollectionsForInventoryMovementsForNewBackorderQueues(array $backorderQueueMappings, Collection $activeMovementUsages, Collection $reservedMovementUsages): array
    {
        $inventoryMovementCollectionForInsert = new InventoryMovementCollection();

        foreach ($backorderQueueMappings as $backorderQueueMapping) {
            $activeMovement = $activeMovementUsages
                ->where('link_id', $backorderQueueMapping['sales_order_line_id'])
                ->where('link_type', SalesOrderLine::class)
                ->first();

            if (! $activeMovement) {
                continue;
            }
            $reservedMovement = $reservedMovementUsages
                ->where('link_id', $backorderQueueMapping['sales_order_line_id'])
                ->where('link_type', SalesOrderLine::class)
                ->first();

            $active = $this->createInventoryMovementDtoFromBackorderQueueMapping($activeMovement, $backorderQueueMapping, true, false);
            $reserved = $this->createInventoryMovementDtoFromBackorderQueueMapping($reservedMovement, $backorderQueueMapping, false, false);

            $inventoryMovementCollectionForInsert->add($active);
            $inventoryMovementCollectionForInsert->add($reserved);
        }

        return [
            'inventoryMovementCollectionForInsert' => $inventoryMovementCollectionForInsert,
        ];
    }
}
