<?php

namespace App\Services\InventoryManagement\Actions;

use App\DTO\SalesOrderLineLayerDto;
use App\Exceptions\InsufficientSalesOrderLineQuantityException;
use App\Models\FifoLayer;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Repositories\SalesOrderLineLayerRepository;

class MoveSalesOrderLineLayersQuantity
{
    private SalesOrderLineLayerRepository $lineLayers;

    public function __construct(
        private readonly SalesOrderLine $salesOrderLine,
        private readonly int $quantity,
        private readonly FifoLayer $sourceFifoLayer,
        private readonly FifoLayer $destinationFifoLayer,
    )
    {
        $this->lineLayers = app(SalesOrderLineLayerRepository::class);
    }

    /**
     * @throws InsufficientSalesOrderLineQuantityException
     */
    public function handle(): void
    {
        $quantityToMove = $this->quantity;
        $lineLayers = $this->salesOrderLine->salesOrderLineLayers
            ->where('layer_id', $this->sourceFifoLayer->id)
            ->where('layer_type', FifoLayer::class);

            if ($lineLayers->isEmpty()) {
                throw new InsufficientSalesOrderLineQuantityException('No sales order line layers found to move for sales order line id: ' . $this->salesOrderLine->id);
            }

            $lineLayers->each(function (SalesOrderLineLayer $salesOrderLineLayer) use (&$quantityToMove) {
                if ($quantityToMove <= 0) {
                    return;
                }
                $movedQuantity = $this->moveQuantity($salesOrderLineLayer, $quantityToMove);
                $quantityToMove -= $movedQuantity;
            });

        if ($quantityToMove > 0) {
            throw new InsufficientSalesOrderLineQuantityException('Existing sales order line layers did not have enough quantity to move.  May try fixing sales order line layers cache first');
        }
    }

    private function moveQuantity(SalesOrderLineLayer $salesOrderLineLayer, int $quantityToMove): int
    {
        $movedQuantity = min($quantityToMove, $salesOrderLineLayer->quantity);
        $salesOrderLineLayer->quantity -= $movedQuantity;
        if ($salesOrderLineLayer->quantity === 0) {
            $salesOrderLineLayer->delete();
        } else {
            $salesOrderLineLayer->save();
        }

        $this->createNewLayer($movedQuantity);

        return $movedQuantity;
    }

    private function createNewLayer(int $quantity): void
    {
        $this->lineLayers->create(SalesOrderLineLayerDto::from([
            'sales_order_line_id' => $this->salesOrderLine->id,
            'quantity' => $quantity,
            'layer_id' => $this->destinationFifoLayer->id,
            'layer_type' => FifoLayer::class,
        ]));
    }
}