<?php

namespace App\Services\InventoryManagement\Actions;

use App\Exceptions\InsufficientMovableNegativeInventoryMovementsException;
use App\Exceptions\InsufficientStockException;
use App\Exceptions\InventoryMovementTypeException;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Models\StockTakeItem;
use App\Models\WarehouseTransferShipmentLine;
use App\Models\WarehouseTransferShipmentReceiptLine;
use App\Repositories\SalesOrderLineRepository;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\InventoryManagement\InventoryReductionCause;
use App\Services\StockTake\OpenStockTakeException;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Throwable;

class FixInvalidFifoLayerFulfilledQuantityCache
{
    private Collection $inventoryMovementsToMove;

    /**
     * @throws InsufficientStockException
     * @throws OpenStockTakeException
     * @throws Throwable
     */
    public function fix(FifoLayer $fifoLayer): void
    {
        $this->inventoryMovementsToMove = collect();

        // First, fix the originating movement if missing.
        if ($this->fifoHasNoOriginatingMovement($fifoLayer)) {
            $this->createOriginatingMovementForFifo($fifoLayer);
        }

        // To fix the over-subscription, we get negating movements
        // to the tune of the discrepancy quantity on the fifo layer
        // and attempt to move them to other fifo layers, prioritizing
        // any active fifo layers on the product.
        $usageFromMovements = abs($fifoLayer->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('layer_type', FifoLayer::class)
            ->where('quantity', '<', 0)
            ->sum('quantity'));
        $discrepancy = $fifoLayer->fulfilled_quantity - $usageFromMovements;

        if ($discrepancy == 0) {
            return;
        }

        $newFulfilledQuantity = $fifoLayer->fulfilled_quantity - $discrepancy;
        if ($newFulfilledQuantity <= $fifoLayer->original_quantity)
        {
            $fifoLayer->fulfilled_quantity -= $discrepancy;
            $fifoLayer->save();
            return;
        }

        // If we're here, the discrepancy is really the amount of the oversubscription, not the fulfilled quantity discrepancy.
        $discrepancy = $usageFromMovements - $fifoLayer->original_quantity;

        DB::beginTransaction();
        try {

            /*
             * We calculate the available FIFO layers for the product/warehouse, going directly to how much is available, and
             * ignoring cache
             */
            $availableFifoLayers = $this->getAvailableFifoLayers($fifoLayer, $fifoLayer->warehouse_id, $fifoLayer->product_id);
            $totalAvailableFromOtherFifoLayers = $availableFifoLayers->sum('available_quantity');

            // We get negating movements that can be moved from the fifo
            // layer to bring the fifo layer back in sync with its original quantity.
            $this->gatherAffectedMovements($fifoLayer, $discrepancy, $totalAvailableFromOtherFifoLayers);
            $this->transferMovementsToOtherFifos($fifoLayer, $availableFifoLayers);

            if (!$this->inventoryMovementsToMove->isEmpty()) {
                Log::error('UNMOVED MOVEMENTS', $this->inventoryMovementsToMove->toArray());
                dump("Could not fix fifo layer: $fifoLayer->id due to unmovable movements");
                DB::rollBack();

                // Create positive adjustment and try again.
                $quantity = abs(
                    $this->inventoryMovementsToMove->pluck('movement')->where('inventory_status', 'active')->sum('quantity')
                );

                $this->createPositiveAdjustment($quantity, $fifoLayer);

                $this->fix($fifoLayer);
                return;


            }

            $newUsagesFromMovements = abs($fifoLayer->inventoryMovements()
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->where('layer_type', FifoLayer::class)
                ->where('quantity', '<', 0)
                ->sum('quantity'));

            $fifoLayer->fulfilled_quantity = min($newUsagesFromMovements, $fifoLayer->original_quantity);
            $fifoLayer->save();

            DB::commit();
        } catch (Throwable $e) {
            DB::rollBack();
            throw $e;
        }
    }

    private function createPositiveAdjustment(int $quantity, FifoLayer $fifoLayer): void
    {
        $adjustment = new InventoryAdjustment();
        $adjustment->warehouse_id = $fifoLayer->warehouse_id;
        $adjustment->product_id = $fifoLayer->product_id;
        $adjustment->quantity = $quantity;
        $adjustment->adjustment_date = $fifoLayer->fifo_layer_date;
        $adjustment->notes = 'sku.io integrity';
        $adjustment->save();

        InventoryManager::with(
            $adjustment->warehouse_id,
            $adjustment->product
        )->addToStock(abs($adjustment->quantity), $adjustment, unitCost: $adjustment->unit_cost);
    }

    private function fifoHasNoOriginatingMovement(FifoLayer $fifoLayer): bool
    {
        return ! $fifoLayer->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '>', 0)
            ->exists();
    }

    private function createOriginatingMovementForFifo(FifoLayer $fifoLayer): void
    {
        // We need to get the inventory movement type from
        // the fifo layer's link.
        $type = match ($fifoLayer->link_type) {
            PurchaseOrderShipmentReceiptLine::class => InventoryMovement::TYPE_PURCHASE_RECEIPT,
            WarehouseTransferShipmentReceiptLine::class => InventoryMovement::TYPE_TRANSFER,
            SalesCreditReturnLine::class => InventoryMovement::TYPE_RETURN,
            InventoryAdjustment::class => InventoryMovement::TYPE_ADJUSTMENT,
            StockTakeItem::class => InventoryMovement::TYPE_STOCK_TAKE
        };

        $movement = new InventoryMovement();
        $movement->inventory_movement_date = $fifoLayer->fifo_layer_date;
        $movement->quantity = abs($fifoLayer->original_quantity);
        $movement->product_id = $fifoLayer->product_id;
        $movement->link_id = $fifoLayer->link_id;
        $movement->link_type = $fifoLayer->link_type;
        $movement->type = $type;
        $movement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
        $movement->warehouse_id = $fifoLayer->warehouse_id;
        $movement->reference = $fifoLayer->link_reference;
        $movement->warehouse_location_id = $fifoLayer->warehouse->defaultLocation?->id;
        $fifoLayer->inventoryMovements()->save($movement);
    }

    public function getAvailableFifoLayers(FifoLayer $fifoLayer, int $warehouseId, int $productId): Collection
    {
        $availableFifoLayers = collect();

        $fifoLayers = FifoLayer::where('id', '!=', $fifoLayer->id)
            ->where('product_id', $productId)
            ->where('warehouse_id', $warehouseId)
            ->orderBy('fifo_layer_date');

        $fifoLayers->each(function (FifoLayer $fifoLayer) use ($availableFifoLayers) {
            /*
             * We can't rely on the cache in the case of inventory integrity issues, so we should calculate inventory available based on movements
             */
            $calculatedUsages = abs($fifoLayer->inventoryMovements()
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->where('quantity', '<', 0)
                ->sum('quantity'));

            $calculatedAvailableQuantity = $fifoLayer->original_quantity - $calculatedUsages;

            if ($calculatedAvailableQuantity <= 0)
            {
                return;
            }

            if ($fifoLayer->fulfilled_quantity != $calculatedUsages)
            {
                /*
                 * The fifo layer has available quantity but the cache is wrong, so let's update the fulfilled quantity cache
                 */
                $fifoLayer->fulfilled_quantity = min($calculatedUsages, $fifoLayer->original_quantity);
                $fifoLayer->save();
            }

            $availableFifoLayers->push($fifoLayer);
        });

        return $availableFifoLayers;
    }

    /**
     * @throws InsufficientMovableNegativeInventoryMovementsException
     */
    private function gatherAffectedMovements(FifoLayer $fifoLayer, int $discrepancy, int $totalAvailableFromOtherFifoLayers): void
    {
        $tallyAvailableFromOtherFifoLayers = $totalAvailableFromOtherFifoLayers;
        $tallyDiscrepancy = $discrepancy;

        do {
            // Eligible unfulfilled sales order lines to move
            $movementToMove = $fifoLayer->inventoryMovements()
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->where('quantity', '<', 0)
                ->whereIn('link_type', [SalesOrderLine::class, WarehouseTransferShipmentLine::class])
                ->whereNotIn('id', $this->inventoryMovementsToMove->pluck('movement.id'))
                ->orderBy('inventory_movement_date', 'DESC')->first();

            if ($movementToMove)
            {
                $quantityToMove = min(abs($movementToMove->quantity), $tallyDiscrepancy, $tallyAvailableFromOtherFifoLayers);
                    $tallyDiscrepancy -= $quantityToMove;
                    $tallyAvailableFromOtherFifoLayers -= $quantityToMove;

                    // Sales order lines can be backordered even if $tallyAvailableFromOtherFifoLayers is insufficient
                    if ($movementToMove->link_type == SalesOrderLine::class && $quantityToMove == 0)
                    {
                        $unfulfilledQuantity = abs($movementToMove->quantity) - app(SalesOrderLineRepository::class)->calculateFulfilledQuantity($movementToMove->link);
                        $quantityToBackorder = min($unfulfilledQuantity, $tallyDiscrepancy);
                        $tallyDiscrepancy -= $quantityToBackorder;
                    }
                    $this->inventoryMovementsToMove->add([
                        'movement' => $movementToMove,
                        'quantities' => collect([
                            'move' => $quantityToMove,
                            'backorder' => $quantityToBackorder ?? 0,
                            'adjust' => 0
                        ]),
                    ]);
                    continue;
            }

            // Try and find movement from adjustments
            $adjustmentMovement = $fifoLayer->inventoryMovements()
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->whereHasMorph('link', InventoryAdjustment::class)
                ->where('quantity', '<', 0)
                ->whereNotIn('id', $this->inventoryMovementsToMove->pluck('movement.id'))
                ->orderBy('inventory_movement_date', 'DESC')
                ->first();

            if ($adjustmentMovement) {
                /*
                 * TODO: Right now we allow deleting or reducing the quantity of negative adjustments, but this is dangerous
                 *  because it could lead to unintentional increases in inventory.  The proper way would be to prioritize
                 *  moving inventory adjustments to available fifo layers and then if there are still discrepancies ($total < $discrepancy)
                 *  report to user to allow them to manually delete the negative inventory adjustments.
                 */
                $quantityToAdjust = min(abs($adjustmentMovement->quantity), $tallyDiscrepancy);
                $adjustmentMovement->quantity_to_adjust = $quantityToAdjust;
                $this->inventoryMovementsToMove->add([
                    'movement' => $adjustmentMovement,
                    'quantities' => collect([
                        'move' => 0,
                        'backorder' => 0,
                        'adjust' => $quantityToAdjust
                    ]),
                ]);
                $tallyDiscrepancy -= $quantityToAdjust;
            } else {
                if ($tallyDiscrepancy > 0) {
                    throw new InsufficientMovableNegativeInventoryMovementsException("There are still " . $tallyDiscrepancy . " quantity needing to move from fifo layer id " . $fifoLayer->id . " but there are no more eligible movements to move");
                }
            }
        } while ($tallyDiscrepancy > 0);
    }

    /**
     * @throws InsufficientStockException
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     * @throws Throwable
     */
    private function transferMovementsToOtherFifos(FifoLayer $fifoLayer, Collection $availableFifoLayers): void
    {
        $manager = InventoryManager::with(
            $fifoLayer->warehouse_id,
            $fifoLayer->product
        );

        foreach ($this->inventoryMovementsToMove as $key => $movementObject)
        {
            if ($movementObject['quantities']['move'] > 0)
            {
                $layers = collect($manager->reduceInventory(
                    $movementObject['quantities']['move'],
                    $movementObject['movement']['link'],
                    InventoryReductionCause::INCREASED_NEGATIVE_EVENT,
                    true, // No backorders, can only move to available layers
                    $movementObject['movement']['link'],
                    [$fifoLayer->id],
                ));

                $this->portMovementToLayers($movementObject['movement'], $layers);

                if (($received = $layers->sum('quantity')) >= $movementObject['quantities']['move']) {
                    $this->inventoryMovementsToMove->forget($key);
                } else {
                    $this->inventoryMovementsToMove[$key]->quantities->move -= $received;
                }
            }

            if ($movementObject['quantities']['backorder'] > 0)
            {
                $layers = collect($manager->reduceInventory(
                    $movementObject['quantities']['backorder'],
                    $movementObject['movement']['link'],
                    InventoryReductionCause::INCREASED_NEGATIVE_EVENT,
                    false, // Allow backorders
                    $movementObject['movement']['link'],
                    [$fifoLayer->id],
                ));

                $this->portMovementToLayers($movementObject['movement'], $layers);

                if (($received = $layers->sum('quantity')) >= $movementObject['quantities']['backorder']) {
                    $this->inventoryMovementsToMove->forget($key);
                } else {
                    $this->inventoryMovementsToMove[$key]->quantities->backorder -= $received;
                }
            }

            if ($movementObject['quantities']['adjust'] > 0)
            {
                $this->handleAdjustmentMovement($movementObject['movement'], $key, $movementObject['quantities']['adjust']);
            }
        }
    }

    /**
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     * @throws Throwable
     */
    private function handleAdjustmentMovement(InventoryMovement $movement, int $key, int $quantityToMove): void
    {
        // We only handle adjustments based on stricter conditions
        if ($movement->quantity >= 0) {
            // adjustment should be the only negating movement.
            return;
        }

        // We have a negative inventory movement that we now want to either reduce the quantity of or delete.  At this
        // point the inventory adjustment movement has already been selected to be moved and since there are no sales order
        // lines to move... we can safely reduce the quantity or delete the movement/adjustment.
        $diff = abs($movement->quantity) - $quantityToMove;

        // This means the adjustment is fully used by the quantity to move, so we just delete it (and its movement)
        if ($diff == 0) {
            // need to use this to avoid triggering the delete method on the movement or adjustment, which would unintentionally allocate backorders to the fifo layer now that we've deleted a negative event
            InventoryMovement::where('id', $movement->id)->delete();
            InventoryAdjustment::where('id', $movement->link_id)->delete();
            $this->inventoryMovementsToMove->forget($key);
            return;
        }

        if ($diff > 0) {
            if(isset($movement->quantity_to_adjust)){
                unset($movement->quantity_to_adjust);
            }
            // Movements are negative quantity so this is reducing the adjustment movement quantity
            $movement->quantity += $diff;
            $movement->save();

            // The adjustment is negative so this is reducing the adjustment quantity
            $movement->link->quantity += $diff;
            $movement->link->save();

            $this->inventoryMovementsToMove->forget($key);
        }
    }

    /**
     * @throws OpenStockTakeException
     * @throws InventoryMovementTypeException
     */
    private function portMovementToLayers(InventoryMovement $movement, Collection $layers): void
    {
        $layers->each(function ($layerInfo) use ($movement) {
            $this->moveQuantityToFifo($layerInfo['quantity'], $movement, $layerInfo['layer'], $movement->fifo_layer);
        });
    }

    /**
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     */
    private function moveQuantityToFifo(
        int $quantity,
        InventoryMovement $movement,
        FifoLayer|BackorderQueue $layer,
        FifoLayer $oldLayer
    ): void {
        if ($quantity >= abs($movement->quantity)) {
            $this->moveAllQuantityToLayer($movement, $layer, $oldLayer);

            return;
        }

        $copy = $movement->replicate();
        $copy->quantity = -$quantity;
        if ($this->isFifoLayer($layer)) {
            $copy->fifo_layer = $layer->id;
        } else {
            $copy->backorder_queue = $layer->id;
        }

        $copy->save();

        $movement->quantity += $quantity; // Movement quantity is negative, adding reduces the quantity.
        $movement->save();

        if ($movement->link_type === SalesOrderLine::class) {
            $this->handleSupportingMovementsForSalesOrderLine($quantity, $movement, $layer, $oldLayer);
        }

        if (! $this->isFifoLayer($layer)) {
            if ($movement->link instanceof SalesOrderLine && $movement->link->salesOrder->order_status === SalesOrder::STATUS_CLOSED) {
                $movement->link->salesOrder->order_status = SalesOrder::STATUS_OPEN;
                $movement->link->salesOrder->save();
            }
            dispatch(new SyncBackorderQueueCoveragesJob(null, null, [$movement->product_id]));
        }
    }

    private function isFifoLayer(FifoLayer|BackorderQueue $layer): bool
    {
        return $layer instanceof FifoLayer;
    }

    /**
     * @throws OpenStockTakeException
     */
    protected function handleSupportingMovementsForSalesOrderLine(
        int $quantity,
        InventoryMovement $movement,
        FifoLayer|BackorderQueue $layer,
        FifoLayer $oldLayer
    ): void {
        /** @var SalesOrderLine $orderLine */
        $orderLine = $movement->link;

        // First, we handle reservation movement.
        /** @var InventoryMovement $reservationMovement */
        $reservationMovement = $orderLine->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('quantity', '>', 0)
            ->firstOrFail();

        $copyReservation = $reservationMovement->replicate();
        $copyReservation->update([
            'quantity' => $quantity,
            'layer_id' => $layer->id,
            'layer_type' => $layer::class,
        ]);
        $reservationMovement->quantity -= $quantity;
        $reservationMovement->save();

        // This approach of re-creating line layers ensures that
        // we don't double count duplicate sales order lines.

        if ($this->isFifoLayer($layer)) {
            SalesOrderLineLayer::with([])->where('sales_order_line_id', $orderLine->id)
                ->where('layer_type', FifoLayer::class)
                ->where('layer_id', $oldLayer->id)
                ->delete();

            SalesOrderLineLayer::with([])
                ->create([
                    'sales_order_line_id' => $orderLine->id,
                    'quantity' => abs($movement->quantity),
                    'layer_id' => $oldLayer->id,
                    'layer_type' => FifoLayer::class,
                ]);
            SalesOrderLineLayer::with([])
                ->create([
                    'sales_order_line_id' => $orderLine->id,
                    'quantity' => $quantity,
                    'layer_id' => $layer->id,
                    'layer_type' => FifoLayer::class,
                ]);
        } else {
            SalesOrderLineLayer::with([])
                ->where('sales_order_line_id', $orderLine->id)
                ->where('layer_type', FifoLayer::class)
                ->where('layer_id', $oldLayer->id)
                ->decrement('quantity', $quantity);
        }

        // Next, we update fulfillment line inventory movements
        $fulfillmentsCount = $orderLine->salesOrderFulfillmentLines()->count();
        if ($fulfillmentsCount == 1) {
            // Only one fulfillment line, we split the movement.
            /** @var SalesOrderFulfillmentLine $fulfillmentLine */
            $fulfillmentLine = $orderLine->salesOrderFulfillmentLines()->firstOrFail();
            /** @var InventoryMovement $fulfillmentMovement */
            $fulfillmentMovement = $fulfillmentLine->inventoryMovements()->firstOrFail();
            $this->moveSupportingPositiveMovementQtyToNewLayer($fulfillmentMovement, $quantity, $layer);
        } elseif ($fulfillmentsCount > 1) {
            $fulfillmentLines = $orderLine->salesOrderFulfillmentLines()->get();
            $remainingQuantity = $quantity;

            /** @var SalesOrderFulfillmentLine $fulfillmentLine */
            foreach ($fulfillmentLines as $fulfillmentLine) {
                if ($remainingQuantity <= 0) {
                    break;
                }

                /** @var InventoryMovement $fulfillmentMovement */
                $fulfillmentMovement = $fulfillmentLine->inventoryMovements()->firstOrFail();
                if ($fulfillmentMovement->quantity == $remainingQuantity) {
                    if ($this->isFifoLayer($layer)) {
                        $fulfillmentMovement->fifo_layer = $layer->id;
                    } else {
                        $fulfillmentMovement->quantity -= $quantity;
                        $fulfillmentLine->quantity -= $quantity;
                        $fulfillmentLine->save();
                    }
                    $fulfillmentMovement->save();
                    break;
                } elseif ($fulfillmentMovement->quantity > $remainingQuantity) {
                    $this->moveSupportingPositiveMovementQtyToNewLayer($fulfillmentMovement, $remainingQuantity, $layer);
                    break;
                } else {
                    if ($this->isFifoLayer($layer)) {
                        $fulfillmentMovement->fifo_layer = $layer->id;
                    } else {
                        $fulfillmentMovement->quantity -= $quantity;
                        $fulfillmentLine->quantity -= $quantity;
                        $fulfillmentLine->save();
                    }
                    $fulfillmentMovement->save();
                    $remainingQuantity -= $fulfillmentMovement->quantity;
                }
            }
        }
    }

    /**
     * @throws OpenStockTakeException
     * @throws Exception
     */
    private function moveSupportingPositiveMovementQtyToNewLayer(InventoryMovement $movement, int $quantity, FifoLayer|BackorderQueue $layer): void
    {
        $movement->quantity -= $quantity;
        $movement->save();
        if ($this->isFifoLayer($layer)) {
            $copy = $movement->replicate();
            $copy->quantity = $quantity;
            $copy->fifo_layer = $layer->id;
            $copy->save();
        } else {
            $movement->link->quantity -= $quantity;
            $movement->link->save();
        }
    }

    /**
     * @throws OpenStockTakeException
     */
    private function moveAllQuantityToLayer(
        InventoryMovement $movement,
        FifoLayer|BackorderQueue $layer,
        FifoLayer $oldLayer
    ): void {
        if ($this->isFifoLayer($layer)) {
            $movement->fifo_layer = $layer->id;
        } else {
            $movement->backorder_queue = $layer->id;
        }
        $movement->save();

        if ($movement->link_type == SalesOrderLine::class)
        {
            // Update the layer of other sibling movements.
            /** @var SalesOrderLine $orderLine */
            $orderLine = $movement->link;
            InventoryMovement::with([])
                ->where('link_id', $orderLine->id)
                ->where('link_type', SalesOrderLine::class)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                ->where('layer_id', $oldLayer->id)
                ->update([
                    'layer_id' => $layer->id,
                    'layer_type' => $layer::class,
                ]);
            if ($this->isFifoLayer($layer)) {
                // For fifo layers, we should also update the layer
                // in the line layers.
                SalesOrderLineLayer::with([])
                    ->where('sales_order_line_id', $orderLine->id)
                    ->where('layer_id', $oldLayer->id)
                    ->update(['layer_id' => $layer->id]);

                // We also update any fulfillment movements.
                $orderLine->salesOrderFulfillmentLines()
                    ->each(function (SalesOrderFulfillmentLine $fulfillmentLine) use ($layer, $oldLayer) {
                        $fulfillmentLine
                            ->inventoryMovements()
                            ->where('layer_id', $oldLayer->id)
                            ->where('layer_type', FifoLayer::class)
                            ->update(['layer_id' => $layer->id]);
                    });
            } else {
                // In case of a backorder queue layer,
                // we remove the sales order line layer and
                // delete any fulfillments on the sales order.
                SalesOrderLineLayer::with([])
                    ->where('sales_order_line_id', $orderLine->id)
                    ->where('layer_id', $oldLayer->id)
                    ->delete();

                $orderLine->salesOrderFulfillmentLines()
                    ->each(function (SalesOrderFulfillmentLine $fulfillmentLine) {
                        $fulfillmentLine->deleteWithFulfillmentIfLast(true);
                    });
            }
        } elseif ($movement->link_type == WarehouseTransferShipmentLine::class)
        {
            /** @var WarehouseTransferShipmentLine $warehouseTransferShipmentLine */
            $warehouseTransferShipmentLine = $movement->link;
            InventoryMovement::with([])
                ->where('link_id', $warehouseTransferShipmentLine->id)
                ->where('link_type', WarehouseTransferShipmentLine::class)
                ->whereIn('inventory_status', [InventoryMovement::INVENTORY_STATUS_RESERVED, InventoryMovement::INVENTORY_STATUS_IN_TRANSIT])
                ->where('layer_id', $oldLayer->id)
                ->update([
                    'layer_id' => $layer->id,
                    'layer_type' => $layer::class,
                ]);
        }

    }
}
