<?php

namespace App\Services\InventoryAssembly;

use App\DataTable\DataTableConfiguration;
use App\Exceptions\InventoryAssemblyComponentException;
use App\Exceptions\InventoryAssemblyStockException;
use App\Exceptions\InventoryAssemblyTypeException;
use App\Http\Resources\ProductInventoryDetailsResource;
use App\Models\InventoryAssembly;
use App\Models\InventoryAssemblyLine;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\Warehouse;
use App\Services\InventoryManagement\InventoryManager;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\DB;

class InventoryAssemblyManager
{
    protected Product $kit;

    protected $components;

    public function __construct(Product $kit, $components)
    {
        $this->kit = $kit;
        $this->components = $components;
    }

    public function assemble($quantity, $assemblyDate = null, ?Product $product = null, $warehouse_id = null)
    {
        DB::beginTransaction();

        try {
            if ($product) {
                $this->kit = $product;
            }

            $warehouse_id = $warehouse_id ?? Warehouse::query()->first()->id;
            $assemblyDate = $assemblyDate ?? Carbon::now();

            $product = $this->kit;

            if ($product->type !== Product::TYPE_KIT) {
                throw new InventoryAssemblyTypeException();
            }

            if (! $product->components) {
                throw new InventoryAssemblyComponentException($product->sku);
            }

            $assembly = new InventoryAssembly();
            $assembly->warehouse_id = $warehouse_id;
            $assembly->action = InventoryAssembly::INVENTORY_ASSEMBLY_ACTION_ASSEMBLE;
            $assembly->action_date = $assemblyDate;
            $assembly->save();

            /*
             * TODO: useInventory and makeInventory should be generalized methods belonging to a Product service
             */

            $new_fifo_layer_cost = $this->useInventory($product->components, $quantity, $assembly);

            $this->makeKitInventory($product, $quantity, $assembly, $new_fifo_layer_cost);

            $assembly->load(DataTableConfiguration::getRequiredRelations(InventoryAssembly::class));

            $assembly->updateProductsInventory();
        } catch (\Exception $e) {
            DB::rollback();
            throw $e;
        }

        DB::commit();

        return $assembly;
    }

    /**
     * @throws Exception
     */
    private function useInventory($products, $inventory_needed, InventoryAssembly $assembly): float|int
    {
        if (! is_countable($products)) {
            $products = [$products];
            $is_component = false;
        } else {
            $is_component = true;
        }
        $new_fifo_layer_cost = 0;

        /** @var Product $product */
        foreach ($products as $product) {
            $used_product_cost_tally = 0;

            $inventory_still_needed = $is_component ? $product->pivot->quantity * $inventory_needed : $inventory_needed;

            $productResource = ProductInventoryDetailsResource::make($product);
            $product_inventory_available = $productResource->productInventory()->where('warehouse_id', $assembly->warehouse_id)->sum('inventory_available');

            if ($product_inventory_available < $inventory_still_needed) {
                if ($product->kit)
                {
                    $sku = $product->kit->sku;
                    $quantity = $inventory_needed;
                }
                else {
                    $sku = $product->sku;
                    $quantity = $inventory_still_needed;
                }
                throw new InventoryAssemblyStockException("Not enough component inventory to assemble $quantity kits of $sku.");
            }

            $current_fifo_layer = $product->getCurrentFifoLayerForWarehouse($assembly->warehouse);

            $fifo_layer_usage = [];

            if ($current_fifo_layer->available_quantity >= $inventory_still_needed) {
                $fifo_layer_usage[] = [
                    'fifo_layer' => $current_fifo_layer,
                    'quantity_to_use' => $inventory_still_needed,
                ];
            } else {
                $active_fifo_layers = $product->activeFifoLayers->where('warehouse_id', $assembly->warehouse_id)->values();
                $i = 0;

                while ($inventory_still_needed > 0) {
                    $quantity_to_use = min($active_fifo_layers->offsetGet($i)->available_quantity, $inventory_still_needed);

                    $fifo_layer_usage[] = [
                        'fifo_layer' => $active_fifo_layers->offsetGet($i),
                        'quantity_to_use' => $quantity_to_use,
                    ];
                    $inventory_still_needed -= $quantity_to_use;
                    $i++;
                }
            }

            $inventoryMovements = [];

            foreach ($fifo_layer_usage as $usage) {
                $fifoLayer = $usage['fifo_layer'];
                $fifoLayer->fulfilled_quantity += $usage['quantity_to_use'];
                $fifoLayer->save();

                $used_product_cost_tally += $usage['quantity_to_use'] * $fifoLayer->avg_cost;

                $inventoryMovement = new InventoryMovement();
                $inventoryMovement->product_id = $product->id;
                $inventoryMovement->inventory_movement_date = $assembly->action_date;
                $inventoryMovement->quantity = -$usage['quantity_to_use'];
                $inventoryMovement->type = InventoryMovement::TYPE_ASSEMBLY;
                $inventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
                $inventoryMovement->warehouse_id = $assembly->warehouse_id;
                $inventoryMovement->warehouse_location_id = $fifoLayer->warehouse->defaultLocation?->id ?? null;
                $inventoryMovement->fifo_layer = $fifoLayer->id;

                $inventoryMovements[] = $inventoryMovement;
            }

            $assemblyLine = new InventoryAssemblyLine();
            $assemblyLine->inventory_assembly_id = $assembly->id;
            $assemblyLine->product_id = $product->id;
            $assemblyLine->quantity = -$inventory_needed * ($is_component ? $product->pivot->quantity : 1);
            $assemblyLine->product_type = $is_component ? InventoryAssemblyLine::INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPE_COMPONENT : InventoryAssemblyLine::INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPE_KIT;
            $assemblyLine->unit_cost = $used_product_cost_tally / $inventory_needed;

            $assemblyLine->save();

            foreach ($inventoryMovements as $inventoryMovement) {
                $assemblyLine->inventoryMovements()->save($inventoryMovement);
            }

            $new_fifo_layer_cost += $used_product_cost_tally;
        }

        return $new_fifo_layer_cost;
    }

    /**
     * @throws Exception
     */
    private function makeKitInventory($kitProduct, $quantity, $assembly, $new_fifo_layer_cost): void
    {

        $assemblyLine = new InventoryAssemblyLine();
        $assemblyLine->inventory_assembly_id = $assembly->id;
        $assemblyLine->product_id = $kitProduct->id;
        $assemblyLine->quantity = $quantity;
        $assemblyLine->product_type = InventoryAssemblyLine::INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPE_KIT;
        $assemblyLine->unit_cost = $new_fifo_layer_cost / $quantity;
        $assemblyLine->save();

        // Handle inventory
        $this->handleStock($assemblyLine);

    }

    /**
     * @throws Exception
     */
    private function handleStock(InventoryAssemblyLine $assemblyLine): void{
        $manager = InventoryManager::with(
            warehouseId: $assemblyLine->getWarehouseId(),
            product: $assemblyLine->product
        );

        $manager->addToStock(
            quantity: $assemblyLine->getQuantity(),
            event: $assemblyLine,
            unitCost: $assemblyLine->unit_cost
        );
    }


    /**
     * @throws Exception
     */
    private function makeInventoryForComponents($products, $inventory_made, $assembly, $fifoLayerCost): void
    {

        $totalComponentCost = 0;
        foreach ($products as $component) {
            $totalComponentCost += $component->getUnitCostAtWarehouse($assembly->warehouse_id) * $component->pivot->quantity;
            if ($totalComponentCost == 0) {
                throw new Exception('Unable to determine cost for '.$component->sku);
            }
        }

        foreach ($products as $product) {

            $quantity = $product->pivot->quantity * $inventory_made;
            $assemblyLine = new InventoryAssemblyLine();
            $assemblyLine->inventory_assembly_id = $assembly->id;
            $assemblyLine->product_id = $product->id;
            $assemblyLine->quantity = $quantity;
            $assemblyLine->product_type = InventoryAssemblyLine::INVENTORY_ASSEMBLY_LINE_PRODUCT_TYPE_COMPONENT;
            $assemblyLine->unit_cost = $fifoLayerCost * ($product->getUnitCostAtWarehouse($assembly->warehouse_id) / $totalComponentCost);
            $assemblyLine->save();

            // Handle inventory
            $this->handleStock($assemblyLine);
        }

    }

    public function disassemble($quantity, $assemblyDate = null, ?Product $product = null, $warehouse_id = null)
    {
        DB::beginTransaction();

        try {
            if ($product) {
                $this->kit = $product;
            }

            $warehouse_id = $warehouse_id ?? Warehouse::query()->first()->id;
            $assemblyDate = $assemblyDate ?? Carbon::now();

            $product = $this->kit;

            if ($product->type !== Product::TYPE_KIT) {
                throw new InventoryAssemblyTypeException();
            }
            if (! $product->components) {
                throw new InventoryAssemblyComponentException($product->sku);
            }
            $assembly = new InventoryAssembly();
            $assembly->warehouse_id = $warehouse_id;
            $assembly->action = InventoryAssembly::INVENTORY_ASSEMBLY_ACTION_DISASSEMBLE;
            $assembly->action_date = $assemblyDate;
            $assembly->save();

            $new_fifo_layer_cost = $this->useInventory($product, $quantity, $assembly);

            $this->makeInventoryForComponents($this->components, $quantity, $assembly, $new_fifo_layer_cost);

            $assembly->load(DataTableConfiguration::getRequiredRelations(InventoryAssembly::class));

            $assembly->updateProductsInventory();
        } catch (\Exception $e) {
            DB::rollback();
            throw $e;
        }

        DB::commit();

        return $assembly;
    }
}
