<?php

namespace App\Services\FinancialManagement;

use App\DTO\SalesOrderLineFinancialDto;
use App\Enums\FinancialLineClassificationEnum;
use App\Helpers;
use App\Jobs\CalculateDailyFinancialsJob;
use App\Jobs\CalculateSalesOrderLineFinancialsJob;
use App\Jobs\ProcessFinancialAlertsJob;
use App\Models\FifoLayer;
use App\Models\SalesCreditLine;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineFinancial;
use App\Models\SalesOrderLineLayer;
use App\Models\Setting;
use App\Repositories\DailyFinancialRepository;
use App\Repositories\FinancialLineRepository;
use App\Repositories\SalesOrderLineFinancialsRepository;
use App\Repositories\SupplierProductPricingRepository;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\LazyCollection;

class SalesOrderLineFinancialManager
{
    private SalesOrderLineFinancialsRepository $salesOrderLineFinancialsRepository;

    private FinancialLineRepository $financialLineRepository;

    private ?array $sales_order_ids;

    public function __construct(?array $sales_order_ids = null)
    {
        $this->sales_order_ids = $sales_order_ids;
        $this->salesOrderLineFinancialsRepository = app(SalesOrderLineFinancialsRepository::class);
        $this->financialLineRepository = app(FinancialLineRepository::class);
    }

    public function calculate(?int $numLines = 1000): void
    {
        customlog('financials', "Calculating sales order line financials for {$numLines} lines.");
        if (! $this->salesOrderLineFinancialsRepository->hasInvalidFinancials()) {
            return;
        }

        $salesOrderLines = $this->salesOrderLineFinancialsRepository->getInvalidatedLines($this->sales_order_ids, $numLines);
        customlog('financials', "Found {$salesOrderLines->count()} lines to calculate financials for.");

        $days = $this->calculateAndCacheFinancials(
            $salesOrderLines
        );

        customlog('financials', "Clearing financials cache for {$salesOrderLines->count()} lines.");
        $this->salesOrderLineFinancialsRepository->clearInvalidFinancialsCache($salesOrderLines, $this->sales_order_ids);

        if (Helpers::setting(Setting::KEY_FINANCIAL_ALERTS_ENABLED)) {
            dispatch(new ProcessFinancialAlertsJob($salesOrderLines))->onQueue('financials');
        }

        if ($this->salesOrderLineFinancialsRepository->hasInvalidFinancials()) {
            customlog('financials', "Dispatching CalculateSalesOrderLineFinancialsJob for {$numLines} lines.");
            dispatch(new CalculateSalesOrderLineFinancialsJob($numLines))->onQueue('financials');
        } else {
            if (! empty($days)) {
                customlog('financials', 'Dispatching CalculateDailyFinancialsJob for '.count($days).' days.');
                dispatch(new CalculateDailyFinancialsJob($days))->onQueue('financials');
            }
        }

    }

    public function recalculateLines(Collection $salesOrderLines): void
    {
        $days = $this->calculateAndCacheFinancials($salesOrderLines);
        $this->salesOrderLineFinancialsRepository->clearInvalidFinancialsCache($salesOrderLines);
        if (! empty($days)) {
            customlog('financials', 'Dispatching CalculateDailyFinancialsJob for '.count($days).' days.');
            dispatch(new CalculateDailyFinancialsJob($days))->onQueue('financials');
        }
    }

    /**
     * TODO: Pending handling of financial lines tied to products (marketplace fee, landed cost %).
     */
    public function calculateAndCacheFinancials(LazyCollection|Collection $salesOrderLines): array
    {
        if ($salesOrderLines->count() == 0) {
            return [];
        }

        // The following is needed because by the time the job is processed, the sales order may have been deleted
        $salesOrderLines = $salesOrderLines->reject(function ($salesOrderLine) {
            return ! $salesOrderLine->salesOrder;
        })->values();

        // To properly perform allocations, we need to get all sales order lines for each sales order with an invalidated sales order line
        $salesOrderLines = $salesOrderLines->pluck('salesOrder')->unique()->map(function (SalesOrder $salesOrder) {
            return $salesOrder->salesOrderLines;
        })->flatten();
        customlog('financials', 'Get all sales order lines for each sales order with an invalidated sales order line.');

        $userTimezone = Helpers::getAppTimezone();

        $financialsDataCollection = new Collection();
        $days = [];

        /** @var SalesOrderLine $salesOrderLine */
        foreach ($salesOrderLines as $salesOrderLine) {
            $financialsDataCollection->add($this->getSalesOrderLineFinancialDto($salesOrderLine));
            $days[] = $salesOrderLine->salesOrder->order_date->setTimezone($userTimezone)->startOfDay()->setTimezone('UTC')->toDateTimeString();
        }
        customlog('financials', "Generated financials DTO for {$financialsDataCollection->count()} sales order lines.");
        // Recalculation of allocations for sales orders lines for already existing financial lines
        $salesOrderAllocatedData = $this->financialLineRepository->allocateLinesToSalesOrderLines($salesOrderLines->pluck('salesOrder')->unique(), $financialsDataCollection);
        customlog('financials', "Calculated allocations for {$salesOrderLines->count()} sales order lines.");
        // Now that we have the allocations, we can map them to the DTOs
        $financialsDataCollection->map(function (SalesOrderLineFinancialDto $salesOrderLineFinancialDto) use ($salesOrderAllocatedData) {
            $salesOrderLineId = $salesOrderLineFinancialDto->sales_order_line_id;
            $salesOrderLineFinancialDto->revenue_allocated = @$salesOrderAllocatedData[$salesOrderLineId][FinancialLineClassificationEnum::REVENUE->value] ?? 0;
            $salesOrderLineFinancialDto->cost_allocated = @$salesOrderAllocatedData[$salesOrderLineId][FinancialLineClassificationEnum::COST->value] ?? 0;
        });
        customlog('financials', "Mapped allocations for {$financialsDataCollection->count()} sales order lines.");

        $days = array_unique($days);

        app(DailyFinancialRepository::class)->invalidateForDates($days);
        customlog('financials', "Saving {$financialsDataCollection->count()} sales order line financials.");
        $this->salesOrderLineFinancialsRepository->save($financialsDataCollection, SalesOrderLineFinancial::class);

        // We must touch the sales order updated at to trigger accounting transaction update, but we should do this in bulk for efficiency

        $salesOrderIds = $salesOrderLines->pluck('sales_order_id')->unique()->toArray();
        SalesOrder::whereIn('id', $salesOrderIds)->update(['updated_at' => Carbon::now()->addMinute()]);

        return $days;
    }

    public function getSalesOrderLineFinancialDto(SalesOrderLine $salesOrderLine): SalesOrderLineFinancialDto
    {
        $salesOrderLineFinancialDto = SalesOrderLineFinancialDto::from([
            'sales_order_line_id' => $salesOrderLine->id,
            'revenue' => $this->getRevenue($salesOrderLine),
            //'revenue_allocated' => @$salesOrderAllocatedData[$salesOrderLine->sales_order_id][$salesOrderLine->id][FinancialLineClassificationEnum::REVENUE->value] ?? 0,
            /**
             * Credit cost is not being well maintained.  I don't know where in the codebase sales credit line unit cost is set
             * I think it makes more sense to have this as a calculated field based on the corresponding sales order line
             * TODO: Check if anything else relies on sales_credits_line.unit_cost... if not, may consider dropping it
             */
            //'proforma_sales_credit_cost' => $salesOrderLine->creditedCost,
            'credits' => $this->getProformaSalesCredit($salesOrderLine),
            'cogs' => $salesOrderLine->salesOrder->canceled_at ? 0 : $this->getCogs($salesOrderLine) * $salesOrderLine->quantity,
            'cogs_returned' => $this->getProformaReturnedUnitCost($salesOrderLine),
            //'cost_allocated' => @$salesOrderAllocatedData[$salesOrderLine->sales_order_id][$salesOrderLine->id][FinancialLineClassificationEnum::COST->value] ?? 0,
        ]);

        return $salesOrderLineFinancialDto;
    }

    public function getRevenue(SalesOrderLine $salesOrderLine): float
    {
        // Canceled orders have no sales order line revenue
        if ($salesOrderLine->salesOrder->canceled_at) {
            return 0.00;
        }

        // TODO: Need tax allocation in tenant currency
        return $salesOrderLine->quantity * $salesOrderLine->amount_in_tenant_currency - ($salesOrderLine->salesOrder->is_tax_included ? $salesOrderLine->tax_allocation_in_tenant_currency : 0.00);
    }

    public function getCogs(SalesOrderLine $salesOrderLine): float
    {
        if (is_null($salesOrderLine->product)) {
            return 0.00;
        }

        /*
         * If the line has no backorder layers, then we use the assigned fifo layers
         */

        $salesOrderLineLayers = SalesOrderLineLayer::where('sales_order_line_id', $salesOrderLine->id)
            ->get();

        // Only use sales order line layers if they exist and there are no active backorders
        if ($salesOrderLineLayers->count() > 0 && ! $salesOrderLine->activeBackorderQueue) {
            $totalCost = 0;
            foreach ($salesOrderLineLayers as $salesOrderLineLayer) {
                /** @var FifoLayer $fifoLayer */
                $fifoLayer = $salesOrderLineLayer->layer;
                if (! $fifoLayer) {
                    customlog('financials', "No fifo layer found for sales order line layer {$salesOrderLineLayer->id}.");
                }
                $totalCost += $fifoLayer->avg_cost * $salesOrderLineLayer->quantity;
            }

            return $salesOrderLine->quantity == 0 ? 0 : ($totalCost / $salesOrderLine->quantity);
        }

        /*
         * We use the most recent active layer
         */
        /** @var FifoLayer $activeFifoLayer */
        $activeFifoLayer = $salesOrderLine->fifoLayers()->first();
        if ($activeFifoLayer) {
            return $activeFifoLayer->avg_cost ?? 0.00;
        }

        /*
        * At this point we are checking if fifo layers exist at other warehouses
        */
        /** @var FifoLayer $otherLayer */
        $otherLayer = FifoLayer::query()
            ->where('product_id', $salesOrderLine->product->id)
            ->where('total_cost', '>', 0)
            ->orderBy('fifo_layer_date', 'desc')
            ->first();
        if ($otherLayer && $otherLayer->avg_cost > 0) {
            return $otherLayer->avg_cost ?? 0;
        }

        /*
         * No active fifo layer,
         * We use weighted average cost if there is a stock history.
         */
        if ($weighted_average_cost = $salesOrderLine->product->getWeightedAverageCost($salesOrderLine->warehouse_id)) {
            return $weighted_average_cost;
        }

        $supplierPrice = app(SupplierProductPricingRepository::class)->getSensibleSupplierProductPrice($salesOrderLine->product);
        if ($supplierPrice > 0) {
            return $supplierPrice;
        }

        if ($initial_cost = $salesOrderLine->product->unit_cost) {
            return $initial_cost;
        }

        return 0.00;
    }

    public function getProformaSalesCredit(SalesOrderLine $salesOrderLine): float
    {
        /*
         * Get sales credits associated to the sales order line
         * Non-associated sales credit lines will be accounted for separately
         * TODO: We have to make sure we consider them on the sales order proforma modal
         */
        $percentage_credited = $salesOrderLine->quantity == 0 ? 0 : $salesOrderLine->salesCreditLines->sum('quantity') / $salesOrderLine->quantity;

        return $this->getRevenue($salesOrderLine) * $percentage_credited;
    }

    public function getProformaReturnedUnitCost(SalesOrderLine $salesOrderLine): float
    {
        $returned_unit_cost = 0;
        $salesOrderLine->salesCreditLines->each(function (SalesCreditLine $salesCreditLine) use (&$returned_unit_cost) {
            $salesCreditLine->salesCreditReturnLines()->createsFifoLayer()->each(function (SalesCreditReturnLine $salesCreditReturnLine) use (&$returned_unit_cost) {
                try {
                    $returned_unit_cost += $salesCreditReturnLine->getFifoLayer()->avg_cost * $salesCreditReturnLine->quantity;
                } catch (\Throwable $exception) {
                    slack('sales credit return line '.$salesCreditReturnLine->id.' does not have a fifo layer');
                }
            });
        });

        return $returned_unit_cost;
    }

    // TODO: Implement in a way that creates a financial line for shipping cost when a fulfillment is created
    //    public function getProformaShippingCost(SalesOrderLine $salesOrderLine) : float
    //    {
    //        if (is_null($salesOrderLine->product)) {
    //            return 0.00;
    //        }
    //
    //        // Fully shipped, best source from fulfillments cost
    //        if ($salesOrderLine->fulfilled_quantity == $salesOrderLine->quantity) {
    //            $shipping_cost_tally = 0;
    //            /** @var SalesOrderFulfillmentLine $fulfilledLine */
    //            foreach ($salesOrderLine->fulfilledLines()->get() as $fulfilledLine) {
    //                /*
    //                 * TODO: Have a user defined setting for proration instead of hard coded
    //                 */
    //                $shipping_cost_tally +=
    //                    $fulfilledLine->getProration(SalesOrderFulfillmentLine::PRORATION_HYBRID) *
    //                    $fulfilledLine->salesOrderFulfillment->cost;
    //            }
    //            if ($shipping_cost_tally > 0) {
    //                return $shipping_cost_tally;
    //            }
    //        }
    //
    //        if ($product_shipping_cost = $salesOrderLine->product->proforma_shipping_cost) {
    //            return $product_shipping_cost * $salesOrderLine->quantity;
    //        }
    //
    //         return 0.00
    //    }
}
