<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Helpers;
use App\Models\Setting;
use App\Models\WarehouseTransferShipmentReceiptLine;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionLineDataFromWarehouseTransferShipmentReceiptLine
{
    public function __construct(
        private readonly WarehouseTransferShipmentReceiptLine $line,
    ) {}
    
    /**
     * @throws Throwable
     */
    public function handle(): Collection
    {
        $lineData = collect();
        foreach ($this->getTypeCodeMappings() as $type => $nominalCodeId)
        {
            $lineData->add(AccountingTransactionLineData::from([
                'type' => $type,
                'nominal_code_id' => $nominalCodeId,
                'description' => $this->getDescription(),
                'quantity' => $this->line->quantity,
                'amount' => $this->getAmount(),
                'tax_amount' => 0,
                'tax_rate_id' => null,
                'link_id' => $this->line->id,
                'link_type' => WarehouseTransferShipmentReceiptLine::class,
            ]));
        }
        return $lineData;
    }

    private function getDescription(): string
    {
        $warehouseTransferLine = $this->line->shipmentLine->warehouseTransferLine;
        return $warehouseTransferLine->product?->sku ?? $warehouseTransferLine->description;
    }

    private function getAmount(): float
    {
        return $this->line->totalAmount;
    }

    private function getTypeCodeMappings(): array
    {
        $warehouse = $this->line->shipmentLine->warehouseTransferShipment->warehouseTransfer->toWarehouse;
        $debit = $warehouse->nominal_code_id ?? Helpers::setting(Setting::KEY_NC_MAPPING_INVENTORY_CONTROL);

        return [
            AccountingTransactionLineTypeEnum::DEBIT->value => $debit,
            AccountingTransactionLineTypeEnum::CREDIT->value => Helpers::setting(Setting::KEY_NC_MAPPING_INVENTORY_IN_TRANSIT),
        ];
    }
}