<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Helpers;
use App\Models\Setting;
use App\Models\WarehouseTransferShipmentLine;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionLineDataFromWarehouseTransferShipmentLine
{
    public function __construct(
        private readonly WarehouseTransferShipmentLine $line,
    ) {}
    
    /**
     * @throws Throwable
     */
    public function handle(): Collection
    {
        $lineData = collect();
        foreach ($this->getTypeCodeMappings() as $type => $nominalCodeId)
        {
            $lineData->add(AccountingTransactionLineData::from([
                'type' => $type,
                'nominal_code_id' => $nominalCodeId,
                'description' => $this->getDescription(),
                'quantity' => $this->line->quantity,
                'amount' => $this->getAmount(),
                'tax_amount' => 0,
                'tax_rate_id' => null,
                'link_id' => $this->line->id,
                'link_type' => WarehouseTransferShipmentLine::class,
            ]));
        }
        return $lineData;
    }

    private function getDescription(): string
    {
        $warehouseTransferLine = $this->line->warehouseTransferLine;
        return $warehouseTransferLine->product?->sku ?? $warehouseTransferLine->description;
    }

    private function getAmount(): float
    {
        return $this->line->totalAmount;
    }

    private function getTypeCodeMappings(): array
    {
        $warehouse = $this->line->warehouseTransferShipment->warehouseTransfer->fromWarehouse;
        $credit = $warehouse->nominal_code_id ?? Helpers::setting(Setting::KEY_NC_MAPPING_INVENTORY_CONTROL);

        return [
            AccountingTransactionLineTypeEnum::DEBIT->value => Helpers::setting(Setting::KEY_NC_MAPPING_INVENTORY_IN_TRANSIT),
            AccountingTransactionLineTypeEnum::CREDIT->value => $credit,
        ];
    }
}