<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Helpers;
use App\Models\Setting;
use App\Models\StockTakeItem;
use App\Repositories\InventoryMovementRepository;
use App\Repositories\NominalCodeRepository;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionLineDataFromStockTakeItem
{
    private NominalCodeRepository $nominalCodes;
    private InventoryMovementRepository $movements;

    public function __construct(
        private readonly StockTakeItem $line,
    ) {
        $this->nominalCodes = app(NominalCodeRepository::class);
        $this->movements = app(InventoryMovementRepository::class);
    }
    
    /**
     * @throws Throwable
     */
    public function handle(): Collection
    {
        $lineData = collect();
        foreach ($this->getTypeCodeMappings() as $type => $nominalCodeId)
        {
            $lineData->add(AccountingTransactionLineData::from([
                'type' => $type,
                'nominal_code_id' => $nominalCodeId,
                'description' => $this->getDescription(),
                'quantity' => abs($this->line->quantity_adjusted),
                'amount' => $this->getAmount(),
                'tax_amount' => 0,
                'tax_rate_id' => null,
                'link_id' => $this->line->id,
                'link_type' => StockTakeItem::class,
            ]));
        }
        return $lineData;
    }

    private function getTypeCodeMappings(): array
    {
        $cogsNominalCodeId = $this->nominalCodes->getCogsNominalCodeId($this->line);
        $warehouse = $this->line->stockTake->warehouse;
        $inventoryAccount = $warehouse->nominal_code_id ?? Helpers::setting(Setting::KEY_NC_MAPPING_INVENTORY_CONTROL, true);
        return [
            AccountingTransactionLineTypeEnum::DEBIT->value =>
                $this->line->quantity_adjusted > 0 ?
                    $inventoryAccount :
                    $cogsNominalCodeId,
            AccountingTransactionLineTypeEnum::CREDIT->value =>
                $this->line->quantity_adjusted > 0 ?
                    $cogsNominalCodeId :
                    $inventoryAccount,
        ];
    }

    private function getAmount(): float
    {
        return abs($this->line->quantity_adjusted == 0 ?
            0 :
            (
                $this->movements->getTotalAmount($this->line->inventoryMovements) /
                $this->line->quantity_adjusted
            ));
    }

    private function getDescription(): string
    {
        return 'Stock Count Adjustment for ' . $this->line->product->sku;
    }
}