<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Models\SalesCreditLine;
use Throwable;

class BuildAccountingTransactionLineDataFromSalesCreditLine
{
    public function __construct(
        private readonly SalesCreditLine $line,
    ) {}
    
    /**
     * @throws Throwable
     */
    public function handle(): AccountingTransactionLineData
    {
        return AccountingTransactionLineData::from([
            'type' => AccountingTransactionLineTypeEnum::SALES_CREDIT_LINE,
            'nominal_code_id' => $this->line->nominal_code_id,
            'description' => $this->getDescription(),
            'quantity' => $this->line->quantity,
            'amount' => $this->getAmount(),
            'tax_amount' => $this->line->tax_allocation_in_tenant_currency ?? 0,
            'tax_rate_id' => $this->line->tax_rate_id,
            'link_id' => $this->line->id,
            'link_type' => SalesCreditLine::class,
        ]);
    }

    private function getAmount(): float
    {
        return $this->line->quantity == 0 ?
            0 :
            (
                $this->line->salesCredit->is_tax_included ?
                    ($this->line->subtotal_in_tenant_currency - $this->line->tax_allocation_in_tenant_currency) / $this->line->quantity :
                    $this->line->subtotal_in_tenant_currency / $this->line->quantity
            );
    }

    private function getDescription(): string
    {
        return ($this->line->product?->sku ? $this->line->product->sku . ': ' : '') . $this->line->description;
    }
}