<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Helpers;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\Setting;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionLineDataFromPurchaseOrderReceiptLine
{
    public function __construct(
        private readonly PurchaseOrderShipmentReceiptLine $line,
    ) {}
    
    /**
     * @throws Throwable
     */
    public function handle(): Collection
    {
        $lineData = collect();
        foreach ($this->getTypeCodeMappings() as $type => $nominalCodeId)
        {
            $lineData->add(AccountingTransactionLineData::from([
                'type' => $type,
                'nominal_code_id' => $nominalCodeId,
                'description' => $this->getDescription(),
                'quantity' => $this->line->quantity,
                'amount' => $this->getAmount(),
                'tax_amount' => 0,
                'tax_rate_id' => null,
                'link_id' => $this->line->id,
                'link_type' => PurchaseOrderShipmentReceiptLine::class,
            ]));
        }
        return $lineData;
    }

    private function getTypeCodeMappings(): array
    {
        $warehouse = $this->line->purchaseOrderLine->purchaseOrder->destinationWarehouse;
        $debit = $warehouse->nominal_code_id ?? Helpers::setting(Setting::KEY_NC_MAPPING_INVENTORY_CONTROL, true);

        return [
            AccountingTransactionLineTypeEnum::DEBIT->value => $debit,
            AccountingTransactionLineTypeEnum::CREDIT->value => Helpers::setting(Setting::KEY_NC_MAPPING_ACCRUED_PURCHASES, true),
        ];
    }

    private function getAmount(): float
    {
        $purchaseOrderLine = $this->line->purchaseOrderLine;

        // For 0'd out purchase order lines, there will be no prorated amount so just return the value of the line
        if ($purchaseOrderLine->quantity == 0) {
            return $purchaseOrderLine->amount;
        }

        $purchaseOrderLineSubtotal = $purchaseOrderLine->purchaseOrder->is_tax_included ?
            ($purchaseOrderLine->total_cost_in_tenant_currency - $purchaseOrderLine->tax_allocation_in_tenant_currency) :
            $purchaseOrderLine->total_cost_in_tenant_currency;

        return $this->line->quantity == 0 ?
            0 :
            (
                ($this->line->prorationOfPurchaseOrderLine * $purchaseOrderLineSubtotal) /
                $this->line->quantity
            );
    }

    private function getDescription(): string
    {
        $purchaseOrderLine = $this->line->purchaseOrderLine;
        return "Receipt for {$purchaseOrderLine->product->sku}";
    }
}