<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Helpers;
use App\Models\PurchaseInvoiceLine;
use App\Models\Setting;
use Throwable;

class BuildAccountingTransactionLineDataFromPurchaseInvoiceLine
{
    public function __construct(
        private readonly PurchaseInvoiceLine $line,
    ) {}
    
    /**
     * @throws Throwable
     */
    public function handle(): AccountingTransactionLineData
    {
        $purchaseOrderLine = $this->line->purchaseOrderLine;
        return AccountingTransactionLineData::from([
            'type' => AccountingTransactionLineTypeEnum::PURCHASE_INVOICE_LINE,
            'nominal_code_id' => $this->getNominalCodeId(),
            'description' => $this->getDescription(),
            'quantity' => $this->line->quantity_invoiced,
            'amount' => $this->getAmount(),
            'tax_amount' => $purchaseOrderLine->tax_allocation_in_tenant_currency,
            'tax_rate_id' => $purchaseOrderLine->tax_rate_id,
            'link_id' => $this->line->id,
            'link_type' => PurchaseInvoiceLine::class,
        ]);
    }

    private function getDescription(): string
    {
        $purchaseOrderLine = $this->line->purchaseOrderLine;
        return ($purchaseOrderLine->product?->sku ? $purchaseOrderLine->product->sku . ': ' : '') . $purchaseOrderLine->description;
    }

    private function getAmount(): float
    {
        $purchaseOrderLine = $this->line->purchaseOrderLine;

        return $this->line->quantity_invoiced == 0 ?
            0 :
            (
                $purchaseOrderLine->purchaseOrder->is_tax_included ?
                    ($purchaseOrderLine->subtotal_in_tenant_currency - $purchaseOrderLine->tax_allocation_in_tenant_currency) / $purchaseOrderLine->quantity :
                    $purchaseOrderLine->subtotal_in_tenant_currency / $purchaseOrderLine->quantity
            );
    }

    private function getNominalCodeId(): int
    {
        $purchaseOrder = $this->line->purchaseOrderLine->purchaseOrder;
        if ($purchaseOrder->dropshipping || !$purchaseOrder->destinationWarehouse ) {
            return Helpers::setting(Setting::KEY_NC_MAPPING_COGS);
        }

        return Helpers::setting(Setting::KEY_NC_MAPPING_ACCRUED_PURCHASES);
    }
}