<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Helpers;
use App\Models\InventoryAdjustment;
use App\Models\PurchaseOrderLine;
use App\Models\Setting;
use App\Repositories\InventoryMovementRepository;
use App\Repositories\NominalCodeRepository;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionLineDataFromInventoryAdjustment
{
    private NominalCodeRepository $nominalCodes;
    private InventoryMovementRepository $movements;

    public function __construct(
        private readonly InventoryAdjustment $line,
    ) {
        $this->nominalCodes = app(NominalCodeRepository::class);
        $this->movements = app(InventoryMovementRepository::class);
    }
    
    /**
     * @throws Throwable
     */
    public function handle(): Collection
    {
        $lineData = collect();
        foreach ($this->getTypeCodeMappings() as $type => $nominalCodeId)
        {
            $lineData->add(AccountingTransactionLineData::from([
                'type' => $type,
                'nominal_code_id' => $nominalCodeId,
                'description' => $this->getDescription(),
                'quantity' => $this->line->quantity,
                'amount' => $this->getAmount(),
                'tax_amount' => 0,
                'tax_rate_id' => null,
                'link_id' => $this->line->id,
                'link_type' => InventoryAdjustment::class,
            ]));
        }
        return $lineData;
    }

    private function getTypeCodeMappings(): array
    {
        $warehouse = $this->line->warehouse;
        $inventoryAccount = $warehouse->nominal_code_id ?? Helpers::setting(Setting::KEY_NC_MAPPING_INVENTORY_CONTROL, true);
        $cogsNominalCodeId = $this->nominalCodes->getCogsNominalCodeId($this->line);
        $accruedPurchasesNominalCodeId = Helpers::setting(Setting::KEY_NC_MAPPING_ACCRUED_PURCHASES, true);
        if ($this->line->link_type == PurchaseOrderLine::class) {
            $expenseOrAssetAccount = $accruedPurchasesNominalCodeId;
        } else {
            $expenseOrAssetAccount = $cogsNominalCodeId;
        }
        return [
            AccountingTransactionLineTypeEnum::DEBIT->value =>
                $this->line->quantity > 0 ?
                    $inventoryAccount :
                    $expenseOrAssetAccount,
            AccountingTransactionLineTypeEnum::CREDIT->value =>
                $this->line->quantity > 0 ?
                    $expenseOrAssetAccount :
                    $inventoryAccount
        ];
    }

    private function getAmount(): float
    {
        return $this->line->quantity == 0 ?
            0 :
            (
                $this->movements->getTotalAmount($this->line->inventoryMovements) /
                $this->line->quantity
            );
    }

    private function getDescription(): string
    {
        return "Stock Adjustment {$this->line->product->sku}";
    }
}