<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Abstractions\FinancialDocumentInterface;
use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Enums\AccountingTransactionTypeEnum;
use App\Models\Currency;
use App\Models\WarehouseTransferShipmentReceipt;
use App\Models\WarehouseTransferShipmentReceiptLine;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionDataFromWarehouseTransferShipmentReceipt
{
    public function __construct(
        private readonly WarehouseTransferShipmentReceipt|FinancialDocumentInterface $warehouseTransferReceipt,
    ) {}

    /**
     * @throws Throwable
     */
    public function handle(): AccountingTransactionData
    {
        $warehouseTransfer = $this->warehouseTransferReceipt->shipment->warehouseTransfer;
        $linesCollection = $this->getLines();
        $name = $warehouseTransfer->fromWarehouse->name . '->' . $warehouseTransfer->toWarehouse->name;
        $reference = $name . ': ' . $warehouseTransfer->warehouse_transfer_number . ': Receipt ' . $this->warehouseTransferReceipt->id;
        return AccountingTransactionData::from([
            'type' => AccountingTransactionTypeEnum::WAREHOUSE_TRANSFER_RECEIPT,
            'link_id' => $this->warehouseTransferReceipt->id,
            'link_type' => WarehouseTransferShipmentReceipt::class,
            'name' => $name,
            'email' => null,
            'transaction_date' => $this->warehouseTransferReceipt->received_at,
            'reference' => $reference,
            'is_tax_included' => false,
            'currency_code' => Currency::default()->code,
            'currency_rate' => 1,
            'is_batchable' => false,
            'parent_id' => $this->warehouseTransferReceipt->getParentAccountingTransaction()?->id,
            'accounting_transaction_lines' => AccountingTransactionLineData::collection($linesCollection),
            'total' => $linesCollection->where('type', AccountingTransactionLineTypeEnum::DEBIT)
                ->sum(fn ($line) => abs($line->quantity * $line->amount)),
        ]);
    }

    private function getLines(): Collection
    {
        return $this->warehouseTransferReceipt->receiptLines->map(function (WarehouseTransferShipmentReceiptLine $line)
        {
            return (new BuildAccountingTransactionLineDataFromWarehouseTransferShipmentReceiptLine($line))->handle();
        })->flatten();
    }
}