<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Abstractions\FinancialDocumentInterface;
use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Enums\AccountingTransactionTypeEnum;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionDataFromSalesOrderFulfillment
{
    public function __construct(
        private readonly SalesOrderFulfillment|FinancialDocumentInterface $salesOrderFulfillment,
    ) {}

    /**
     * @throws Throwable
     */
    public function handle(): AccountingTransactionData
    {
        $salesOrderFulfillment = $this->salesOrderFulfillment;

        // Validate financial cache before doing calculations that rely on it
        if ($salesOrderFulfillment->salesOrder) {
            $manager = new SalesOrderLineFinancialManager(Arr::wrap($salesOrderFulfillment->salesOrder->id));
            $manager->calculate();
        }

        $name = $this->salesOrderFulfillment->salesOrder->customer?->name;
        $reference = 'Fulfillment ' . $salesOrderFulfillment->fulfillment_number;
        if ($name) {
            $reference = $name . ': ' . $reference;
        }
        $linesCollection = $this->getLines();
        return AccountingTransactionData::from([
            'type' => AccountingTransactionTypeEnum::SALES_ORDER_FULFILLMENT,
            'link_id' => $salesOrderFulfillment->id,
            'link_type' => SalesOrderFulfillment::class,
            'name' => $name,
            'email' => $this->salesOrderFulfillment->salesOrder->customer?->email,
            'transaction_date' => $salesOrderFulfillment->fulfilled_at,
            'reference' => $reference,
            'is_tax_included' => false,
            'currency_code' => null,
            'currency_rate' => null,
            'is_batchable' => false,
            'parent_id' => $salesOrderFulfillment->getParentAccountingTransaction()?->id,
            'accounting_transaction_lines' => AccountingTransactionLineData::collection($linesCollection),
            'total' => $linesCollection->where('type', AccountingTransactionLineTypeEnum::DEBIT)
                ->sum(fn ($line) => abs($line->quantity * $line->amount)),
        ]);
    }

    private function getLines(): Collection
    {
        return $this->salesOrderFulfillment->salesOrderFulfillmentLines->map(function (SalesOrderFulfillmentLine $line)
        {
            return (new BuildAccountingTransactionLineDataFromSalesOrderFulfillmentLine($line))->handle();
        })->flatten();
    }
}