<?php

namespace App\Services\Accounting\Actions\FinancialDocuments;

use App\Abstractions\FinancialDocumentInterface;
use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionLineData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Enums\AccountingTransactionTypeEnum;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\PurchaseOrderShipmentReceiptLine;
use Illuminate\Support\Collection;
use Throwable;

class BuildAccountingTransactionDataFromPurchaseOrderReceipt
{
    public function __construct(
        private readonly PurchaseOrderShipmentReceipt|FinancialDocumentInterface $purchaseOrderReceipt,
    ) {}

    /**
     * @throws Throwable
     */
    public function handle(): AccountingTransactionData
    {
        $purchaseOrderReceipt = $this->purchaseOrderReceipt;
        $purchaseOrderShipment = $purchaseOrderReceipt->purchaseOrderShipment;
        $purchaseOrder = $purchaseOrderShipment->purchaseOrder;

        $linesCollection = $this->getLines();
        $name = $purchaseOrderReceipt->purchaseOrderShipment->purchaseOrder->supplier->name;
        $reference = "Receipt #{$purchaseOrderReceipt->id} on $purchaseOrder->purchase_order_number";
        if ($name) {
            $reference = $name . ': ' . $reference;
        }
        return AccountingTransactionData::from([
            'type' => AccountingTransactionTypeEnum::PURCHASE_ORDER_RECEIPT,
            'link_id' => $purchaseOrderReceipt->id,
            'link_type' => PurchaseOrderShipmentReceipt::class,
            'name' => $name,
            'email' => $purchaseOrderReceipt->purchaseOrderShipment->purchaseOrder->supplier->email,
            'transaction_date' => $purchaseOrderReceipt->received_at,
            'reference' => $reference,
            'is_tax_included' => false,
            'currency_code' => null,
            'currency_rate' => null,
            'is_batchable' => false,
            'parent_id' => $purchaseOrderReceipt->getParentAccountingTransaction()?->id,
            'accounting_transaction_lines' => AccountingTransactionLineData::collection($linesCollection),
            'total' => $linesCollection->where('type', AccountingTransactionLineTypeEnum::DEBIT)
                ->sum(fn ($line) => abs($line->quantity * $line->amount)),
        ]);
    }

    private function getLines(): Collection
    {
        return $this->purchaseOrderReceipt->purchaseOrderShipmentReceiptLines->map(function (PurchaseOrderShipmentReceiptLine $line)
        {
            return (new BuildAccountingTransactionLineDataFromPurchaseOrderReceiptLine($line))->handle();
        })->flatten();
    }
}