<?php

namespace App\Services\Accounting;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Data\AccountingBulkReplaceNominalCodesData;
use App\Data\AccountingTransactionBulkEnableSyncData;
use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionLineData;
use App\Data\AccountingTransactionLineNominalCodeUpdateData;
use App\Data\AccountingTransactionUpdateData;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Enums\AccountingTransactionTypeEnum;
use App\Exceptions\ReceivingDiscrepanciesMissingNominalCodeMappingException;
use App\Exceptions\ReceivingDiscrepancyAlreadyExistsException;
use App\Helpers;
use App\Models\AccountingTransaction;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\SalesOrder;
use App\Models\Setting;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferLine;
use App\Repositories\Accounting\AccountingTransactionRepository;
use App\Repositories\PurchaseOrderLineRepository;
use App\Repositories\WarehouseTransferLineRepository;
use App\Services\Accounting\Actions\BuildAccountingTransactionsForCreate;
use App\Services\Accounting\Actions\BuildAccountingTransactionsForUpdate;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Pipeline;
use Throwable;

class AccountingTransactionManager
{
    const RECORDS_LIMIT = 100;

    public function __construct(
        private readonly AccountingTransactionRepository $transactions,
        private readonly PurchaseOrderLineRepository $purchaseOrderLines,
        private readonly WarehouseTransferLineRepository $warehouseTransferLines,
    ) {}

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function sync(array $ids = []): void
    {
        //Here offset and limits are used to avoid infinite loop
        $limit = self::RECORDS_LIMIT;
        $offset = 0;

        do {
            $accountingTransactionCollection = collect();

            $this->transactions->deleteTransactionsWithoutLink();
            $this->transactions->deleteTransactionLinesWithoutLink();

            $accountingTransactionCollection = $this->initializeTransactionCollection($ids, $limit, $offset);

            // Update always, Create only if no ids are provided
            $pipeline = [
                BuildAccountingTransactionsForUpdate::class,
            ];

            if (empty($ids)) {
                $pipeline[] = BuildAccountingTransactionsForCreate::class;
            }

            $accountingTransactionCollection = Pipeline::send($accountingTransactionCollection)
                ->through($pipeline)
                ->through(array_map(function ($pipe) use ($limit, $offset) {
                    return function ($accountingTransactionCollection, $next) use ($pipe, $limit, $offset) {
                        return app($pipe)->handle($accountingTransactionCollection, $limit, $offset, $next);
                    };
                }, $pipeline))
                ->thenReturn()->flatten();

            $this->transactions->saveWithRelations(AccountingTransactionData::collection($accountingTransactionCollection));

            $offset = $offset + $limit;

        } while ($accountingTransactionCollection->count() != 0);
    }

    private function initializeTransactionCollection(array $ids, int $limit, int $offset): Collection
    {
        return ! empty($ids) ?
            $this->transactions->getForValues(
                $ids,
                'id',
                AccountingTransaction::class,
                [
                    'accountingTransactionLines',
                ],
                1000,
                $limit,
                $offset
            ) : collect();
    }

    public function clearErrors(array $ids = []): void
    {
        $this->transactions->clearErrors($ids);
    }

    /**
     * @throws Throwable
     */
    public function saveAccountingTransaction(AccountingTransactionData $data): AccountingTransaction
    {
        return $this->transactions->saveWithRelations(AccountingTransactionData::collection([$data]))->first();
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function createReceivingDiscrepancyFromPurchaseOrder(PurchaseOrder $purchaseOrder): AccountingTransaction
    {
        if (!Helpers::setting(Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES)) {
            throw new ReceivingDiscrepanciesMissingNominalCodeMappingException('Nominal code for receiving discrepancies is not set');
        }

        if ($purchaseOrder->receivingDiscrepancy) {
            throw new ReceivingDiscrepancyAlreadyExistsException('Receiving discrepancy already exists for purchase order ' . $purchaseOrder->purchase_order_number);
        }

        $accountingTransactionLines = $this->createReceivingDiscrepanciesLineDataFromPurchaseOrder($purchaseOrder);

        $accountingTransactionData = AccountingTransactionData::from([
            'transaction_date' => $purchaseOrder->purchase_order_date,
            'type' => AccountingTransactionTypeEnum::RECEIVING_DISCREPANCY,
            'name' => $purchaseOrder->supplier->name,
            'reference' => $purchaseOrder->supplier->name . ': Receiving discrepancy for ' . $purchaseOrder->purchase_order_number,
            'link_id' => $purchaseOrder->id,
            'link_type' => PurchaseOrder::class,
            'accounting_transaction_lines' => $accountingTransactionLines,
        ]);

        $purchaseOrder->receipt_status = PurchaseOrder::RECEIPT_STATUS_RECEIVED;
        $purchaseOrder->order_status = PurchaseOrder::STATUS_CLOSED;
        $purchaseOrder->shipment_status = PurchaseOrder::SHIPMENT_STATUS_SHIPPED_WAREHOUSE;
        $purchaseOrder->save();

        return $this->saveAccountingTransaction($accountingTransactionData);
    }

    private function createReceivingDiscrepanciesLineDataFromPurchaseOrder(PurchaseOrder $purchaseOrder): Collection
    {
        $underReceivedTransactionTypes = [
            ['type' => AccountingTransactionLineTypeEnum::DEBIT, 'nominal_code_id' => Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES],
            ['type' => AccountingTransactionLineTypeEnum::CREDIT, 'nominal_code_id' => Setting::KEY_NC_MAPPING_ACCRUED_PURCHASES]
        ];

        $overReceivedTransactionTypes = [
            ['type' => AccountingTransactionLineTypeEnum::DEBIT, 'nominal_code_id' => Setting::KEY_NC_MAPPING_ACCRUED_PURCHASES],
            ['type' => AccountingTransactionLineTypeEnum::CREDIT, 'nominal_code_id' => Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES]
        ];

        return $this->purchaseOrderLines->getUnderReceivedLinesForPurchaseOrder($purchaseOrder)
            ->map(function (PurchaseOrderLine $purchaseOrderLine) use ($underReceivedTransactionTypes) {
                return $this->buildReceivingDiscrepanciesLineDataFromPurchaseOrderLine($purchaseOrderLine, $underReceivedTransactionTypes, $purchaseOrderLine->unreceived_quantity);
            })
            ->concat($this->purchaseOrderLines->getOverReceivedLinesForPurchaseOrder($purchaseOrder)
                ->map(function (PurchaseOrderLine $purchaseOrderLine) use ($overReceivedTransactionTypes) {
                    return $this->buildReceivingDiscrepanciesLineDataFromPurchaseOrderLine($purchaseOrderLine, $overReceivedTransactionTypes, $purchaseOrderLine->received_quantity - $purchaseOrderLine->quantity);
                }))
            ->flatten(1);
    }

    private function buildReceivingDiscrepanciesLineDataFromPurchaseOrderLine(
        PurchaseOrderLine $purchaseOrderLine,
        array $transactionTypes,
        float $quantity,
    ): Collection
    {
        return collect($transactionTypes)->map(function ($transactionType) use ($purchaseOrderLine, $quantity) {
            return AccountingTransactionLineData::from([
                'type' => $transactionType['type'],
                'nominal_code_id' => Helpers::setting($transactionType['nominal_code_id']),
                'description' => 'Receiving discrepancy for ' . $purchaseOrderLine->product?->sku ?? $purchaseOrderLine->description,
                'quantity' => $quantity,
                'amount' => $purchaseOrderLine->total_cost_in_tenant_currency / $purchaseOrderLine->quantity,
                'link_id' => $purchaseOrderLine->id,
                'link_type' => PurchaseOrderLine::class,
            ]);
        });
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function createReceivingDiscrepancyFromWarehouseTransfer(WarehouseTransfer $warehouseTransfer): AccountingTransaction
    {
        if (!Helpers::setting(Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES)) {
            throw new ReceivingDiscrepanciesMissingNominalCodeMappingException('Nominal code for receiving discrepancies is not set');
        }

        if ($warehouseTransfer->receivingDiscrepancy) {
            throw new ReceivingDiscrepancyAlreadyExistsException('Receiving discrepancy already exists for warehouse transfer ' . $warehouseTransfer->warehouse_transfer_number);
        }

        $accountingTransactionLines = $this->createReceivingDiscrepanciesLineDataFromWarehouseTransfer($warehouseTransfer);

        $accountingTransactionData = AccountingTransactionData::from([
            'transaction_date' => $warehouseTransfer->transfer_date,
            'type' => AccountingTransactionTypeEnum::RECEIVING_DISCREPANCY,
            'name' => $warehouseTransfer->toWarehouse->name,
            'reference' => $warehouseTransfer->fromWarehouse->name . '->' . $warehouseTransfer->toWarehouse->name . ': Receiving discrepancy for ' . $warehouseTransfer->warehouse_transfer_number,
            'link_id' => $warehouseTransfer->id,
            'link_type' => WarehouseTransfer::class,
            'accounting_transaction_lines' => $accountingTransactionLines,
        ]);

        $warehouseTransfer->receipt_status = WarehouseTransfer::TRANSFER_RECEIPT_STATUS_RECEIVED;
        $warehouseTransfer->transfer_status = WarehouseTransfer::TRANSFER_STATUS_CLOSED;
        $warehouseTransfer->shipment_status = WarehouseTransfer::TRANSFER_SHIPMENT_STATUS_SHIPPED;
        $warehouseTransfer->save();

        return $this->saveAccountingTransaction($accountingTransactionData);
    }

    private function createReceivingDiscrepanciesLineDataFromWarehouseTransfer(WarehouseTransfer $warehouseTransfer): Collection
    {
        $underReceivedTransactionTypes = [
            ['type' => AccountingTransactionLineTypeEnum::DEBIT, 'nominal_code_id' => Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES],
            ['type' => AccountingTransactionLineTypeEnum::CREDIT, 'nominal_code_id' => Setting::KEY_NC_MAPPING_INVENTORY_IN_TRANSIT]
        ];

        $overReceivedTransactionTypes = [
            ['type' => AccountingTransactionLineTypeEnum::DEBIT, 'nominal_code_id' => Setting::KEY_NC_MAPPING_INVENTORY_IN_TRANSIT],
            ['type' => AccountingTransactionLineTypeEnum::CREDIT, 'nominal_code_id' => Setting::KEY_NC_MAPPING_RECEIVING_DISCREPANCIES]
        ];

        return $this->warehouseTransferLines->getUnderReceivedLinesForWarehouseTransfer($warehouseTransfer)
            ->map(function (WarehouseTransferLine $warehouseTransferLine) use ($underReceivedTransactionTypes) {
                return $this->buildReceivingDiscrepanciesLineDataFromWarehouseTransferLine($warehouseTransferLine, $underReceivedTransactionTypes, $warehouseTransferLine->quantity_unreceived);
            })
            ->concat($this->warehouseTransferLines->getOverReceivedLinesForWarehouseTransfer($warehouseTransfer)
                ->map(function (WarehouseTransferLine $warehouseTransferLine) use ($overReceivedTransactionTypes) {
                    return $this->buildReceivingDiscrepanciesLineDataFromWarehouseTransferLine($warehouseTransferLine, $overReceivedTransactionTypes, $warehouseTransferLine->quantity_received - $warehouseTransferLine->quantity);
                }))
            ->flatten(1);
    }

    private function buildReceivingDiscrepanciesLineDataFromWarehouseTransferLine(
        WarehouseTransferLine $warehouseTransferLine,
        array $transactionTypes,
        float $quantity,
    ): Collection
    {
        return collect($transactionTypes)->map(function ($transactionType) use ($warehouseTransferLine, $quantity) {
            return AccountingTransactionLineData::from([
                'type' => $transactionType['type'],
                'nominal_code_id' => Helpers::setting($transactionType['nominal_code_id']),
                'description' => 'Receiving discrepancy for ' . $warehouseTransferLine->product->sku,
                'quantity' => $quantity,
                'amount' => $warehouseTransferLine->shipmentLine->inventoryMovements->first()->fifo_layer->avg_cost,
                'link_id' => $warehouseTransferLine->id,
                'link_type' => WarehouseTransferLine::class,
            ]);
        });
    }

    public function updateTransaction(AccountingTransaction $accountingTransaction, AccountingTransactionUpdateData $data): AccountingTransaction
    {
        $data->lines?->each(function (AccountingTransactionLineNominalCodeUpdateData $lineData) use (
            $accountingTransaction
        ) {
            $line = $accountingTransaction->accountingTransactionLines->firstWhere('id', $lineData->id);
            $line->nominal_code_id = $lineData->nominal_code_id;
            $line->save();
        });

        $accountingTransaction->fill($data->only('is_sync_enabled', 'is_locked')->toArray());
        $accountingTransaction->timestamps = !empty($data->lines);
        $accountingTransaction->save();

        return $accountingTransaction->refresh()->load('accountingTransactionLines.nominalCode');
    }

    public function bulkReplaceNominalCodes(AccountingBulkReplaceNominalCodesData $data): void
    {
        $this->transactions->bulkReplaceNominalCodes($data);
    }

    public function syncAccountingTransactionsSalesOrderIsSyncEnabledStatus(IntegrationInstanceInterface $integrationInstance): void
    {
        Model::withoutTimestamps( fn()=> AccountingTransaction::whereHasMorph(
            'link',
            [SalesOrder::class],
            function ($query) use ($integrationInstance) {
                $query->where('sales_channel_id', $integrationInstance->salesChannel->id);
            }
        )->update([
            'is_sync_enabled' => $integrationInstance->integration_settings['sync_sales_order_invoices_to_accounting'] ?? true
        ]));
    }

    public function bulkEnableSync(AccountingTransactionBulkEnableSyncData $data): void
    {
        $this->transactions->bulkEnableSync($data);
    }
}
