<?php

namespace App\SDKs\Starshipit;

use App\Models\ApiLog;
use Illuminate\Support\Str;

abstract class Starshipit
{
    /**
     * Base url of ShipStation API.
     *
     * @var string
     */
    protected $base_url = 'https://api.starshipit.com/api';

    /**
     * The endpoint associated with the service.
     *
     * @var string
     */
    protected $serviceEndpoint;

    /**
     * Http Guzzle Client.
     */
    public $client;

    /**
     * API Configurations.
     *
     * @var null[]
     */
    public static $config = [
        'apiKey' => null,
        'subscriptionKey' => null,
    ];

    /**
     * Last URL requested.
     *
     * @var string
     */
    private $lastRequestUrl;

    /**
     * @var StarshipitResponse
     */
    private $lastResponse;

    public static ?int $integrationInstanceId;

    /**
     * ShipStation constructor.
     */
    public function __construct(?string $apiKey = null, ?string $subscriptionKey = null)
    {
        if (! empty($apiKey)) {
            static::$config['apiKey'] = $apiKey;
        }

        if (! empty($subscriptionKey)) {
            static::$config['subscriptionKey'] = $subscriptionKey;
        }
    }

    /**
     * Call the endpoint from ShipStation.
     *
     * <warn> Don't use getContents() function to get content from the response OR reset the pointer </warn>
     *
     *
     * @return StarshipitResponse
     *
     * @throws StarshipitException
     */
    protected function request(string $actionUrl = '', ?string $body = null, string $method = 'GET')
    {
        try {
            $curl = curl_init();
            $header = [
                'Content-Type: application/json',
                'StarShipIT-Api-Key: '.static::$config['apiKey'],
                'Ocp-Apim-Subscription-Key: '.static::$config['subscriptionKey'],
            ];
            curl_setopt_array($curl, [
                CURLOPT_URL => $this->base_url.'/'.$this->getRequestUrl($actionUrl),
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => '',
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 0,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => $method,
                CURLOPT_POSTFIELDS => $body,
                CURLOPT_HTTPHEADER => $header,
            ]);
            $response = curl_exec($curl);
            $httpcode = curl_getinfo($curl, CURLINFO_HTTP_CODE);

            curl_close($curl);

            // Rate limit is exceeded. Try again in 1 seconds.
            if ($httpcode == 429) {
                sleep(3); // sleep 3 seconds in case there are other requests

                return $this->request($actionUrl, $body, $method);
            }

            $this->lastResponse = new StarshipitResponse(json_decode($response, true), $httpcode);
            ApiLog::query()->create([
                'integration_instance_id' => static::$integrationInstanceId,
                'url' => $this->base_url.'/'.$this->getRequestUrl($actionUrl),
                'requestHeaders' => $header,
                'requestBody' => ! is_null($body) ? $body : $actionUrl,
                'responseStatusCode' => $httpcode,
                'responseHeaders' => [],
                'responseBody' => $this->lastResponse->body['message'] ?? $this->lastResponse->body['Message'] ?? $response,
            ]);
            if ($httpcode >= 300) {
                throw new StarshipitException($this->lastResponse->body['message'] ?? $this->lastResponse->body['Message'] ?? $response, $httpcode);
            }

            return $this->lastResponse;
        } catch (\Throwable $exception) {
            $message = $exception->getMessage();
            $code = $httpcode ?? 0;
            $this->lastResponse = new StarshipitResponse($message, $code);

            ApiLog::query()->create([
                'integration_instance_id' => static::$integrationInstanceId,
                'url' => $this->base_url.'/'.$this->getRequestUrl($actionUrl),
                'requestHeaders' => $header,
                'requestBody' => ! is_null($body) ? $body : $actionUrl,
                'responseHeaders' => [],
                'responseStatusCode' => $exception->getCode(),
                'responseBody' => $this->lastResponse->body,
            ]);
            throw new StarshipitException($message, $code);
        }
    }

    /**
     * Get the endpoint associated with the service.
     *
     * @return string
     */
    public function getServiceEndpoint()
    {
        return $this->serviceEndpoint ? trim($this->serviceEndpoint, '/') : mb_strtolower(
            Str::plural(class_basename($this))
        );
    }

    private function getRequestUrl(string $actionUrl = '')
    {
        $actionUrl = trim($actionUrl, '/');

        $this->lastRequestUrl = $this->getServiceEndpoint();
        $this->lastRequestUrl .= Str::startsWith($actionUrl, '?') ? $actionUrl : "/{$actionUrl}";
        $this->lastRequestUrl = trim($this->lastRequestUrl, '/');

        return $this->lastRequestUrl;
    }

    /**
     * Set API Configurations.
     */
    public static function config(string $apiKey, string $subscriptionKey, ?int $integrationInstanceId = null)
    {
        static::$config['apiKey'] = $apiKey;
        static::$config['subscriptionKey'] = $subscriptionKey;
        static::$integrationInstanceId = $integrationInstanceId;
    }

    public function getLastRequestUrl(): string
    {
        return $this->lastRequestUrl;
    }

    public function getLastResponse(): StarshipitResponse
    {
        return $this->lastResponse;
    }
}
