<?php

namespace App\SDKs\Starshipit\Services;

use App\SDKs\Starshipit\Model\Order;
use App\SDKs\Starshipit\Starshipit;
use App\SDKs\Starshipit\StarshipitException;
use App\SDKs\Starshipit\StarshipitResponse;
use InvalidArgumentException;
use Symfony\Component\HttpFoundation\Request;

class Orders extends Starshipit
{
    /**
     * Create Order.
     *
     *
     * @throws StarshipitException
     */
    public function create(Order $order): StarshipitResponse
    {
        return $this->request('', json_encode(['order' => $order->toArray()]), Request::METHOD_POST);
    }

    /**
     * Create Orders.
     *
     * @param  Order[]  $orders
     *
     * @throws StarshipitException
     */
    public function import(array $orders): StarshipitResponse
    {
        return $this->request('/import', json_encode(['orders' => array_map(function (Order $order) {
            return $order->toArray();
        }, $orders)]), Request::METHOD_POST);
    }

    /**
     * Update Order.
     *
     *
     * @throws StarshipitException|InvalidArgumentException
     */
    public function update(Order $order): StarshipitResponse
    {
        if (empty($order->order_id)) {
            throw new InvalidArgumentException('The order_id is required for updating order');
        }

        return $this->request('', json_encode(['order' => $order->toArray()]), Request::METHOD_PUT);
    }

    /**
     * Get Order Details.
     *
     *
     * @throws StarshipitException
     */
    public function get(int|string $id, string $idType = 'order_id'): StarshipitResponse
    {
        return $this->request("?{$idType}={$id}");
    }

    /**
     * Get Unshipped Orders
     *
     *
     * @throws StarshipitException
     */
    public function getUnshipped(array $options = []): StarshipitResponse
    {
        $query = http_build_query($options);

        return $this->request('/unshipped'.($query ? "?{$query}" : ''));
    }

    /**
     * Get Shipped Orders
     *
     *
     * @throws StarshipitException
     */
    public function getShipped(array $options = []): StarshipitResponse
    {
        $query = http_build_query($options);

        return $this->request('/shipped'.($query ? "?{$query}" : ''));
    }

    /**
     * Search printed orders
     *
     *
     * @throws StarshipitException|InvalidArgumentException
     */
    public function search(array $options = []): StarshipitResponse
    {
        if (empty($options['phrase'])) {
            throw new InvalidArgumentException('The phrase parameter is required');
        }

        $query = http_build_query($options);

        return $this->request('/search'.($query ? "?{$query}" : ''));
    }

    /**
     * Delete Order Details.
     *
     *
     * @throws StarshipitException
     */
    public function delete(int $orderId): StarshipitResponse
    {
        return $this->request("delete?order_id={$orderId}", null, Request::METHOD_DELETE);
    }

    /**
     * Restore an archived order.
     *
     *
     * @throws StarshipitException
     */
    public function restore(int $orderId): StarshipitResponse
    {
        return $this->request("restore?order_id={$orderId}", null, Request::METHOD_POST);
    }
}
