<?php

namespace App\SDKs\ShipStation;

use App\Exceptions\ShipStationGatewayTimeoutException;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use Psr\Http\Message\ResponseInterface;
use Symfony\Component\HttpFoundation\Request;

abstract class ShipStation
{
    /**
     * Base url of ShipStation API.
     *
     * @var string
     */
    protected $base_url = 'https://ssapi.shipstation.com';

    /**
     * The endpoint associated with the service.
     *
     * @var string
     */
    protected $serviceEndpoint;

    /**
     * Http Guzzle Client.
     */
    public $client;

    /**
     * API Configurations.
     *
     * @var null[]
     */
    public static $config = [
        'apiKey' => null,
        'apiSecret' => null,
    ];

    /**
     * Last URL requested.
     *
     * @var string
     */
    private $lastRequestUrl;

    /**
     * @var ResponseInterface
     */
    private $lastResponse;

    /**
     * ShipStation constructor.
     */
    public function __construct(?string $apiKey = null, ?string $apiSecret = null)
    {
        if (! empty($apiKey)) {
            static::$config['apiKey'] = $apiKey;
        }

        if (! empty($apiSecret)) {
            static::$config['apiSecret'] = $apiSecret;
        }

        /**
         * New Instance from Guzzle Client with base url and basic authentication.
         */
        $this->client = new Client(
            [
                'base_uri' => $this->base_url,
                'auth' => [static::$config['apiKey'], static::$config['apiSecret']],
                'headers' => [
                    'Content-Type' => 'application/json',
                ],
            ]
        );
    }

    /**
     * Call the endpoint from ShipStation.
     *
     * <warn> Don't use getContents() function to get content from the response OR reset the pointer </warn>
     *
     * @param  null  $body
     * @param  string  $method
     * @return array
     *
     * @throws ShipStationGatewayTimeoutException|ShipStationException
     */
    protected function request(string $actionUrl = '', $body = null, $method = 'GET')
    {
        try {
            $this->lastResponse = $this->client->request($method, $this->getRequestUrl($actionUrl), $method == Request::METHOD_DELETE ? [] : ['body' => $body]);

            return json_decode((string) $this->lastResponse->getBody(), true);
        } catch (GuzzleException $exception) {
            if ($exception->hasResponse()) {
                $this->lastResponse = $exception->getResponse();

                $responseContents = (string) $this->lastResponse->getBody();
                $message = $responseContents;
                $code = $exception->getResponse()->getStatusCode();
            } else {
                $message = $exception->getMessage();
                $code = $exception->getCode();
            }

            customlog('shipstation', 'shipstation error', [
                'message' => $message,
                'code' => $code,
            ]);

            if ($code == 504) {
                throw new ShipStationGatewayTimeoutException('ShipStation Gateway Timeout.  Please try again later.');
            }

            customlog('SKU-6888', 'Unhandled Shipstation Error', [
                'message' => $message,
                'code' => $code,
            ], 7);
            throw new ShipStationException($message, $code);
        }
    }

    /**
     * Call the endpoint from ShipStation.
     *
     * <warn> Don't use getContents() function to get content from the response OR reset the pointer </warn>
     *
     * @param  null  $body
     * @param  string  $method
     * @return array
     */
    protected function customRequest(string $actionUrl = '', $body = null, $method = 'GET')
    {
        try {
            $this->client = new Client(
                [
                    'auth' => [static::$config['apiKey'], static::$config['apiSecret']],
                    'headers' => [
                        'Content-Type' => 'application/json',
                    ],
                ]
            );

            $this->lastRequestUrl = $actionUrl;
            $this->lastResponse = $this->client->request($method, $actionUrl, ['body' => $body]);

            return json_decode((string) $this->lastResponse->getBody(), true);
        } catch (GuzzleException $exception) {
            if ($exception->hasResponse()) {
                $this->lastResponse = $exception->getResponse();

                $responseContents = (string) $this->lastResponse->getBody();

                $message = ($arrayResponse = json_decode($responseContents, true)) ? $arrayResponse['ExceptionMessage'] : $responseContents;
                $code = $exception->getResponse()->getStatusCode();
            } else {
                $message = $exception->getMessage();
                $code = $exception->getCode();
            }

            customlog('shipstation', 'shipstation error', [
                'message' => $message,
                'code' => $code,
            ]);

            throw new ShipStationException($message, $code);
        }
    }

    /**
     * Get the endpoint associated with the service.
     *
     * @return string
     */
    public function getServiceEndpoint()
    {
        return $this->serviceEndpoint ? trim($this->serviceEndpoint, '/') : mb_strtolower(Str::plural(class_basename($this)));
    }

    public function getRequestUrl(string $actionUrl = '')
    {
        $actionUrl = trim($actionUrl, '/');

        $this->lastRequestUrl = $this->getServiceEndpoint();
        $this->lastRequestUrl .= Str::startsWith($actionUrl, '?') ? $actionUrl : "/{$actionUrl}";

        return $this->lastRequestUrl;
    }

    /**
     * Set API Configurations.
     */
    public static function config(string $apiKey, string $apiSecret)
    {
        static::$config['apiKey'] = $apiKey;
        static::$config['apiSecret'] = $apiSecret;
    }

    public function getLastRequestUrl(): string
    {
        return $this->lastRequestUrl;
    }

    public function getLastResponse(): ResponseInterface
    {
        return $this->lastResponse;
    }
}
