<?php

namespace App\SDKs\ShipStation\Model;

/**
 * Class Order.
 *
 * @property-read int $orderId The system generated identifier for the order. This is a read-only field.
 * @property string $orderNumber A user-defined order number used to identify an order.
 * @property string $orderKey A user-provided key that should be unique to each order.
 * @property string $orderDate The date the order was placed.
 * @property-read string $createDate Read-Only.
 * @property-read string $modifyDate Read-Only.
 * @property string $paymentDate The date the order was paid for.
 * @property string $shipByDate The date the order is to be shipped before or on.
 * @property string $orderStatus {@see Order::STATUSES
 * @property-read string $customerId Unique identifier for the customer. Generated by ShipStation the first time the customer record is created. Read-Only
 * @property string $customerUsername The customer's username. Note: This property needs to be defined in order to generate a customer profile
 * @property string $customerEmail
 * @property Address $billTo
 * @property Address $shipTo
 * @property OrderItem[] $items
 * @property float $orderTotal
 * @property float $amountPaid
 * @property float $taxAmount
 * @property string $customerNotes
 * @property string $internalNotes
 * @property bool $gift
 * @property string $giftMessage
 * @property string $paymentMethod
 * @property string $requestedShippingService
 * @property string $carrierCode
 * @property string $serviceCode
 * @property string $packageCode
 * @property string $confirmation The type of delivery confirmation that is to be used(or was used) when this order is shipped(was shipped)
 * @property string $shipDate
 * @property string $holdUntilDate
 * @property Weight $weight
 * @property Dimensions $dimensions
 * @property InsuranceOptions $insuranceOptions
 * @property InternationalOptions $internationalOptions
 * @property AdvancedOptions $advancedOptions
 * @property array $tagIds
 * @property-read string $userId User assigned to Order/Shipment in the GUID. Read-Only.
 * @property-read bool $externallyFulfilled States whether the order has is current marked as externally fulfilled by the marketplace. Read-Only.
 * @property-read string $externallyFulfilledBy If externallFulfilled is true then this string will return how the order is being fulfilled by the marketplace. Read-Only.
 */
class Order extends Model
{
    use Validatable;

    /**
     * Order Statuses.
     */
    const STATUS_AWAITING_PAYMENT = 'awaiting_payment';

    const STATUS_AWAITING_SHIPMENT = 'awaiting_shipment';

    const STATUS_SHIPPED = 'shipped';

    const STATUS_ON_HOLD = 'on_hold';

    const STATUS_CANCELED = 'cancelled';

    const STATUSES = [
        self::STATUS_AWAITING_PAYMENT,
        self::STATUS_AWAITING_SHIPMENT,
        self::STATUS_SHIPPED,
        self::STATUS_ON_HOLD,
        self::STATUS_CANCELED,
    ];

    /**
     * The type of delivery confirmation.
     */
    const TYPE_CONFIRMATION_NONE = 'none';

    const TYPE_CONFIRMATION_DELIVERY = 'delivery';

    const TYPE_CONFIRMATION_SIGNATURE = 'signature';

    const TYPE_CONFIRMATION_ADULT_SIGNATURE = 'adult_signature';

    const TYPE_CONFIRMATION_DIRECT_SIGNATURE = 'direct_signature'; // direct_signature is available for FedEx only.

    const TYPES_CONFIRMATION = [
        self::TYPE_CONFIRMATION_NONE,
        self::TYPE_CONFIRMATION_DELIVERY,
        self::TYPE_CONFIRMATION_SIGNATURE,
        self::TYPE_CONFIRMATION_ADULT_SIGNATURE,
    ];

    protected $casts = [
        'billTo' => Address::class,
        'shipTo' => Address::class,
        'weight' => Weight::class,
        'insuranceOptions' => InsuranceOptions::class,
        'internationalOptions' => InternationalOptions::class,
        //'advancedOptions'      => AdvancedOptions::class,
    ];

    public function getItemsAttribute()
    {
        $items = [];
        foreach ($this->attributes['items'] as $item) {
            $items[] = $item instanceof OrderItem ? $item : new OrderItem($item);
        }

        return $items;
    }

    /**
     * Rules of model.
     */
    public function rules(): array
    {
        return [
            'orderNumber' => 'required',
            'orderKey' => 'string',
            'orderDate' => 'required|date:Y-m-d h:m:s',
            'orderStatus' => 'required|in:'.implode(',', self::STATUSES),
            'billTo' => 'required|array',
            'shipTo' => 'required|array',
            'shipTo.street1' => 'required',
            'shipTo.postalCode' => 'required',
            'shipTo.city' => 'required',
            'shipTo.country' => 'required',
            'items' => 'array',
            'items.*.name' => 'required',
            'gift' => 'nullable|boolean',
            'confirmation' => 'nullable|in:'.implode(',', self::TYPES_CONFIRMATION),
            'weight.value' => 'nullable|numeric',
            'weight.units' => 'in:'.implode(',', Weight::UNITS),
            'dimensions.length' => 'nullable|numeric',
            'dimensions.width' => 'nullable|numeric',
            'dimensions.height' => 'nullable|numeric',
            'dimensions.units' => 'nullable|in:'.implode(',', Dimensions::UNITS),
            'tagIds' => 'array',
        ];
    }

    /**
     * {@inheritDoc}
     */
    public function customAttributes(): array
    {
        return [
            'shipTo.street1' => 'Shipping details are missing, review Shipping Address line 1',
            'shipTo.postalCode' => 'Shipping details are missing, review Shipping Address Postal Code',
            'shipTo.city' => 'Shipping details are missing, review Shipping Address City',
            'shipTo.country' => 'Shipping details are missing, review Shipping Address Country',
        ];
    }
}
