<?php

namespace App\SDKs\Magento;

use OAuth\Common\Consumer\Credentials;
use OAuth\Common\Consumer\CredentialsInterface;
use OAuth\Common\Http\Uri\UriInterface;
use OAuth\OAuth1\Signature\Exception\UnsupportedHashAlgorithmException;
use OAuth\OAuth1\Signature\SignatureInterface;

class Signature implements SignatureInterface
{
    /**
     * @var Credentials
     */
    protected $credentials;

    /**
     * @var string
     */
    protected $algorithm;

    /**
     * @var string
     */
    protected $tokenSecret = null;

    public function __construct(CredentialsInterface $credentials)
    {
        $this->credentials = $credentials;
    }

    public function setHashingAlgorithm(string $algorithm)
    {
        $this->algorithm = $algorithm;
    }

    public function setTokenSecret(string $token)
    {
        $this->tokenSecret = $token;
    }

    /**
     * @throws UnsupportedHashAlgorithmException
     */
    public function getSignature(UriInterface $uri, array $params, string $method = 'POST'): string
    {
        return base64_encode($this->hash($this->getBaseSignatureString($uri, $params, $method)));
    }

    /**
     * Get base signature string.
     *
     * @param  null  $method
     */
    protected function getBaseSignatureString(UriInterface $uri, array $params, $method = null): string
    {
        $encodedParams = [];

        /**
         * Get Params from url.
         */
        $queries = [];
        foreach (explode('&', $uri->getQuery()) as $query) {
            if ($query) {
                $parts = explode('=', $query);
                $queries[$parts[0]] = $parts[1] ?? '';
            }
        }

        foreach (array_merge($params, $queries) as $key => $value) {
            $encodedParams[urlencode($key)] = urlencode($value);
        }

        /**
         * Add Method.
         */
        $baseStrings = [];
        if (isset($method)) {
            $baseStrings[] = strtoupper($method);
        }
        /**
         * Add base url.
         */
        if (isset($uri)) {
            $baseStrings[] = urlencode($this->getBaseSignatureUrl($uri));
        }

        if (isset($encodedParams['oauth_signature'])) {
            unset($encodedParams['oauth_signature']);
        }

        $baseStrings[] = urlencode($this->toOrderedQueryString($encodedParams));

        return implode('&', $baseStrings);
    }

    /**
     * Transform an array to a byte value ordered query string.
     */
    protected function toOrderedQueryString(array $params): string
    {
        $return = [];
        uksort($params, 'strnatcmp');
        foreach ($params as $key => $value) {
            if (is_array($value)) {
                natsort($value);
                foreach ($value as $keyduplicate) {
                    $return[] = $key.'='.$keyduplicate;
                }
            } else {
                $return[] = $key.'='.$value;
            }
        }

        return implode('&', $return);
    }

    /**
     * Generate key from consumer and token secrets.
     */
    protected function getSigningKey(): string
    {
        $parts = [$this->credentials->getConsumerSecret()];
        if ($this->tokenSecret !== null) {
            $parts[] = $this->tokenSecret;
        }
        foreach ($parts as $key => $secret) {
            $parts[$key] = urlencode($secret);
        }

        return implode('&', $parts);
    }

    /**
     * Get Base uri from url.
     */
    protected function getBaseSignatureUrl(UriInterface $uri): string
    {
        $baseUri = $uri->getScheme().'://'.$uri->getRawAuthority();

        if ($uri->getPath() === '/') {
            $baseUri .= $uri->hasExplicitTrailingHostSlash() ? '/' : '';
        } else {
            $baseUri .= $uri->getPath();
        }

        return $baseUri;
    }

    /**
     * @throws UnsupportedHashAlgorithmException
     */
    protected function hash(string $data): string
    {
        switch (strtoupper($this->algorithm)) {
            case 'HMAC-SHA1':
                return hash_hmac('sha1', $data, $this->getSigningKey(), true);
            default:
                throw new UnsupportedHashAlgorithmException(
                    'Unsupported hashing algorithm ('.$this->algorithm.') used.'
                );
        }
    }
}
