<?php

namespace App\SDKs\Magento;

use Closure;
use Illuminate\Support\Arr;
use InvalidArgumentException;
use OAuth\Common\Consumer\Credentials;
use OAuth\Common\Http\Uri\Uri;
use OAuth\Common\Storage\Memory;

class MagentoSDK
{
    private $oauth_credentials;

    private $oAuthClient;

    /**
     * MagentoSDK constructor.
     *
     *
     * @throws InvalidArgumentException
     */
    public function __construct(array $credentials, ?Closure $afterRetrievingResponse = null)
    {
        /**
         * Check required data.
         */
        if (array_diff(
            [
                'oauth_consumer_key',
                'oauth_consumer_secret',
                'store_base_url',
                'access_token',
                'secret_token',
            ],
            array_keys($credentials)
        )) {
            throw new InvalidArgumentException('oauth_consumer_key, oauth_consumer_secret, store_base_url, access_token and secret_token are required.');
        }

        /**
         * OAuth Credentials.
         */
        $this->oauth_credentials = new Credentials(
            $credentials['oauth_consumer_key'],
            $credentials['oauth_consumer_secret'],
            rtrim($credentials['store_base_url'], '/')
        );

        /**
         * Magento OAuth client with our custom parameters.
         */
        $storage = new Memory();
        $storage->storeAccessToken('OauthClient', new Token($credentials));

        $this->oAuthClient = new OauthClient(
            $this->oauth_credentials,
            new HttpClient($afterRetrievingResponse),
            $storage,
            new Signature($this->oauth_credentials)
        );
    }

    public function getCategories()
    {
        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/categories');

        return $this->oAuthClient->request($uri);
    }

    /**
     * Get Orders.
     */
    public function getOrders(array $searchCriteria = []): array
    {
        $searchCriteria = Arr::only($searchCriteria, ['currentPage', 'filterGroups', 'pageSize', 'sortOrders']);
        $params = http_build_query(['searchCriteria' => $searchCriteria]);

        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/orders?'.urldecode($params));

        return $this->oAuthClient->request($uri);
    }

    /**
     * Get Products
     */
    public function getProducts(array $searchCriteria): array
    {
        $searchCriteria = Arr::only($searchCriteria, ['currentPage', 'filterGroups', 'pageSize', 'sortOrders']);
        $params = http_build_query(['searchCriteria' => $searchCriteria]);

        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/products?'.urldecode($params));

        return $this->oAuthClient->request($uri);
    }

    /**
     * Get Shipments
     */
    public function getShipments(array $searchCriteria): array
    {
        $searchCriteria = Arr::only($searchCriteria, ['currentPage', 'filterGroups', 'pageSize', 'sortOrders']);
        $params = http_build_query(['searchCriteria' => $searchCriteria]);

        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/shipments?'.urldecode($params));

        return $this->oAuthClient->request($uri);
    }

    /**
     * Get Credit Memos
     */
    public function getCreditMemos(array $searchCriteria): array
    {
        $searchCriteria = Arr::only($searchCriteria, ['currentPage', 'filterGroups', 'pageSize', 'sortOrders']);
        $params = http_build_query(['searchCriteria' => $searchCriteria]);

        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/creditmemos?'.urldecode($params));

        return $this->oAuthClient->request($uri);
    }

    /**
     * Get Returns (RMA)
     */
    public function getReturns(array $searchCriteria): array
    {
        $searchCriteria = Arr::only($searchCriteria, ['currentPage', 'filterGroups', 'pageSize', 'sortOrders']);
        $params = http_build_query(['searchCriteria' => $searchCriteria]);

        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/returns?'.urldecode($params));

        return $this->oAuthClient->request($uri);
    }

    /**
     * Get Invoices
     */
    public function getInvoices(array $searchCriteria): array
    {
        $searchCriteria = Arr::only($searchCriteria, ['currentPage', 'filterGroups', 'pageSize', 'sortOrders']);
        $params = http_build_query(['searchCriteria' => $searchCriteria]);

        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/invoices?'.urldecode($params));

        return $this->oAuthClient->request($uri);
    }

    /**
     * Get Customer Groups
     */
    public function getCustomerGroups(array $searchCriteria): array
    {
        $searchCriteria = Arr::only($searchCriteria, ['currentPage', 'filterGroups', 'pageSize', 'sortOrders']);
        $params = http_build_query(['searchCriteria' => $searchCriteria]);

        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/customerGroups/search?'.urldecode($params));

        return $this->oAuthClient->request($uri);
    }

    /**
     * Get Product Attributes
     */
    public function getProductAttributes(array $searchCriteria): array
    {
        $searchCriteria = Arr::only($searchCriteria, ['currentPage', 'filterGroups', 'pageSize', 'sortOrders']);
        $params = http_build_query(['searchCriteria' => $searchCriteria]);

        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/products/attributes?'.urldecode($params));

        return $this->oAuthClient->request($uri);
    }

    public function getStoresViews(array $searchCriteria): array
    {
        $searchCriteria = Arr::only($searchCriteria, ['currentPage', 'filterGroups', 'pageSize', 'sortOrders']);
        $params = http_build_query(['searchCriteria' => $searchCriteria]);

        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/store/storeViews?'.urldecode($params));

        return $this->oAuthClient->request($uri);
    }

    public function getStoresGroups(array $searchCriteria): array
    {
        $searchCriteria = Arr::only($searchCriteria, ['currentPage', 'filterGroups', 'pageSize', 'sortOrders']);
        $params = http_build_query(['searchCriteria' => $searchCriteria]);

        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/store/storeGroups?'.urldecode($params));

        return $this->oAuthClient->request($uri);
    }

    public function getStoresWebsites(array $searchCriteria): array
    {
        $searchCriteria = Arr::only($searchCriteria, ['currentPage', 'filterGroups', 'pageSize', 'sortOrders']);
        $params = http_build_query(['searchCriteria' => $searchCriteria]);

        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/store/websites?'.urldecode($params));

        return $this->oAuthClient->request($uri);
    }

    /**
     * Creates new Shipment for the given Order.
     */
    public function shipOrder(int $orderId, array $shipment): int
    {
        $uri = new Uri($this->oauth_credentials->getCallbackUrl()."/rest/V1/order/{$orderId}/ship");

        return $this->oAuthClient->request($uri, 'POST', $shipment);
    }

    /**
     * Update base price of products
     */
    public function updateProductBasePrices(array $prices): array
    {
        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/products/base-prices');

        return $this->oAuthClient->request($uri, 'POST', ['prices' => $prices]);
    }

    /**
     * Get base price of products
     */
    public function getProductBasePrices(array $skus): array
    {
        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/products/base-prices-information');

        return $this->oAuthClient->request($uri, 'POST', ['skus' => $skus]);
    }

    /**
     * Add or update product prices.
     */
    public function updateProductTierPrices(array $prices): array
    {
        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/products/tier-prices');

        return $this->oAuthClient->request($uri, 'POST', ['prices' => $prices]);
    }

    /**
     * Get inventory sources
     */
    public function getInventorySources(array $searchCriteria): array
    {
        $searchCriteria = Arr::only($searchCriteria, ['currentPage', 'filterGroups', 'pageSize', 'sortOrders']);
        $params = http_build_query(['searchCriteria' => $searchCriteria]);

        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/inventory/sources?'.urldecode($params));

        return $this->oAuthClient->request($uri);
    }

    /**
     * Get inventory sources
     */
    public function getSourceItems(array $searchCriteria): array
    {
        $searchCriteria = Arr::only($searchCriteria, ['currentPage', 'filterGroups', 'pageSize', 'sortOrders']);
        $params = http_build_query(['searchCriteria' => $searchCriteria]);

        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/inventory/source-items?'.urldecode($params));

        return $this->oAuthClient->request($uri);
    }

    /**
     * Update Source Items
     */
    public function updateSourceItems(array $sourceItems): string
    {
        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/inventory/source-items');

        return $this->oAuthClient->request($uri, 'POST', ['sourceItems' => $sourceItems]);
    }

    /**
     * Bulk operations
     */
    public function bulkOperation(string $endpoint, string $method, array $data): mixed
    {
        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/async/bulk/V1/'.$endpoint);

        return $this->oAuthClient->request($uri, $method, $data);
    }

    /**
     * Update a product
     */
    public function updateProduct(string $sku, array $product): array
    {
        $uri = new Uri($this->oauth_credentials->getCallbackUrl().'/rest/V1/products/'.$sku);

        return $this->oAuthClient->request($uri, 'PUT', ['product' => $product]);
    }
}
