<?php

namespace App\Rules;

use Carbon\Carbon;
use Closure;
use Illuminate\Contracts\Validation\ValidationRule;
use Illuminate\Translation\PotentiallyTranslatedString;

class ValidISO8601DateRule implements ValidationRule
{
    /**
     * Run the validation rule.
     *
     * @param  Closure(string): PotentiallyTranslatedString  $fail
     */
    public function validate(string $attribute, mixed $value, Closure $fail): void
    {
        $format = $this->extractDateFormat($value);
        $today = today();
        $date = Carbon::parse($today)->isoFormat($format);
        if (! $this->isValidIso8601Date($date)) {
            $fail('The date format must be a valid ISO 8601 date format.');
        }
    }

    public function isValidIso8601Date($value): bool
    {
        //FROM : https://www.myintervals.com/blog/2009/05/20/iso-8601-date-validation-that-doesnt-suck/
        //        $regex = '^([\+-]?\d{4}(?!\d{2}\b))((-?)((0[1-9]|1[0-2])(\3([12]\d|0[1-9]|3[01]))?|W([0-4]\d|5[0-2])(-?[1-7])?|(00[1-9]|0[1-9]\d|[12]\d{2}|3([0-5]\d|6[1-6])))([T\s]((([01]\d|2[0-3])((:?)[0-5]\d)?|24\:?00)([\.,]\d+(?!:))?)?(\17[0-5]\d([\.,]\d+)?)?([zZ]|([\+-])([01]\d|2[0-3]):?([0-5]\d)?)?)?)?$^';
        try {
            $regex = '/^(\d{4}(-(0[1-9]|1[0-2])(-(0[1-9]|[12]\d|3[01]))?)?|\d{4}(0[1-9]|1[0-2])(0[1-9]|[12]\d|3[01])?|[0-9]{2})((T|\s)([01]\d|2[0-3]):([0-5]\d):([0-5]\d)([\.,]\d+)?([zZ]|([\+-]([01]\d|2[0-3]):?([0-5]\d)?))?)?$/';

            return preg_match($regex, $value) == 1;
        } catch (\Exception $exception) {
            return false;
        }
    }

    public function extractDateFormat($string): ?string
    {
        if (str_contains($string, 'date:')) {
            $pattern = '/{{date:([^}]+)}}/';
            if (preg_match($pattern, $string, $matches)) {
                $string = $matches[1];
            }
        } else {
            //assuming we are getting {{XXXX}} format without date: prefix
            $string = str_replace('{{', '', $string);
            $string = str_replace('}}', '', $string);
        }

        return $string;
    }
}
