<?php

namespace App;

use Illuminate\Http\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Class Response.
 */
class Response extends JsonResponse
{
    const HTTP_OK_WITH_WARNING = 299;

    const STATUS_OK_WITH_WARNING_TEXT = 'OK With Warning';

    const CODE_MATRIX_ATTRIBUTE_NO_DELETE = 'MatrixAttributeDeleteNotAllowed';

    const CODE_UNRESOLVABLE_IMAGE = 'UnresolvableImage';

    const CODE_UNAUTHORIZED = 'Unauthorized';

    const CODE_FORBIDDEN = 'Forbidden';

    const CODE_JWT_KEY_MESSING = 'JwtKeyMissing';

    const CODE_NOT_FOUND = 'NotFound';

    const CODE_METHOD_NOT_ALLOWED = 'MethodNotAllowed';

    const CODE_ALREADY_ARCHIVED = 'IsAlreadyArchived';

    const CODE_HAS_STOCK = 'HasStock';

    const CODE_ALREADY_UNARCHIVED = 'IsAlreadyUnarchived';

    const CODE_RESOURCE_LINKED = 'IsLinked';

    const CODE_UNEXPECTED = 'Unexpected';

    const CODE_EMPTY = 'IsEmpty';

    const CODE_NOT_LEAF = 'IsNonLeaf';

    const CODE_UNACCEPTABLE = 'IsUnacceptable';

    const CODE_INVALID_ROW = 'InvalidRow';

    const CODE_INVALID_Template = 'InvalidTemplate';

    const CODE_FILE_NOT_FOUND = 'FileNotFound';

    const CODE_CUSTOMER_EXISTS = 'CustomerExists';

    const CODE_SO_EXISTS = 'SalesOrderExists';

    const CODE_IS_NOT_DRAFT = 'IsNotDraft';

    const CODE_IS_LOCKED = 'IsLocked';

    const CODE_ALREADY_APPROVED = 'IsAlreadyApproved';

    const CODE_RESERVE_NOT_FOUND = 'DoesNotHaveReservationMovements';

    const CODE_DROPSHIP_PO_NOT_FOUND = 'DoesNotHaveDropshipPurchaseOrder';

    const CODE_DROPSHIP_PO_DRAFT = 'DraftDropshipPurchaseOrder';

    const CODE_IGNORE_DETAILS = 'IgnoreDetails';

    const CODE_IS_ALREADY_OPEN = 'IsAlreadyOpen';

    const CODE_IS_NOT_OPEN = 'IsNotOpen';

    const CODE_SUPPLIER_INVENTORY_RESET = 'SupplierInventoryReset';

    const CODE_NOT_FULLY_FULFILLED = 'NotFullyFulfilled';

    const CODE_NO_ENOUGH_QUANTITY = 'NoEnoughQuantity';

    const CODE_SAME_QUANTITY = 'SameQuantity';

    const CODE_UPDATE_MANUALLY = 'UpdateManually';

    const CODE_NOT_APPROVED = 'IsNotApproved';

    const CODE_CLOSED = 'IsClosed';

    const CODE_EMPTY_PO = 'EmptyPurchaseOrders';

    const CODE_NOT_INTEGRATED = 'NotIntegrated';

    const CODE_INTEGRATION_API_ERROR = 'IntegrationApiError';

    const CODE_INTERNAL_VALIDATION = 'InternalValidation'; // code for detect the error is a validation error

    const CODE_HAS_UNMAPPED_PRODUCTS = 'HasUnmappedProducts';

    const CODE_ALREADY_MAPPED = 'IsAlreadyMapped';

    const CODE_ARE_NOT_PRINTED = 'AreNotPrinted';

    const CODE_HAS_FULFILLMENTS = 'HasFulfillments';

    const CODE_UNFULFILLABLE_TO_SHIPPING_PROVIDER = 'Unfulfillable';

    const CODE_ACTION_UNAVAILABLE_TEMPORARY = 'ActionUnavailableTemporary';

    const CODE_UNAVAILABLE = 'IsUnavailable';

    const CODE_OPEN_STOCK_TAKE = 'OpenStockTake';

    const CODE_NEGATIVE_SNAPSHOT_INVENTORY = 'NegativeSnapshotInventory';

    const CODE_INSUFFICIENT_STOCK = 'InsufficientStock';

    const CODE_DUPLICATE_ENTRY = 'DuplicateEntry';

    /**
     * @var Response
     */
    private static $instance = null;

    /**
     * Get/Create a new Instance.
     */
    public static function make(?int $statusCode = null, bool $newInstance = false): self
    {
        if (! static::$instance || $newInstance) {
            static::$instance = new self(null, $statusCode ?: self::HTTP_OK);
        }

        // set status code if sent after first creation
        if ($statusCode) {
            static::$instance->setStatusCode($statusCode);
        }

        return static::$instance;
    }

    /**
     * Get Instance of response.
     */
    public static function instance(?int $statusCode = null): self
    {
        return self::make($statusCode);
    }

    public function addData($data, ?string $key = null): self
    {
        if (! isset($this->original['data'])) {
            $this->original['data'] = [];
        }

        if ($key !== null) {
            $this->original['data'][$key] = $data;
        } else {
            $this->original['data'] = $data;
        }

        return $this;
    }

    public function getOriginalData(): mixed
    {
        return $this->original['data'];
    }

    public function addAdditional(string $key, $data)
    {
        $this->original[$key] = $data;

        return $this;
    }

    public function setMessage(string $message): self
    {
        $this->original['message'] = $message;

        return $this;
    }

    /**
     * Set Errors to response.
     */
    public function setErrors(array $errors): self
    {
        $this->original['errors'] = $errors;

        return $this;
    }

    /**
     * Add error to response errors.
     *
     * @param  array|Validator  $data
     */
    public function addError(string $message, string $code, string $key, $data = [], bool $prepare = false): self
    {
        if ($code === self::CODE_INTERNAL_VALIDATION) {
            $this->setMessage($message);
            $this->addErrorsFromValidator($data);
        } else {
            if (! isset($this->original['errors'])) {
                $this->original['errors'] = [];
            }

            if (! isset($this->original['errors'][$key])) {
                $this->original['errors'][$key] = [];
            }

            $this->original['errors'][$key][] = ['message' => $message, 'code' => $code, 'data' => $data];
        }

        // for unknown reason sometimes response does not call "prepare" function !!
        if ($prepare) {
            $this->prepare(request());
        }

        return $this;
    }

    public static function getError(string $message, string $code, string $key, $data = [])
    {
        if ($code === self::CODE_INTERNAL_VALIDATION) {
            $validationErrors = is_array($data) ? $data : $data->errors()->toArray();
            $e = [];
            foreach ($validationErrors as $attribute => $errors) {
                foreach ($errors as $error) {
                    $e[$attribute][] = ['message' => $error['message'], 'code' => $error['code'], 'data' => $error['data'] ?? []];
                }
            }

            return $e;
        }

        return [$key => [['message' => $message, 'code' => $code, 'data' => $data]]];
    }

    /**
     * Set warnings to response.
     */
    public function setWarnings(array $warnings): self
    {
        $this->original['warnings'] = $warnings;

        return $this;
    }

    /**
     * Add warning to response warnings.
     */
    public function addWarning(string $message, string $code, string $key, array $data = []): self
    {
        if (! isset($this->original['warnings'])) {
            $this->original['warnings'] = [];
        }

        if (! isset($this->original['warnings'][$key])) {
            $this->original['warnings'][$key] = [];
        }

        $this->original['warnings'][$key][] = ['message' => $message, 'code' => $code, 'data' => $data];

        return $this;
    }

    /**
     * @param  \Illuminate\Contracts\Validation\Validator|array  $validator
     */
    public function addWarningsFromValidator($validator, ?string $prefix = null): self
    {
        $errors = is_array($validator) ? $validator : $validator->errors()->toArray();
        foreach ($errors as $key => $attributeErrors) {
            foreach ($attributeErrors as $error) {
                $this->addWarning($error['message'], $error['code'], $prefix ? "$prefix.$key" : $key, $error['data'] ?? []);
            }
        }

        return $this;
    }

    /**
     * @param  \Illuminate\Contracts\Validation\Validator|array  $validator
     */
    public function addErrorsFromValidator($validator, ?string $prefix = null): self
    {
        $validationErrors = is_array($validator) ? $validator : $validator->errors()->toArray();
        foreach ($validationErrors as $key => $errors) {
            foreach ($errors as $error) {
                $this->addError($error['message'], $error['code'], $prefix ? "$prefix.$key" : $key, $error['data'] ?? []);
            }
        }

        return $this;
    }

    /**
     * Set as success response.
     */
    public function success(int $statusCode = self::HTTP_OK, string $message = ''): self
    {
        $this->setStatusCode($statusCode);

        if (! empty($message)) {
            $this->setMessage($message);
        }

        return $this;
    }

    /**
     * Set as warning response.
     */
    public function warning(int $statusCode = self::HTTP_OK_WITH_WARNING, array $warnings = []): self
    {
        if (! empty($warnings)) {
            $this->setWarnings($warnings);
        }

        return $this->setStatusCode($statusCode, self::STATUS_OK_WITH_WARNING_TEXT);
    }

    public function error(int $statusCode = self::HTTP_INTERNAL_SERVER_ERROR, array $errors = []): self
    {
        if (! empty($errors)) {
            $this->setErrors($errors);
        }

        return $this->setStatusCode($statusCode);
    }

    /**
     * Prepares the Response before it is sent to the client.
     */
    public function prepare(Request $request): static
    {
        $this->setStatus();

        $this->setData($this->original);

        // return status code 200 #SKU-847
        if ($this->statusCode < 300) {
            $this->setStatusCode(self::HTTP_OK);
        }

        return parent::prepare($request);
    }

    /**
     * Set status to response depends on data.
     */
    public function setStatus(string $status = null): self
    {
        if (!is_null($status)) {
            return $this->setStatusCode($status);
        }

        if (! empty($this->original['errors'])) {
            $this->original['status'] = __('messages.status_failure');

            if (empty($this->original['message'])) {
                $this->original['message'] = $this->original['errors'][array_key_first($this->original['errors'])][0]['message'];
            }
        } elseif (! empty($this->original['warnings'])) {
            $this->setStatusCode(self::HTTP_OK_WITH_WARNING, self::STATUS_OK_WITH_WARNING_TEXT);

            $this->original['status'] = __('messages.status_warning');

            if (empty($this->original['message'])) {
                $this->original['message'] = $this->original['warnings'][array_key_first($this->original['warnings'])][0]['message'];
            }
        } else {
            $this->original['status'] = __('messages.status_success');
        }
        return $this;
    }

    /**
     * @param  null  $key
     * @param  null  $default
     * @return mixed
     */
    public function getOriginal($key = null, $default = null)
    {
        if ($key) {
            return $this->original['data'][$key] ?? $default;
        }

        return $this->original;
    }
}
