<?php

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Data\UpdateWarehouseTransferLinesData;
use App\DTO\WarehouseTransferDto;
use App\DTO\WarehouseTransferLineDto;
use App\Managers\WarehouseTransferManager;
use App\Models\Setting;
use App\Models\ShippingMethod;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferLine;
use App\Models\WarehouseTransferShipment;
use App\Models\WarehouseTransferShipmentLine;
use App\Models\WarehouseTransferShipmentReceipt;
use App\Models\WarehouseTransferShipmentReceiptLine;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Spatie\LaravelData\Attributes\DataCollectionOf;
use Spatie\LaravelData\DataCollection;
use Throwable;

class WarehouseTransferRepository extends AbstractRepository
{
    /**
     * @throws Throwable
     */
    public function saveWithRelations(#[DataCollectionOf(WarehouseTransferDto::class)] DataCollection $data): Collection|WarehouseTransfer
    {
        $data = $data->toCollection();

        return DB::transaction(function () use ($data) {
            $warehouseTransferCollection = $data->map(function (WarehouseTransferDto $warehouseTransferDto) {
                return WarehouseTransferDto::from(new WarehouseTransfer($warehouseTransferDto->toArray()));
            });
            $warehouseTransferCollection = $this->save($warehouseTransferCollection, WarehouseTransfer::class);
            $this->saveLines($warehouseTransferCollection, $data);

            if ($warehouseTransferCollection->count() === 1) {
                return WarehouseTransfer::find($warehouseTransferCollection->first()['id']);
            }

            return $warehouseTransferCollection;
        });
    }

    private function saveLines(Collection $warehouseTransferCollection, $data): void
    {
        if ($warehouseTransferCollection->isEmpty()) {
            collect();
            return;
        }

        $warehouseTransferLineCollection = collect();

        $warehouseTransferCollection->each(function ($warehouseTransfer) use ($data, $warehouseTransferLineCollection) {
            $warehouseTransferDto = $data->where('warehouse_transfer_number', $warehouseTransfer['warehouse_transfer_number'])->first();
            $warehouseTransferDto->warehouseTransferLines->each(function (WarehouseTransferLineDto $warehouseTransferLineDto) use (
                $warehouseTransferDto,
                $warehouseTransfer,
                $warehouseTransferLineCollection
            ) {
                $warehouseTransferLineDto->warehouse_transfer_id = $warehouseTransfer['id'];
                $warehouseTransferLineCollection->add(WarehouseTransferLineDto::from(new WarehouseTransferLine($warehouseTransferLineDto->toArray())));
            });
        });

        $this->save($warehouseTransferLineCollection, WarehouseTransferLine::class);
    }

    public function getFromWarehouseTransferNumber(string $warehouseTransferNumber): ?WarehouseTransfer
    {
        /** @var WarehouseTransfer $warehouseTransfer */
        $warehouseTransfer = WarehouseTransfer::query()
            ->where('warehouse_transfer_number', $warehouseTransferNumber)
            ->first();

        return $warehouseTransfer;
    }

    public function getWarehouseTransferShipmentReceiptForDate(WarehouseTransferShipment $warehouseTransferShipment, Carbon $received_at): ?WarehouseTransferShipmentReceipt
    {
        /** @var WarehouseTransferShipmentReceipt $warehouseTransferShipmentReceipt */
        $warehouseTransferShipmentReceipt = WarehouseTransferShipmentReceipt::query()
            ->where('warehouse_transfer_shipment_id', $warehouseTransferShipment->id)
            ->whereDate('received_at', $received_at)
            ->first();

        return $warehouseTransferShipmentReceipt;
    }

    public function getSimilarWarehouseTransferShipment(WarehouseTransfer $warehouseTransfer, Carbon $shipped_at, ShippingMethod $shippingMethod, string $trackingNumber): ?WarehouseTransferShipment
    {
        /** @var WarehouseTransferShipment $warehouseTransferShipment */
        $warehouseTransferShipment = WarehouseTransferShipment::query()
            ->where('warehouse_transfer_id', $warehouseTransfer->id)
            ->whereDate('shipped_at', $shipped_at)
            ->where('tracking_number', $trackingNumber)
            ->first();

        return $warehouseTransferShipment;
    }

    public function createTransfer(array $data): WarehouseTransfer
    {
        $data['transfer_status'] = WarehouseTransfer::TRANSFER_STATUS_DRAFT;
        $data['shipment_status'] = WarehouseTransfer::TRANSFER_SHIPMENT_STATUS_UNSHIPPED;
        $data['receipt_status'] = WarehouseTransfer::TRANSFER_RECEIPT_STATUS_UNRECEIVED;

        // Default to shipping method in settings if doesn't have shipping method
        if (! isset($data['shipping_method_id'])) {
            $setting = Setting::with([])->where('key', Setting::KEY_WH_TRANSFER_SHIPPING_METHOD)->first();
            if ($setting) {
                $data['shipping_method_id'] = $setting->value;
            }
        }

        // Set transfer date to current date if not set
        if (empty($data['transfer_date'])) {
            $data['transfer_date'] = Carbon::now()->format('Y-m-d H:i:s');
        } else {
            $data['transfer_date'] = Carbon::parse($data['transfer_date']);
        }
        $transfer = new WarehouseTransfer($data);
        $transfer->save();

        return $transfer;
    }

    public function createWarehouseTransferLine(
        WarehouseTransfer $transfer,
        $productId,
        $quantity
    ): WarehouseTransferLine {
        $transferLine = new WarehouseTransferLine([
            'warehouse_transfer_id' => $transfer->id,
            'product_id' => $productId,
            'quantity' => $quantity,
        ]);
        $transfer->warehouseTransferLines()->save($transferLine);

        return $transferLine;
    }

    public function createTransferShipment(WarehouseTransfer $transfer, array $data): WarehouseTransferShipment
    {
        // If no shipping method is specified, we use that of the transfer
        $shipment = new WarehouseTransferShipment(array_merge($data, [
            'warehouse_transfer_id' => $transfer->id,
            'shipped_at' => $data['shipment_date'],
            'shipping_method_id' => $data['shipping_method_id'] ?? $transfer->shipping_method_id,
        ]));

        $shipment->save();

        return $shipment;
    }

    public function createTransferShipmentLine(WarehouseTransferShipment $shipment, $transferLineId, $quantity): WarehouseTransferShipmentLine
    {
        $shipmentLine = new WarehouseTransferShipmentLine([
            'warehouse_transfer_line_id' => $transferLineId,
            'quantity' => $quantity,
        ]);
        $shipment->shipmentLines()->save($shipmentLine);

        return $shipmentLine;
    }

    /**
     * @return $this|Model
     */
    public function createTransferShipmentReceipt($shipmentId, $receiptDate)
    {
        return WarehouseTransferShipmentReceipt::with([])->create([
            'warehouse_transfer_shipment_id' => $shipmentId,
            'user_id' => auth()->user()?->id,
            'received_at' => $receiptDate,
        ]);
    }

    public function createWarehouseTransferReceiptLine($shipmentReceiptId, $shipmentLineId, $quantityReceived): WarehouseTransferShipmentReceiptLine
    {
        $receiptLine = new WarehouseTransferShipmentReceiptLine();
        $receiptLine->warehouse_transfer_shipment_receipt_id = $shipmentReceiptId;
        $receiptLine->warehouse_transfer_shipment_line_id = $shipmentLineId;
        $receiptLine->quantity = $quantityReceived;
        $receiptLine->save();

        return $receiptLine;
    }

    /**
     * @throws Throwable
     */
    public function updateWarehouseTransferLines(WarehouseTransfer $warehouseTransfer, UpdateWarehouseTransferLinesData $data): WarehouseTransfer
    {
        $lines = WarehouseTransferLine::where('warehouse_transfer_id', $warehouseTransfer->id)->whereIn('id', $data->ids)->get();

        DB::transaction(function() use($warehouseTransfer, $lines, $data) {
            foreach ($lines as $line) {
                if (isset($data->quantity_shipped_before_start_date)) {
                    $line->quantity_shipped_before_start_date =
                        $data->quantity_shipped_before_start_date > $line->quantity ? $line->quantity : $data->quantity_shipped_before_start_date;
                }

                $line->save();
            }

            $warehouseTransfer->refresh();
            $warehouseTransfer->load('warehouseTransferLines');
            app(WarehouseTransferManager::class)->setTransferReceiptStatus($warehouseTransfer);
        });

        return $warehouseTransfer;
    }
}
