<?php

namespace App\Repositories;

use App\Models\Product;
use App\Models\SupplierProductPricing;
use Illuminate\Support\Facades\DB;

class SupplierProductPricingRepository
{
    /*
     * Get the most sensible supplier product pricing for a product.  The logic is based on the idea that:
     * - Price should be non zero
     * - Prefer the default supplier product
     * - Then prefer the default site currency
     * - Then prefer the default site pricing tier
     *
     * TODO: Add functionality to prefer the default supplier pricing tier (above the site default currency preference)
     */
    public function getSensibleSupplierProductPrice(Product $product): ?float
    {
        return SupplierProductPricing::query()
            ->select([
                'supplier_products.is_default as supplier_product_default',
                'supplier_pricing_tiers.is_default as site_default_pricing_tier',
                'supplier_pricing_tiers.is_default as site_default_currency',
                DB::raw('ROUND(supplier_product_pricing.price * currencies.conversion, 4) as price'),
            ])
            ->join('supplier_products', 'supplier_products.id', 'supplier_product_pricing.supplier_product_id')
            ->join('supplier_pricing_tiers', 'supplier_pricing_tiers.id', 'supplier_product_pricing.supplier_pricing_tier_id')
            ->join('currencies', 'currencies.code', 'supplier_pricing_tiers.currency_code')
            ->where('supplier_products.product_id', $product->id)
            ->where('price', '!=', 0)
            ->orderBy('supplier_product_default', 'desc')
            ->orderBy('site_default_currency', 'desc')
            ->orderBy('site_default_pricing_tier', 'desc')
            ->first()?->price;
    }
}
