<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 7/22/20
 * Time: 9:18 AM.
 */

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Models\Supplier;
use App\Models\SupplierInventory;
use App\Models\SupplierProduct;
use App\Models\Warehouse;
use Exception;
use Illuminate\Database\Eloquent\Builder;

/**
 * Class SupplierInventoryRepository.
 */
class SupplierInventoryRepository extends AbstractRepository
{
    public function find($inventoryId): SupplierInventory
    {
        return SupplierInventory::with([])->findOrFail($inventoryId);
    }

    public function findBy(array $payload): SupplierInventory
    {
        $query = SupplierInventory::with([]);
        foreach ($payload as $key => $value) {
            if ($key === 'supplier_sku') {
                // Bind in supplier sku
                $query = $query->whereHas('supplier', function (Builder $builder) use ($value) {
                    $builder->whereHas('supplierProducts', function (Builder $builder) use ($value) {
                        $builder->where('supplier_sku', $value);
                    });
                });
            } else {
                $query->where($key, $value);
            }
        }

        return $query->first();
    }

    public function createInventory(array $data): SupplierInventory
    {
        if (isset($data['quantity'])) {
            $data['in_stock'] = null;
        }

        return SupplierInventory::with([])->create($data);
    }

    public function updateInventory($inventoryId, array $data): SupplierInventory
    {
        $inventory = $this->find($inventoryId);
        if (isset($data['quantity'])) {
            $data['in_stock'] = null;
        } else {
            $data['quantity'] = null;
        }

        $inventory->fill($data);
        $inventory->save();

        return $inventory;
    }

    public function initializeInventoryForSupplierWarehouse(Warehouse $warehouse)
    {
        $supplier = $warehouse->supplier;
        if (! $supplier) {
            return;
        }

        $supplier->supplierProducts->each(function (SupplierProduct $supplierProduct) use ($supplier, $warehouse) {
            $this->initializeInventory($supplier, $supplierProduct->product_id, $warehouse->id);
        });
    }

    public function initializeInventoryForSupplierProduct(SupplierProduct $supplierProduct)
    {
        // Set the inventory for all warehouses for the supplier
        $supplierProduct->supplier->warehouses()->each(function (Warehouse $warehouse) use ($supplierProduct) {
            if (! $this->inventoryExists($supplierProduct, $warehouse->id)) {
                $this->initializeInventory($supplierProduct->supplier, $supplierProduct->product_id, $warehouse->id);
            }
        });
    }

    public function inventoryExists(SupplierProduct $supplierProduct, $warehouseId): bool
    {
        return SupplierInventory::with([])->where('supplier_id', $supplierProduct->supplier_id)
            ->where('product_id', $supplierProduct->product_id)
            ->where('warehouse_id', $warehouseId)
            ->count() > 0;
    }

    /**
     * @return bool|mixed|null
     *
     * @throws Exception
     */
    public function clearInventoryForSupplierProduct($supplierId, $productId)
    {
        return SupplierInventory::with([])->where('supplier_id', $supplierId)->where('product_id', $productId)->delete();
    }

    /**
     * @param  Supplier|int  $supplierId
     * @return SupplierInventory
     *
     * Note: when updating this function, please reflect your changes to CreateSkuProducts jobs
     *
     * @see \App\Jobs\Magento\CreateSkuProducts::addDefaultSupplierToProducts
     * @see \App\Jobs\Shopify\ShopifyCreateSkuProducts::addDefaultSupplierToProducts
     */
    private function initializeInventory($supplierId, $productId, $warehouseId): SupplierInventory
    {
        $supplier = $supplierId instanceof Supplier ? $supplierId : Supplier::query()->findOrFail($supplierId);

        return $this->createInventory([
            'supplier_id' => $supplier->id,
            'product_id' => $productId,
            'warehouse_id' => $warehouseId,
            'quantity' => $supplier->default_stock_level == Supplier::STOCK_LEVEL_ZERO ? 0 : null,
            'in_stock' => $supplier->default_stock_level != Supplier::STOCK_LEVEL_NOT_IN_STOCK,
            'source' => null,
        ]);
    }
}
