<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 11/26/20
 * Time: 2:51 PM.
 */

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Data\CreateInitialStockTakeData;
use App\Models\FifoLayer;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use App\Models\Warehouse;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Enumerable;
use Spatie\LaravelData\Attributes\DataCollectionOf;
use Spatie\LaravelData\DataCollection;

/**
 * Class StockTakeRepository.
 */
class StockTakeRepository
{
    public function create(array $data): StockTake
    {
        return StockTake::with([])->create($data);
    }

    public function findOrFail($id, $with = ['stockTakeItems', 'warehouse']): StockTake
    {
        return StockTake::with($with)->findOrFail($id);
    }

    public static function findOpenStockTake($warehouseId, $productId): ?StockTake
    {
        return static::openStockTakes($warehouseId, $productId)->first();
    }

    public static function takingStock($warehouseId, $productId): bool
    {
        return static::openStockTakes($warehouseId, $productId)->count() > 0;
    }

    private static function openStockTakes($warehouseId, $productId)
    {
        return StockTake::with(['stockTakeItems'])
            ->where('warehouse_id', $warehouseId)
            ->where('status', StockTake::STOCK_TAKE_STATUS_OPEN)
            ->whereHas('stockTakeItems', function (Builder $builder) use ($productId) {
                return $builder->where('product_id', $productId);
            });
    }

    public function deleteForProductIds(StockTake $stockTake, Enumerable $productIds): void
    {
        $stockTake->stockTakeItems()
            ->whereIn('product_id', $productIds)
            ->each(function (StockTakeItem $stockTakeItem) {
                $stockTakeItem->delete();
            });
    }

    public function recalculateValueChange(StockTake $stockTake): StockTake
    {
        $stockTake->value_change = $stockTake->stockTakeItems->sum(function ($item) {
            return $item->quantity_adjusted * $item->unit_cost;
        });
        $stockTake->save();
        return $stockTake;
    }

    public function getInitialStockTakeForWarehouseId(int $warehouse_id): ?StockTake
    {
        return StockTake::where('warehouse_id', $warehouse_id)
            ->where('is_initial_count', true)
            ->first();
    }

    public function getInitialInventoryForWarehouse(Warehouse $warehouse): Collection
    {
        return StockTakeItem::with('product')
            ->whereHas('stockTake', function (Builder $query) use ($warehouse) {
                $query->where('warehouse_id', $warehouse->id);
                $query->where('is_initial_count', true);
            })
            ->get();
    }

    public function getFifoLayerForInitialInventoryForWarehouseProduct(int $warehouse_id, int $product_id): ?FifoLayer
    {
        return FifoLayer::where('warehouse_id', $warehouse_id)
            ->where('product_id', $product_id)
            ->where('link_type', StockTakeItem::class)
            ->whereHasMorph('link', StockTakeItem::class, function (Builder $query) {
                $query->whereHas('stockTake', function (Builder $query) {
                    $query->where('is_initial_count', true);
                });
            })
            ->first();
    }
}
