<?php

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelIntegrationInstance;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelOrderLine;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Warehouse;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class SalesOrderLineRepository extends AbstractRepository
{
    public function withoutDailyFinancialsQuery(string $userTimezone): Builder
    {
        return SalesOrderLine::with('salesOrder')
            ->select('sales_order_lines.product_id', 'sales_orders.order_date')
            ->join('sales_orders', 'sales_order_lines.sales_order_id', '=', 'sales_orders.id')
            ->leftJoin('reporting_daily_financials', function ($join) use ($userTimezone) {
                $join->on(DB::raw("CONVERT_TZ(DATE(CONVERT_TZ(sales_orders.order_date, 'UTC', '$userTimezone')), '$userTimezone', 'UTC')"), '=', 'reporting_daily_financials.date');
                $join->on('reporting_daily_financials.reportable_id', '=', 'sales_order_lines.product_id');
                $join->where('reporting_daily_financials.reportable_type', '=', Product::class);
            })
            ->where('sales_orders.order_status', '!=', 'draft')
            ->whereNotNull('sales_order_lines.product_id')
            ->whereNull('reporting_daily_financials.id');
    }

    public function calculateFulfilledQuantity(SalesOrderLine $salesOrderLine): int
    {
        return $salesOrderLine->salesOrderFulfillmentLines->sum('quantity');
    }

    public function calculateUnfulfilledQuantity(SalesOrderLine $salesOrderLine): int
    {
        return $salesOrderLine->quantity - $this->calculateFulfilledQuantity($salesOrderLine);
    }

    public function getMappedLinesFromIds(array $saleOrderLineIds): Collection
    {
        $salesOrderLineCollection = collect();
        foreach (array_chunk($saleOrderLineIds, 10000) as $salesOrderLineIdsChunk) {
            $salesOrderLineCollection = $salesOrderLineCollection->merge(
                SalesOrderLine::with('salesOrder')
                    ->whereNotNull('product_id')
                    ->whereIn('id', $salesOrderLineIdsChunk)
                    ->get()
            );
        }

        return $salesOrderLineCollection;
    }

    public function getUnmappedSalesOrderLinesForSalesChannel(AbstractSalesChannelIntegrationInstance|IntegrationInstanceInterface $integrationInstance, ?Collection $productListingCollection = null): EloquentCollection
    {
        $salesChannelOrderClass = app($integrationInstance::getOrderClass());
        /** @var AbstractSalesChannelOrderLine $salesChannelOrderLineClass */
        $salesChannelOrderLineClass = app($integrationInstance::getOrderLineClass());
        $salesChannelProduct = app($integrationInstance::getProductClass());

        $query = SalesOrderLine::select(
            'sales_order_lines.id as id',
            'pl.product_id as product_id',
            'pl.id as product_listing_id'
        )
            ->joinRelationshipUsingAlias('salesOrder', 'so')
            ->join($salesChannelOrderLineClass->getTable().' as scl', 'sales_order_lines.sales_channel_line_id', '=', 'scl.'.$salesChannelOrderLineClass->getTableUniqueId())
            ->join($salesChannelProduct->getTable().' as scp', function ($join) use ($salesChannelProduct, $salesChannelOrderLineClass) {
                $join->on('scl.'.$salesChannelOrderLineClass->getSalesChannelProductUniqueId(), 'scp.'.$salesChannelProduct->getUniqueField());
            })
            ->join((new ProductListing())->getTable().' as pl', function ($join) use ($salesChannelProduct) {
                $join->on('pl.document_id', 'scp.id')->where('pl.document_type', get_class($salesChannelProduct));
            })
            ->where('so.sales_channel_order_type', get_class($salesChannelOrderClass))
            ->whereNull('sales_order_lines.product_listing_id');

        if ($productListingCollection)
        {
            $query->whereIn('pl.id', $productListingCollection->pluck('id')->values());
        }

        return $query->get();
    }

    public function sanitizeLinesForAllocation(Collection $salesOrderLineCollection): Collection
    {
        // Conditions for reserving inventory
        return $salesOrderLineCollection->reject(function (SalesOrderLine $salesOrderLine) {
            $salesOrder = $salesOrderLine->salesOrder;
            return $salesOrder->canceled_at ||
                !in_array($salesOrder->order_status, [SalesOrder::STATUS_OPEN, SalesOrder::STATUS_RESERVED]) ||
                $salesOrderLine->warehouse?->type == Warehouse::TYPE_AMAZON_FBA ||
                $salesOrderLine->is_dropship;
        });
    }
}
