<?php

namespace App\Repositories;

use App\Abstractions\Integrations\SalesChannels\SalesChannelFulfillmentLinkInterface;
use App\Contracts\Repositories\SalesOrderRepositoryInterface;
use App\Data\SalesOrderFulfillmentData;
use App\Models\FifoLayer;
use App\Models\IntegrationInstance;
use App\Models\InventoryMovement;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Repositories\SalesOrder\SalesOrderRepository;
use App\Services\SalesOrder\FulfillSalesOrderService;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Spatie\LaravelData\Optional;
use Throwable;

class SalesOrderFulfillmentRepository extends SalesOrderRepository implements SalesOrderRepositoryInterface
{
    /**
     * @throws Throwable
     */
    public function createSalesOrderFulfillments(SalesOrder $salesOrder, array $fulfillmentsData): SalesOrder
    {
        // TODO: Need a check to make sure the sales order fulfillment is fully fulfillable
        $salesOrderFulfillments = [];
        foreach ($fulfillmentsData as $fulfillmentData) {
            $fulfillmentData['fulfillment_lines'] = array_map(function ($fulfillmentLine) {
                $fulfillmentLine['sales_order_line_id'] =
                    SalesOrderLine::query()
                        ->where('sales_channel_line_id', $fulfillmentLine['sales_channel_line_id'])
                        ->firstOrFail()
                    ->id;
                unset($fulfillmentLine['sales_channel_line_id']);

                return $fulfillmentLine;
            }, $fulfillmentData['fulfillment_lines']);

            $salesOrderFulfillment = FulfillSalesOrderService::make($salesOrder)->fulfillWithInputs($fulfillmentData);

            // failed to create the fulfillment
            if (is_array($salesOrderFulfillment)) {
                dd($salesOrderFulfillment);

                continue;
            }

            /*
             * We do this to avoid resending fulfillments to Shopify when running Submit Tracking Info command
             */
            $salesOrderFulfillment->submitted_to_sales_channel_at = now();
            $salesOrderFulfillment->save();

            /*
             * TODO:
             * This is the link between the sales channel fulfillment (i.e. shopify) and the fulfillment.  Some sort of
             * link is needed to be able to create the shopify order links for example.  In the future this could be
             * refactored to be a polymorphic relationship between sales orer fulfillments and sales channel fuflillments
             * but for now it is not.
             */
            $salesOrderFulfillment['sales_channel_fulfillment_id'] = $fulfillmentData['sales_channel_fulfillment_id'];
            //$this->markFulfillmentAsProcessed($fulfillment['shopify_fulfillment_id'], $salesOrderFulfillment->id);
            $salesOrderFulfillments[] = [
                'sales_order_fulfillment_id' => $salesOrderFulfillment->id,
                'sales_channel_fulfillment_id' => $fulfillmentData['sales_channel_fulfillment_id'],
            ];
        }
        $salesOrder['fulfillment_maps'] = $salesOrderFulfillments;

        return $salesOrder;
    }

    public function getSalesOrderFulfillmentForDate(SalesOrder $salesOrder, Carbon $date): ?SalesOrderFulfillment
    {
        /** @var SalesOrderFulfillment $salesOrderFulfillment */
        $salesOrderFulfillment = SalesOrderFulfillment::query()
            ->where('sales_order_id', $salesOrder->id)
            ->whereDate('created_at', $date)
            ->first();

        return $salesOrderFulfillment;
    }

    public function markAsSubmittedToSalesChannel(SalesOrderFulfillment $salesOrderFulfillment, ?SalesChannelFulfillmentLinkInterface $salesChannelLink = null): void
    {
        $salesOrderFulfillment->submitted_to_sales_channel_at = now();
        if (! is_null($salesChannelLink)) {
            $salesOrderFulfillment->sales_channel_link_type = get_class($salesChannelLink);
            $salesOrderFulfillment->sales_channel_link_id = $salesChannelLink->id;
        }
        $salesOrderFulfillment->save();
    }

    public function markAsNotSubmittedToSalesChannel(SalesOrderFulfillment $salesOrderFulfillment): void
    {
        $salesOrderFulfillment->sales_channel_link_type = null;
        $salesOrderFulfillment->sales_channel_link_id = null;
        $salesOrderFulfillment->submitted_to_sales_channel_at = null;
        $salesOrderFulfillment->save();
    }

    public function getPendingSalesOrderFulfillments(IntegrationInstance $integrationInstance): EloquentCollection
    {
        return SalesOrderFulfillment::whereHas('salesOrder', function ($query) use ($integrationInstance) {
            $query->whereHas('salesChannel', function ($query) use ($integrationInstance) {
                $query->where('integration_instance_id', $integrationInstance->id);
            });
            $query->whereIsFba(false);
        })
            ->with([
                'salesOrder.salesChannelOrder',
                'salesOrderFulfillmentLines',
            ])
            ->whereNull('submitted_to_sales_channel_at')
            ->whereNotNull('tracking_number')
            ->get();
    }

    /**
     * @throws Throwable
     */
    public function createMany(Collection $data): Collection
    {
        return DB::transaction(function() use ($data){
            $fulfillments = $this->save(
                data: $data->map(function(SalesOrderFulfillmentData $fulfillmentData){
                    $current = clone $fulfillmentData;
                    $current->fulfillment_lines = Optional::create();
                    return $current;
                }),
                modelClass: SalesOrderFulfillment::class
            );
            // Then, we create the fulfillment lines
            $fulfillmentLines = [];

            $data->each(function(SalesOrderFulfillmentData $fulfillmentData) use ($fulfillments, &$fulfillmentLines){
                foreach ($fulfillmentData->fulfillment_lines as $fulfillmentLine){
                    $fulfillmentLine['sales_order_fulfillment_id'] = $fulfillments
                        ->where('sales_order_id', $fulfillmentData->sales_order_id)
                        ->where('fulfillment_sequence', $fulfillmentData->fulfillment_sequence)
                        ->firstOrFail()['id'];
                    $fulfillmentLine['quantity'] = $fulfillmentLine['fulfillable_quantity'];
                    $fulfillmentLine['created_at'] = now();
                    $fulfillmentLine['updated_at'] = now();
                    $fulfillmentLines[] = collect($fulfillmentLine)
                        ->only([
                            'sales_order_fulfillment_id',
                            'sales_order_line_id',
                            'quantity',
                            'created_at',
                            'updated_at'
                        ])->toArray();
                }
            });

            SalesOrderFulfillmentLine::query()->insert($fulfillmentLines);

            return SalesOrderFulfillment::with(['salesOrderFulfillmentLines', 'warehouse'])
                ->whereIn('id', $fulfillments->pluck('id'))
                ->get();
        });
    }

    /**
     * @param  array  $orderLineIds
     * @return EloquentCollection
     */
    public function getReservationMovementsForSalesOrderLines(array $orderLineIds): EloquentCollection
    {
        return InventoryMovement::query()
            ->selectRaw('link_id sales_order_line_id, coalesce(sum(quantity), 0) as quantity')
            ->whereIn('link_id', $orderLineIds)
            ->where('quantity', '>', 0)
            ->where('layer_type', FifoLayer::class)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('link_type', SalesOrderLine::class)
            ->groupBy('link_id')
            ->get();
    }

    public function getExistingReservationsForLines(array $orderLineIds): EloquentCollection
    {

        $fulfilledQuantityQuery = SalesOrderLine::query()
            ->selectRaw('id, coalesce(sum(sales_order_lines.fulfilled_quantity), 0) as fulfilled_quantity')
            ->whereIn('id', $orderLineIds)
            ->groupBy('id');

        return InventoryMovement::query()
            ->selectRaw('inventory_movements.*, sol.fulfilled_quantity')
            ->joinSub($fulfilledQuantityQuery, 'sol', 'sol.id', 'inventory_movements.link_id')
            ->whereIn('link_id', $orderLineIds)
            ->where('inventory_movements.quantity', '>', 0)
            ->where('layer_type', FifoLayer::class)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('link_type', SalesOrderLine::class)
            ->orderBy('inventory_movement_date')
            ->get();
    }

    public function createReservationMovements(array $movements): void
    {
        InventoryMovement::query()->insert($movements);
    }

    /**
     * @throws Exception
     */
    public function getForReference(string $reference): ?SalesOrderFulfillment
    {
        $parts = explode('.', $reference);

        if (count($parts) > 2) {
            throw new Exception('Unexpected reference format for: ' . $reference);
        } elseif (count($parts) === 1) {
            return null;
        }

        $salesOrderNumber = $parts[0];
        $fulfillmentSequence = $parts[1];

        if (!$salesOrder = SalesOrder::whereSalesOrderNumber($salesOrderNumber)->first()) {
            return null;
        }

        return SalesOrderFulfillment::query()
            ->whereSalesOrderId($salesOrder->id)
            ->whereFulfillmentSequence($fulfillmentSequence)
            ->first();
    }
}
