<?php

namespace App\Repositories\SalesOrder;

use App\Abstractions\AbstractRepository;
use App\Contracts\Repositories\SalesOrderRepositoryInterface;
use App\Contracts\Repositories\SalesOrderRepositoryInterfaceOld;
use App\Data\BulkActionData;
use App\Data\CustomerData;
use App\Data\FinancialLineData;
use App\Data\OrderLinkData;
use App\Data\SalesOrderLineData;
use App\Data\SalesOrderLineTaxLinesData;
use App\Data\UpdateSalesOrderData;
use App\Data\UpdateSalesOrderPayloadData;
use App\Data\NoteData;
use App\DTO\PaymentDto;
use App\DTO\SalesOrderDto;
use App\DTO\SalesOrderLineFinancialDto;
use App\Events\SalesOrderCreated;
use App\Events\SalesOrderFailed;
use App\Exceptions\ExternallyFulfilledCantBeFulfilledException;
use App\Exceptions\SalesOrder\SalesOrderFulfillmentException;
use App\Exceptions\SalesOrderLineNotFulfillableException;
use App\Helpers;
use App\Integrations\ShipStation;
use App\Jobs\CalculateDailyFinancialsJob;
use App\Jobs\InvalidateDailyAverageConsumptionForProductsJob;
use App\Jobs\PurgeUnusedCustomersAndAddresses;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Managers\TaxRateManager;
use App\Models\Address;
use App\Models\BackorderQueue;
use App\Models\Customer;
use App\Models\FifoLayer;
use App\Models\FinancialLine;
use App\Models\IntegrationInstance;
use App\Models\InventoryMovement;
use App\Models\Magento\Order as MagentoOrder;
use App\Models\OrderLink;
use App\Models\Payment;
use App\Models\PaymentType;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\SalesChannel;
use App\Models\SalesChannelType;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineFinancial;
use App\Models\SalesOrderLineLayer;
use App\Models\ShippingMethodMappingsSalesChannelToSku;
use App\Models\Shopify\ShopifyOrder as ShopifyOrder;
use App\Repositories\CustomerRepository;
use App\Repositories\DailyFinancialRepository;
use App\Repositories\FinancialLineRepository;
use App\Repositories\NoteRepository;
use App\Repositories\OrderLinkRepository;
use App\Repositories\PaymentRepository;
use App\Repositories\PaymentTypeRepository;
use App\Repositories\ProductRepository;
use App\Repositories\SalesOrderLineFinancialsRepository;
use App\Repositories\SalesOrderLineRepository;
use App\Repositories\Shopify\ShopifyOrderMappingRepository;
use App\Response;
use App\SDKs\Starshipit\StarshipitException;
use App\Services\InventoryManagement\BulkInventoryManager;
use App\Services\SalesOrder\SalesOrderManager;
use Carbon\Carbon;
use Carbon\CarbonImmutable;
use Closure;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Modules\Amazon\Entities\AmazonOrder;
use Spatie\LaravelData\Attributes\DataCollectionOf;
use Spatie\LaravelData\DataCollection;
use Spatie\LaravelData\Optional;
use Throwable;
use Validator;

class SalesOrderRepository extends AbstractRepository implements SalesOrderRepositoryInterface, SalesOrderRepositoryInterfaceOld
{
    protected SalesOrderLineFinancialsRepository $salesOrderLineFinancials;
    protected SalesOrderLineRepository $salesOrderLines;
    protected ProductRepository $products;
    protected NoteRepository $notes;
    protected TaxRateManager $taxRateManager;

    protected FinancialLineRepository $financialLineRepository;
    protected OrderLinkRepository $orderLinks;

    public function __construct()
    {
        $this->salesOrderLineFinancials = app(SalesOrderLineFinancialsRepository::class);
        $this->salesOrderLines = app(SalesOrderLineRepository::class);
        $this->products = app(ProductRepository::class);
        $this->financialLineRepository  = app(FinancialLineRepository::class);
        $this->notes = app(NoteRepository::class);
        $this->taxRateManager = app(TaxRateManager::class);
        $this->orderLinks = app(OrderLinkRepository::class);
    }

    private array $lineCursors = [];

    /**
     * Creates a sales order.
     *
     * @throws Exception
     * @throws Throwable
     */
    public function create(array $attributes): SalesOrder
    {
        $salesOrder = new SalesOrder($attributes);
        if (! empty($attributes['sales_channel_id'])) {
            $salesOrder->sales_channel_id = $attributes['sales_channel_id'];
        }

        $salesOrder->fulfillment_status = $attributes['fulfillment_status'] ?? SalesOrder::FULFILLMENT_STATUS_UNFULFILLED;
        $salesOrder->is_tax_included = $attributes['is_tax_included'] ?? false;
        customlog('shopifyOrdersBenchmark', $salesOrder->sales_order_number.': create - setCustomer - start');
        $customer = $salesOrder->setCustomer(
            $attributes['customer'] ?? null,
            false,
            $salesOrder->sales_channel_id ?? SalesChannel::LOCAL_CHANNEL_ID
        );
        customlog('shopifyOrdersBenchmark', $salesOrder->sales_order_number.': create - setCustomer - end');

        if ($customer) {
            $salesOrder->setShippingAddress($attributes['shipping_address'] ?? null);
            $salesOrder->setBillingAddress($attributes['billing_address'] ?? null);
        }

        // Handle shipping method
        if(empty($salesOrder->shipping_method_id) && !empty($salesOrder->requested_shipping_method)){
            $salesOrder->shipping_method_id = $this->getShippingMethodIdFromRequestedMethod($salesOrder);
        }

        $salesOrder->save();

        customlog('shopifyOrdersBenchmark', $salesOrder->sales_order_number.': create - setSalesOrderLines - start');
        $salesOrder->setSalesOrderLines($attributes['sales_order_lines'] ?? null);
        customlog('shopifyOrdersBenchmark', $salesOrder->sales_order_number.': create - setCustomer - end');
        $salesOrder->refresh();

        $this->setFinancialLines($salesOrder, $attributes['financial_lines'] ?? null);

        if (@$attributes['payment']) {
            $this->setPayment($salesOrder, $attributes['payment']);
        }

        if (isset($attributes['tags'])) {
            $salesOrder->syncTags($attributes['tags']);
        }

        return $salesOrder->load('salesOrderLines', 'shippingAddress', 'billingAddress', 'tags');
    }

    private function getShippingMethodIdFromRequestedMethod(SalesOrder $salesOrder): ?int
    {
        if ($salesOrder->requested_shipping_method === SalesChannel::UNSPECIFIED_SHIPPING_METHOD){
            return null;
        }

        $mapping = ShippingMethodMappingsSalesChannelToSku::firstOrCreate([
            'sales_channel_id' => $salesOrder->salesChannel->id,
            'sales_channel_method' => $salesOrder->requested_shipping_method,
        ]);
        return $mapping->shipping_method_id;
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function updateOrder(UpdateSalesOrderData $data): SalesOrder
    {
        $payload = $data->payload;
        $updatedSalesOrder = clone $data->salesOrder;
        $updatedSalesOrder->fill($payload->toArray());
        $updatedSalesOrder->save();

        if (!$payload->notes instanceof Optional) {
            $this->addNotes($updatedSalesOrder, $payload->notes);
        }

        if (!$payload->custom_field_values instanceof Optional) {
            $updatedSalesOrder->syncCustomFieldValues($payload->custom_field_values->toArray());
        }

        $this->updateSalesOrderLines($data);

        if (!$payload->financial_lines instanceof Optional) {
            $this->setFinancialLines($updatedSalesOrder, $payload->financial_lines->toArray());
        }

        if (!$payload->tags instanceof Optional) {
            $updatedSalesOrder->syncTags($payload->tags);
        }

        if (!$payload->ship_by_date instanceof Optional) {
            $this->updateSubmittedFulfillments($updatedSalesOrder);
        }

        if (
            // Changing warehouse
            !$payload->sales_order_lines instanceof Optional &&
            $payload->sales_order_lines
                ->toCollection()
                ->reject(function(SalesOrderLineData $line) {
                    if ($line->warehouse_id instanceof Optional || $line->salesOrderLine instanceof Optional) {
                        return true;
                    }
                    if ($line->salesOrderLine->warehouse_id == $line->warehouse_id) {
                        return true;
                    }
                    return false;
                })
                ->isNotEmpty()
        ) {
            $this->updateSubmittedFulfillments($updatedSalesOrder);
        }

        return $updatedSalesOrder->load('salesOrderLines', 'shippingAddress', 'billingAddress', 'tags');
    }

    private function updateSalesOrderLines(UpdateSalesOrderData $data): void
    {
        if ($data->payload->sales_order_lines instanceof Optional) {
            return;
        }

        $data->payload->sales_order_lines->each(function (SalesOrderLineData $line) use ($data) {
            if ($line->salesOrderLine instanceof Optional) {
                $salesOrderLine = new SalesOrderLine($line->toArray());
            } else {
                $salesOrderLine = $line->salesOrderLine;
            }
            $salesOrderLine->fill($line->toArray());
            $salesOrderLine->quantity = $line->quantity;
            $salesOrderLine->sales_order_id = $data->salesOrder->id;
            $salesOrderLine->save();
        });
    }

    /**
     * @throws Throwable
     */
    public function saveWithRelations(
        #[DataCollectionOf(SalesOrderDto::class)]
        #[DataCollectionOf(UpdateSalesOrderPayloadData::class)]
        DataCollection $data
    ): Collection
    {
        $data = $data->toCollection();

        return DB::transaction(function () use ($data) {
            $salesOrderCollection = $this->saveCustomer($data);
            $salesOrderCollection = $this->save($salesOrderCollection, SalesOrder::class);
            $salesOrderLineCollection = $this->saveLines($salesOrderCollection, $data);
            $this->saveFinancialLines($data, $salesOrderCollection);
            $this->savePayments($data, $salesOrderCollection);
            $this->saveOrderLinks($data, $salesOrderCollection);

            $salesOrderLineCollection = app(SalesOrderLineRepository::class)->getMappedLinesFromIds($salesOrderLineCollection->pluck('id')->toArray());

            $salesOrderLinesToReserve = $this->salesOrderLines->sanitizeLinesForAllocation($salesOrderLineCollection);

            (new BulkInventoryManager())->bulkAllocateNegativeInventoryEvents($salesOrderLinesToReserve);

            return $salesOrderCollection;
        });
    }

    public function addNote(SalesOrder $salesOrder, NoteData $data): SalesOrder
    {
        $salesOrder->notes()->create($data->toArray());

        return $salesOrder;
    }

    public function findFulfillmentById(int $id, array $relations = []): SalesOrderFulfillment|Model
    {
        return SalesOrderFulfillment::with($relations)->findOrFail($id);
    }

    public function getTotalExistingReservationsForLine(SalesOrderLine $orderLine): float
    {
        return abs(collect($orderLine->getReductionActiveMovements()->toArray())->sum('quantity'));
    }

    /**
     * Get or Create sales order by "sales_order_number" and "sales_channel_id", then fire Sales Order events.
     */
    public function firstOrCreateSalesOrder($customer_reference, $sales_channel_id, array $otherFields, bool $is_fba = false): SalesOrder
    {
        $salesOrder = SalesOrder::with('salesChannel', 'salesOrderLines', 'shippingAddress', 'billingAddress', 'customer')
            ->firstOrNew(
                [
                    'sales_order_number' => $customer_reference,
                    'sales_channel_id' => $sales_channel_id,
                ]
            );

        if ($salesOrder->exists) {
            $salesOrder->fill(Arr::except($otherFields, 'status'))->save();
        } else {
            $salesOrder->fill($otherFields)->save();

            // Fire Sales Order Created event.
            event(new SalesOrderCreated($salesOrder, $is_fba));
        }

        return $salesOrder;
    }

    /**
     * Validate and save Sales Order.
     */
    public function saveSalesOrder(SalesOrder $salesOrder): SalesOrder
    {
        $salesOrderValidator = Validator::make($salesOrder->attributesToArray(), [
            'sales_channel_id' => 'required',
            'sales_order_number' => 'required|unique:sales_orders,sales_order_number,'.$salesOrder->id.',id,sales_channel_id,'.$salesOrder->sales_channel_id,
            'status' => 'required',
            'customer_id' => 'required',
            'shipping_address_id' => 'required',
            'billing_address_id' => 'required',
            'sales_order_lines' => 'required|array|min:1',
        ]);

        /**
         * If validation failed, set status to "draft" and fire Sales Order Failed event.
         */
        if ($salesOrderValidator->fails()) {
            if ($salesOrder->status == SalesOrder::STATUS_OPEN) {
                $salesOrder->status = SalesOrder::STATUS_DRAFT;

                Response::instance()->addWarningsFromValidator($salesOrderValidator);
            }

            if ($salesOrder->wasRecentlyCreated) {
                // Fire Sales Order Failed event.
                event(new SalesOrderFailed($salesOrder));
            }
        }

        // "sales_order_lines" for validation only. Unset "sales_order_lines" from object.
        unset($salesOrder->sales_order_lines);

        $salesOrder->save();

        return $salesOrder;
    }

    /**
     * Validate an Address with required fields which enables us to save this address.
     *
     *
     *
     * @example ["status" => true]
     * @example ["status" => false, "message" => ["country_code" => ["required"]]]
     */
    public function validateAddress(Address $address, string $from): array
    {
        $addressValidator = Validator::make($address->toArray(), [
            'name' => 'required',
            'address1' => 'required',
            'city' => 'required',
            'country_code' => 'required',
        ], ['required' => "The :attribute field in {$from} is required."]);

        if ($addressValidator->fails()) {
            return ['status' => false, 'message' => $addressValidator->errors()->toArray()];
        }

        return ['status' => true];
    }

    /**
     * Validate and save shipping address.
     *
     *
     * @return Address|null if validation failed return null.
     */
    public function saveShippingAddress(Address $address, SalesOrder $order): ?Address
    {
        $validateAddress = $this->validateAddress($address, 'shipping address');
        if ($validateAddress['status']) {
            $shippingAddressValidator = Validator::make($address->toArray(), ['zip' => 'required'], ['required' => 'The :attribute field in shipping address is required.']);

            if ($shippingAddressValidator->fails() && $order->status == SalesOrder::STATUS_OPEN) {
                Response::instance()->addWarningsFromValidator($shippingAddressValidator, 'shipping_address');
            }
        } else {
            if ($order->status == SalesOrder::STATUS_OPEN) {
                Response::instance()->addWarningsFromValidator($validateAddress['message'], 'shipping_address');
            }
        }

        // link with existing address if same attributes
        if (! $address->exists && ($existingAddress = $this->addressExists($address))) {
            $existingAddress->fill($address->getAttributes());
            $address = $existingAddress;
        }

        if ($address->save()) {
            Response::instance()->addWarning(__('messages.address.locked'), 'ShippingAddress'.Response::CODE_IS_LOCKED, 'shipping_address.id', ['id' => $address->id]);
        }

        return $address;
    }

    /**
     * Validate and save billing address.
     *
     *
     * @return Address|null if validation failed return null.
     */
    public function saveBillingAddress(Address $address, SalesOrder $order): ?Address
    {
        $validateAddress = $this->validateAddress($address, 'billing address');
        if (! $validateAddress['status'] && $order->status == SalesOrder::STATUS_OPEN) {
            Response::instance()->addWarningsFromValidator($validateAddress['message'], 'billing_address');
        }

        // link with existing address if same attributes
        if (! $address->exists && ($existingAddress = $this->addressExists($address))) {
            $existingAddress->fill($address->getAttributes());
            $address = $existingAddress;
        }

        if ($address->save()) {
            Response::instance()->addWarning(__('messages.address.locked'), 'BillingAddress'.Response::CODE_IS_LOCKED, 'billing_address.id', ['id' => $address->id]);
        }

        return $address;
    }

    /**
     * Get or create customer.
     *
     * If customer not get from DB, Check customer by "email" or ("name" and "zip"), if not exists create new.
     */
    public function firstOrCreateCustomer(Customer $customer, Address $address, ?SalesOrder $order = null): ?Customer
    {
        /**
         * Validate Customer address.
         *
         * if basic validation failed will not save address or customer and return customer null.
         */
        $validateAddress = $this->validateAddress($address, 'customer address');
        if ($validateAddress['status']) {
            $address->save();
        } else {
            Response::instance()->addWarningsFromValidator($validateAddress['message'], 'customer_address');

            return null;
        }

        /**
         * If customer with "id" (exists on DB), save it with default address.
         */
        if ($customer->id) {
            $customer->default_shipping_address_id = $address->id;
            $customer->name = $address->name;
            $customer->email = $address->email;
            $customer->address1 = $address->address1;
            $customer->zip = $address->zip;
            $customer->save();

            return $customer;
        }

        /**
         * Now if customer not exists on DB, we will search by "email" or ("name" and "zip").
         *
         * If customer exists we will return it without editing his address (if we editing his address manually from UI).
         */
        $exists_customer = $this->customerExists($address);

        if ($exists_customer) {
            return $exists_customer;
        }

        /**
         * Now if customer not exists we will create new customer will default address.
         */
        $customer->default_shipping_address_id = $address->id;
        $customer->name = $address->name;
        $customer->email = $address->email;
        $customer->address1 = $address->address1;
        $customer->zip = $address->zip;
        $customer->save();

        return $customer;
    }

    /**
     * Check if customer exists by address.
     *
     * We search by "email" or ("name" and "zip").
     */
    public function customerExists(Address $address): ?Customer
    {
        return Customer::where(function ($query) use ($address) {
            $query->where(function ($query) use ($address) {
                $query->whereNotNull('email')
                    ->where('email', '!=', '')
                    ->where('email', $address->email);
            })->orWhere(function ($query) use ($address) {
                $query->where('name', $address->name)
                    ->whereNotNull('zip')
                    ->where('zip', '!=', '')
                    ->where('zip', $address->zip)
                    ->whereNotNull('address1')
                    ->where('address1', '!=', '')
                    ->where('address1', $address->address1);
            });
        })->first();
    }

    /**
     * Check if address exists by address.
     *
     * We search by "country code" , "zip" , "address1" and "name".
     */
    public function addressExists(Address $address): ?Address
    {
        return Address::where(function (Builder $query) use ($address) {
            // country_code
            $query->whereNotNull('country_code')
                ->where('country_code', '!=', '')
                ->where('country_code', $address->country_code)
                // zip
                ->whereNotNull('zip')
                ->where('zip', '!=', '')
                ->where('zip', $address->zip)
                // address1
                ->whereNotNull('address1')
                ->where('address1', '!=', '')
                ->where('address1', $address->address1)
                // name
                ->whereNotNull('name')
                ->where('name', '!=', '')
                ->where('name', $address->name);
        })->first();
    }

    /**
     * @throws Exception
     */
    public function reserveInventoryFromSalesOrderLine(SalesOrder $salesOrder, SalesOrderLine $salesOrderLine, $saveLine = true)
    {
        if (! empty($salesOrderLine->product_id)) {
            if ($salesOrder->status === SalesOrder::STATUS_OPEN && empty($salesOrderLine->fifo_layer_id)) {
                $salesOrderLine->load('product', 'product.activeFifoLayers');
                /** @var Product $product */
                $product = $salesOrderLine->product;

                $salesOrderLine->setCogsIfFinancialLineExists($product->average_cost * $salesOrderLine->quantity ?: 0);

                $activeFifoLayer = $product->currentFifoLayer;
                if ($activeFifoLayer) {
                    // Set fifo layer and unit cost for sales order line.
                    $salesOrderLine->fifo_layer_id = $activeFifoLayer->id;

                    /**
                     * Create new two inventory movements, one for Active and other for Reserved.
                     */
                    $activeInventoryMovement = new InventoryMovement();
                    $activeInventoryMovement->inventory_movement_date = Carbon::now();
                    $activeInventoryMovement->quantity = -$salesOrderLine->quantity;
                    $activeInventoryMovement->type = InventoryMovement::TYPE_SALE;
                    $activeInventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
                    $activeInventoryMovement->warehouse_id = $salesOrder->warehouse_id;
                    $activeInventoryMovement->warehouse_location_id = $salesOrder->warehouse->defaultLocation->id ?? null;
                    $activeInventoryMovement->link_id = $salesOrder->id;
                    $activeInventoryMovement->link_type = SalesOrder::class;
                    $activeInventoryMovement->fifo_layer_id = $activeFifoLayer->id;
                    $product->inventoryMovements()->save($activeInventoryMovement);

                    $reservedInventoryMovement = $activeInventoryMovement->replicate();
                    $reservedInventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_RESERVED;
                    $reservedInventoryMovement->quantity = $salesOrderLine->quantity;
                    $product->inventoryMovements()->save($reservedInventoryMovement);
                }
            }
        }

        if ($saveLine) {
            $salesOrderLine->save();
        }
    }

    /**
     * Get Nominal Code by sales channel and sales channel attribute.
     */
    public function getNominalCodeBySalesChannelAttribute(SalesChannel $salesChannel, $attributeName): int
    {
        // This array must filled when setup Sales Channel or when Getting-Started wizard.
        $mapSalesChannelAttributeWithNominalCodes = [
            SalesChannelType::TYPE_AMAZON => [
                'Order Line Price' => 1, // Sales Order Line Price
                'ShippingPrice' => 2,
                'ShippingTax' => 3,
                'ShippingDiscount' => 4,
                'PromotionDiscount' => 5,
            ],
            SalesChannelType::TYPE_EBAY => [
                'Order Line Price' => 1, // Sales Order Line Price
                'ActualShippingCost' => 2,
                'ActualHandlingCost' => 2,
            ],
            SalesChannelType::TYPE_SHOPIFY => [
                'Order Line Price' => 1, // Sales Order Line Price
                'total_tax' => 6,
                'total_discounts' => 5,
                'total_shipping_lines' => 2,
                'total_shipping_lines_discounted' => 4,
            ],
            SalesChannelType::TYPE_WOOCOMMERCE => [
                'Order Line Price' => 1, // Sales Order Line Price
                'total_tax' => 6,
                'total_shipping' => 2,
                'shipping_tax' => 3,
                'total_discount' => 5,
            ],
        ];

        return $mapSalesChannelAttributeWithNominalCodes[$salesChannel->sales_channel_type][$attributeName];
    }

    /**
     * Creates a SalesOrderFulfillment entity based on the given SalesOrder and the provided payload.
     * The method verifies if the fulfillment is possible based on the payload, fetches fulfillable lines,
     * and creates a SalesOrderFulfillment entity along with related lines.
     *
     * @param  SalesOrder  $order the SalesOrder entity for which the SalesOrderFulfillment entity is to be created
     * @param  array  $payload the payload containing the data to be used for creating the SalesOrderFulfillment entity
     *
     * @throws SalesOrderFulfillmentException
     */
    public function createFulfillment(SalesOrder $order, array $payload): SalesOrderFulfillment
    {
        // Fetch the lines that can be fulfilled
        $fulfillableLines = $this->getFulfillableLines($payload);

        // If there are no lines that can be fulfilled, throw an exception
        if ($fulfillableLines->isEmpty()) {
            throw new SalesOrderFulfillmentException(
                salesOrder: $order,
                message: "Sales Order: $order->sales_order_number has no fulfillable lines."
            );
        }

        // If there are fulfillable lines, create the fulfillment and its related lines
        return $this->createFulfillmentWithLines($order, $payload, $fulfillableLines);
    }

    /**
     * Gets SalesOrderLine entities based on the given fulfillment line.
     * If the fulfillment line contains a 'sales_channel_line_id', fetch all lines with that ID.
     * If not, fetch the SalesOrderLine with the 'sales_order_line_id' specified in the fulfillment line.
     *
     * @param  array  $fulfillmentLine the fulfillment line data
     * @return Collection the collection of SalesOrderLine entities
     */
    private function getSalesOrderLines(array $fulfillmentLine): Collection
    {
        if (isset($fulfillmentLine['sales_channel_line_id'])) {
            // If 'sales_channel_line_id' is set, fetch all lines with that ID
            return SalesOrderLine::query()
                ->where('sales_channel_line_id', $fulfillmentLine['sales_channel_line_id'])
                ->get();
        }

        // If 'sales_channel_line_id' is not set, fetch the line with the 'sales_order_line_id'
        return collect([SalesOrderLine::query()->findOrFail($fulfillmentLine['sales_order_line_id'])]);
    }

    /**
     * Fetches all fulfillable lines from the given payload.
     * The fulfillable lines are those where the fulfillable quantity is greater than 0.
     *
     * @param  array  $payload the payload containing the data to be used for creating the SalesOrderFulfillment entity
     * @return Collection the collection of fulfillable lines
     */
    private function getFulfillableLines(array $payload): Collection
    {
        return collect($payload['fulfillment_lines'])
            ->map(function ($fulfillmentLine) {
                // For each fulfillment line, get the next SalesOrderLine
                $salesOrderLine = $this->getNextSalesOrderLine($fulfillmentLine);

                // Return the line with additional fields - 'sales_order_line_id' and 'fulfillable_quantity'
                return array_merge($fulfillmentLine, [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'fulfillable_quantity' =>
                    // This function gets the fulfillable quantity for a specific line ID
                        $this->getFulfillableQuantityByLineId($salesOrderLine->id),
                ]);
            })
            // Only return lines where the fulfillable quantity is greater than 0
            ->filter(fn ($fulfillableLine) => $fulfillableLine['fulfillable_quantity'] > 0);
    }

    /**
     * Fetches the next SalesOrderLine based on the given fulfillment line.
     * This function uses static line cursors to maintain state and fetch different lines on subsequent calls.
     * It throws an exception if no more lines are available for the given ID.
     *
     * @param  array  $fulfillmentLine the fulfillment line data
     * @return SalesOrderLine the SalesOrderLine entity
     *
     * @throws Exception
     */
    private function getNextSalesOrderLine(array $fulfillmentLine): SalesOrderLine|Model
    {
        // Determine the line ID key based on whether 'sales_channel_line_id' is set in the fulfillment line
        $lineIdKey = isset($fulfillmentLine['sales_channel_line_id']) ?
            'sales_channel_line_id' : // This corresponds to the actual field name in the SalesOrderLine model
            'id'; // The 'sales_order_line_id' in the array corresponds to the 'id' field in the SalesOrderLine model

        // If the sales order line id is set, we should know the sales order id so that we can filter by it below
        if (isset($fulfillmentLine['sales_order_line_id'])) {
            $salesOrderId = SalesOrderLine::query()
                ->where('id', $fulfillmentLine['sales_order_line_id'])
                ->firstOrFail()
                ->sales_order_id;
        }

        // Fetch the line ID
        $lineId = $fulfillmentLine[$lineIdKey === 'id' ? 'sales_order_line_id' : $lineIdKey];

        // If this is the first time this line ID is being accessed, initialize its cursor to 0
        if (! isset($this->lineCursors[$lineId])) {
            $this->lineCursors[$lineId] = 0;
        }

        // Fetch the SalesOrderLine based on the line ID key and the cursor position
        $salesOrderLine = SalesOrderLine::query()
            ->where($lineIdKey, $lineId)
            ->whereNotNull('product_id');

        if (isset($salesOrderId)) {
            $salesOrderLine->where('sales_order_id', $salesOrderId);
        }

        $salesOrderLine = $salesOrderLine
            ->skip($this->lineCursors[$lineId])
            ->first();

        // If there are no more lines for this ID, throw an exception
        if ($salesOrderLine === null) {
            throw new SalesOrderLineNotFulfillableException('No more fulfillable SalesOrderLines for '.$lineIdKey.': '.$lineId);
        }

        // Increase the cursor position for the next call
        $this->lineCursors[$lineId]++;

        return $salesOrderLine;
    }

    /**
     * Creates a new SalesOrderFulfillment and associated SalesOrderFulfillmentLines.
     * Updates the fulfilled_quantity on the SalesOrderLines linked with these fulfillments.
     *
     * @param  SalesOrder  $order the SalesOrder entity
     * @param  array  $payload the original payload
     * @param  Collection  $fulfillableLines a collection of fulfillable lines
     * @return SalesOrderFulfillment the newly created SalesOrderFulfillment entity
     */
    private function createFulfillmentWithLines(SalesOrder $order, array $payload, Collection $fulfillableLines): SalesOrderFulfillment
    {
        // Create a new SalesOrderFulfillment from the payload and save it to the database
        $fulfillment = new SalesOrderFulfillment($payload);
        $order->salesOrderFulfillments()->save($fulfillment);

        // Prepare an array of fulfillment lines to insert into the database.
        $bulkFulfillmentLines = $fulfillableLines->map(function ($fulfillableLine) use ($fulfillment) {
            return [
                'quantity' => $fulfillableLine['quantity'],
                'sales_order_line_id' => $fulfillableLine['sales_order_line_id'],
                'sales_order_fulfillment_id' => $fulfillment->id,
                'metadata' => @json_encode($fulfillableLine['metadata']),
            ];
        })->toArray();

        // Insert all the fulfillment lines into the database at once (bulk insert)
        $fulfillment->salesOrderFulfillmentLines()->insert($bulkFulfillmentLines);

        // Note that we update fulfilled quantity in sales order lines via fulfillment manager.

        // Return the newly created SalesOrderFulfillment entity
        return $fulfillment;
    }

    private function getFulfillableQuantityByLineId(int $salesOrderLineId): int
    {
        /** @var SalesOrderLine $orderLine */
        $orderLine = SalesOrderLine::with(['backorderQueue'])->findOrFail($salesOrderLineId);

        return max(0, $orderLine->unfulfilled_quantity - $orderLine->active_backordered_quantity);
    }

    public function deleteFulfillmentLines(SalesOrderFulfillment $fulfillment): SalesOrderFulfillment
    {
        $fulfillment->salesOrderFulfillmentLines->each(fn (SalesOrderFulfillmentLine $line) => $line->delete());

        return $fulfillment;
    }

    public function updateFulfillmentMovementsDate(SalesOrderFulfillment $fulfillment, CarbonImmutable $date): void
    {
        InventoryMovement::query()->where('link_type', SalesOrderFulfillmentLine::class)
            ->whereIn('link_id', $fulfillment->salesOrderFulfillmentLines->pluck('id'))
            ->update(['inventory_movement_date' => $date]);
    }

    public function findOrCreateFulfillmentLine(
        SalesOrderFulfillment $fulfillment,
        array $line
    ): SalesOrderFulfillmentLine {
        if (! empty($line['id'])) {
            $fulfillmentLine = $fulfillment->salesOrderFulfillmentLines()->findOrFail($line['id']);
        } else {
            $fulfillmentLine = new SalesOrderFulfillmentLine($line);
            $fulfillment->salesOrderFulfillmentLines()->save($fulfillmentLine);
        }
        // Update the fulfilled quantity for the fulfillment line and the sales order line.
        $fulfillmentLine->update(['quantity' => $line['quantity']]);
        $fulfillmentLine->salesOrderLine->update(['fulfilled_quantity' => $line['quantity']]);

        return $fulfillmentLine;
    }

    public function getFulfillableLinesForOrder(SalesOrder $order): Collection
    {
        return $order->warehousedProductLines()
            ->whereRaw('quantity > (fulfilled_quantity + externally_fulfilled_quantity)')
            ->get();
    }

//    /**
//     * @throws Throwable
//     */
//    public function save(SalesOrderDto|Collection|SalesOrderBulkDto $data, Model $documentTypeClass): SalesOrder|EloquentCollection|BulkImportResponseDto
//    {
//        if ($data instanceof SalesOrderBulkDto) {
//            $data = $data->data;
//        }
//
//        if ($data instanceof SalesOrderDto) {
//            return $this->saveOrder($data, $documentTypeClass);
//        }
//
//        $salesOrders = new EloquentCollection();
//        $data->each(function (SalesOrderDto $salesOrderDto) use ($salesOrders, $documentTypeClass) {
//            $salesOrders->add($this->saveOrder($salesOrderDto, $documentTypeClass));
//        });
//
//        return $salesOrders;
//    }

    /**
     * @param  BulkActionData  $data
     * @param  Closure  $callback
     * @return void
     */
    public function chunkBy(BulkActionData $data, Closure $callback): void
    {
        $size = 1000;
        $query = SalesOrder::with([
            'salesOrderLines.activeBackorderQueue',
            'salesOrderLines.inventoryMovements',
            'salesOrderFulfillments.salesOrderFulfillmentLines',
            'customer'
        ]);

        if(!$data->ids instanceof Optional){
            $query->whereIn('id', $data->ids);
        }

        if(!$data->filters instanceof Optional){
            $query->filter(['filters' => $data->filters]);
        }

        $query->chunk($size, function ($orders) use ($callback) {
            $callback($orders);
        });
    }

    public function getLastFulfillmentSequencesForOrderIds(array $salesOrderIds): array
    {
        return SalesOrderFulfillment::query()
            ->selectRaw('sales_order_id, max(fulfillment_sequence) as fulfillment_sequence')
            ->whereIn('sales_order_id', $salesOrderIds)
            ->groupBy('sales_order_id')
            ->pluck('fulfillment_sequence', 'sales_order_id')
            ->toArray();
    }

    public function updateFulfilledQuantityCacheForLines(array $data): void
    {
        batch()->update(new SalesOrderLine, $data, 'id');
    }

    private function saveOrder(SalesOrderDto $salesOrderDto, Model $documentTypeClass): SalesOrder
    {
        return app(SalesOrderManager::class)->createOrder($salesOrderDto->toArray());
        //        $customer = null;
        //
        //        if ($salesOrderDto->customer) {
        //            $customer = app(CustomerRepository::class)->save($salesOrderDto->customer);
        //        }
        //
        //        $salesOrder = null;
        //        DB::transaction(function () use ($salesOrderDto, &$salesOrder, $customer, $documentTypeClass) {
        //            /** @var SalesOrder $salesOrder */
        //            $salesOrder = SalesOrder::query()->updateOrCreate(['id' => $salesOrderDto->id], $salesOrderDto->toArray());
        //
        //            if ($customer) {
        //                $salesOrder->customer_id = $customer->id;
        //                $salesOrder->shipping_address_id = $customer->default_shipping_address_id;
        //                $salesOrder->billing_address_id = $customer->default_billing_address_id;
        //                $salesOrder->update();
        //            }
        //
        //            $documentTypeClass::where('id', $salesOrder->sales_channel_order_id)
        //                    ->update([
        //                        'error_log' => OrderSyncStatusEnum::SKU_ORDER_CREATED(),
        //                    ]);
        //
        //            $this->saveLines($salesOrder, $salesOrderDto->sales_order_lines);
        //        });
        //
        //        return $salesOrder;
    }

    public function findById(int $id): ?SalesOrder
    {
        /** @var SalesOrder $salesOrder */
        $salesOrder = SalesOrder::query()->find($id);

        return $salesOrder;
    }

    public function delete(SalesOrder|Collection $salesOrders): bool
    {
        return $salesOrders->delete();
    }

    /**
     * @throws Throwable
     */
    public function bulkDelete(array $ids): void
    {
        $productIds = [];
        $invalidDays = [];
        $userTimezone = Helpers::getAppTimezone();

        foreach (array_chunk($ids, 250) as $chunk) {
            DB::transaction(function () use ($chunk, &$productIds, &$invalidProductDays, $userTimezone, &$invalidDays) {
                customlog('salesOrderBulkDeletion', 'Deleting chunk of '.count($chunk).' sales orders');

                $salesOrdersQuery = SalesOrder::with('salesOrderLines')
                    ->whereIn('id', $chunk)
                    ->orderBy('id', 'desc');

                $invalidDays = array_merge($salesOrdersQuery->pluck('order_date')->map(function ($date) use (&$invalidDays, $userTimezone) {
                    return Carbon::parse($date)->setTimezone($userTimezone)->startOfDay()->toDateString();
                })->toArray());

                $salesOrders = $salesOrdersQuery->get();

                /*
                 * We want to first delete usages, since it is easier to delete negative inventory movements before deleting
                 * positive ones
                 */
                $salesOrderLines = SalesOrderLine::query()
                    ->whereIn('sales_order_id', $salesOrders->pluck('id'));
                customlog('salesOrderBulkDeletion', $salesOrderLines->count().' sales order lines to delete');

                $productIds = array_merge($productIds, $salesOrderLines->pluck('product_id')->toArray());

                $backorderQueues = BackorderQueue::query()
                    ->whereIn('sales_order_line_id', $salesOrderLines->pluck('id'));
                customlog('salesOrderBulkDeletion', $backorderQueues->count().' backorder queues to delete');
                $backorderQueues->delete();

                $salesOrderLineLayers = SalesOrderLineLayer::query()
                    ->whereIn('sales_order_line_id', $salesOrderLines->pluck('id'));
                customlog('salesOrderBulkDeletion', $salesOrderLineLayers->count().' sales order line layers to delete');
                $salesOrderLineLayers->delete();

                $salesOrderLineFinancials = SalesOrderLineFinancial::query()
                    ->whereIn('sales_order_line_id', $salesOrderLines->pluck('id'));
                customlog('salesOrderBulkDeletion', $salesOrderLineFinancials->count().' sales order line financials to delete');
                $salesOrderLineFinancials?->delete();

                /*
                 * Delete sales order line reservations (active -1 and reserved +1).  This will help us to delete any
                 * FIFO layers created from sales credit return lines further down in the logic
                 */
                $reservationMovements = InventoryMovement::query()
                    ->whereIn('link_id', $salesOrderLines->pluck('id'))
                    ->where('link_type', SalesOrderLine::class);

                customlog('salesOrderBulkDeletion', $reservationMovements->count().' inventory movements to delete');

                // Update FIFO Layer fulfilled_quantity cache
                $reservationMovements->clone()->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)->each(function (InventoryMovement $inventoryMovement) {
                    if ($inventoryMovement->layer_type == FifoLayer::class) {
                        $fifoLayer = $inventoryMovement->layer;
                        if (! $fifoLayer) {
                            return;
                        }
                        customlog('salesOrderBulkDeletion', 'Updating FIFO Layer Fulfilled Quantity for Layer '.$fifoLayer->id.' by '.$inventoryMovement->quantity);
                        $fifoLayer->fulfilled_quantity += $inventoryMovement->quantity;
                        $fifoLayer->save();
                    }
                });

                $reservationMovements->delete();

                /*
                 * The first usages would be any sales order fulfillments.  We need to delete them even if they are
                 * connected to a shipping provider order
                 */

                $salesOrderFulfillments = SalesOrderFulfillment::query()
                    ->whereIn('sales_order_id', $salesOrders->pluck('id'));
                customlog('salesOrderBulkDeletion', $salesOrderFulfillments->count().' sales order fulfillments to delete');

                $salesOrderFulfillments->each(function (SalesOrderFulfillment $salesOrderFulfillment) {
                    customlog('salesOrderBulkDeletion', 'Deleting fulfillment '.$salesOrderFulfillment->id);
                    $salesOrderFulfillment->delete(true);
                });

                $salesCredits = SalesCredit::query()
                    ->whereIn('sales_order_id', $salesOrders->pluck('id'));
                customlog('salesOrderBulkDeletion', $salesCredits->count().' sales credits to delete');
                $salesCredits->each(function (SalesCredit $salesCredit) {
                    customlog('salesOrderBulkDeletion', 'Deleting sales credit '.$salesCredit->id);
                    $salesCredit->delete();
                });

                // Delete all child links
                $childLinks = OrderLink::query()
                    ->whereIn('child_id', $salesOrders->pluck('id'))
                    ->where('child_type', SalesOrder::class);
                customlog('salesOrderBulkDeletion', $childLinks->count().' child order links to delete');
                $childLinks->delete();

                // Delete all parent links
                $parentLinks = OrderLink::query()
                    ->whereIn('parent_id', $salesOrders->pluck('id'))
                    ->where('parent_type', SalesOrder::class);
                customlog('salesOrderBulkDeletion', $parentLinks->count().' parent order links to delete');
                $parentLinks->delete();

                $payments = Payment::query()
                    ->whereIn('link_id', $salesOrders->pluck('id'))
                    ->where('link_type', SalesOrder::class);
                customlog('salesOrderBulkDeletion', $payments->count().' payments to delete');
                $payments->delete();

                $taggables = DB::table('taggables')
                    ->whereIn('taggable_id', $salesOrders->pluck('id'))
                    ->where('taggable_type', SalesOrder::class);
                customlog('salesOrderBulkDeletion', $taggables->count().' taggables to delete');
                $taggables->delete();

                $purchaseOrderLinks = PurchaseOrder::query()
                    ->whereIn('sales_order_id', $salesOrders->pluck('id'));
                customlog('salesOrderBulkDeletion', $purchaseOrderLinks->count().' purchase orders to detach');
                $purchaseOrderLinks->update(['sales_order_id' => null]);

                // Sales Channel Unlinking
                $shopifyOrders = ShopifyOrder::query()
                    ->whereIn('sku_sales_order_id', $salesOrders->pluck('id'));
                customlog('salesOrderBulkDeletion', $shopifyOrders->count().' shopify orders to detach');

                // Delete Shopify Order Mappings
                app(ShopifyOrderMappingRepository::class)->bulkDeleteForShopifyOrders($shopifyOrders->pluck('id'));

                $shopifyOrders->update([
                    'sku_sales_order_id' => null,
                    'archived_at' => Carbon::now()->setTimezone('UTC'),
                ]);

                $magentoOrders = MagentoOrder::query()
                    ->whereIn('sales_order_id', $salesOrders->pluck('id'));
                customlog('salesOrderBulkDeletion', $magentoOrders->count().' magento orders to detach');
                $magentoOrders->update([
                    'sales_order_id' => null,
                    'fulfillments_map' => null,
                    'refunds_map' => null,
                    'returns_map' => null,
                    'archived_at' => Carbon::now()->setTimezone('UTC'),
                ]);

                /*
                 * This is used as a backup in case there are sales credit lines associated to sales orders but they
                 * lack the sales credit to sales order connection
                 */
                $salesCreditLines = SalesCreditLine::query()
                    ->whereIn('sales_order_line_id', $salesOrderLines->pluck('id'));
                customlog('salesOrderBulkDeletion', $salesCreditLines->count().' sales credit lines to delete', [
                    'sales_credit_lines' => $salesCreditLines->get()->pluck('id'),
                ]);
                $salesCreditLines->each(function (SalesCreditLine $salesCreditLine) {
                    $salesCreditLine->salesCredit?->delete();
                });

                // Delete sales order line financials
                SalesOrderLineFinancial::query()
                    ->whereIn('sales_order_line_id', $salesOrderLines->pluck('id'));

                try {
                    // Delete in proper order (lines split from other lines first)
                    $salesOrderLines->whereNotNull('split_from_line_id')->delete();
                    $salesOrderLines->whereNull('split_from_line_id')->delete();
                } catch (Exception $e) {
                    Log::error($e->getMessage());
                    Log::error($e->getTraceAsString());
                    dd($e->getMessage());
                }

                FinancialLine::query()
                    ->whereIn('sales_order_id', $salesOrders->pluck('id'))
                    ->delete();

                $salesOrdersQuery->delete();
            }, 3);
        }
        $productIds = array_unique($productIds);

        // Get rid of duplicate days
        $invalidDays = array_unique($invalidDays);

        // Invalidate caches for product financial reporting and DAC
        app(DailyFinancialRepository::class)->invalidateForDates($invalidDays);
        dispatch(new InvalidateDailyAverageConsumptionForProductsJob($productIds));

        // Recalculate product financial reporting, which includes DAC
        dispatch(new CalculateDailyFinancialsJob())->onQueue('financials');

        // Update product inventory and average cost caches
        dispatch(new UpdateProductsInventoryAndAvgCost($productIds));

        // Purge unused customers and addresses
        dispatch(new PurgeUnusedCustomersAndAddresses());

        // Sync Backorder Queue Coverages since we removed reservations on deleted sales orders.
        dispatch(new SyncBackorderQueueCoveragesJob(null, null, $productIds));
    }

    public function setFinancialLines(SalesOrder $salesOrder, $financialLinesPayload, bool $sync = true): ?bool
    {
        if ($financialLinesPayload === false) {
            return null;
        }
        $existingFinancialLineIds = [];

        foreach ($financialLinesPayload ?: [] as $financialLinePayload) {
            // Used for bulk, we need to remove it here since we already know the sales order id
            unset($financialLinePayload['sales_order_number']);
            if (! empty($financialLine['id'])) {
                $financialLine = FinancialLine::with([])->findOrFail($financialLinePayload['id']);
            } else {
                $financialLine = null;
            }

            // Try to find existing financial line
            if (! isset($financialLine)) {
                $financialLine = FinancialLine::query()
                    ->where('sales_order_id', $salesOrder->id)
                    ->where('description', $financialLinePayload['description'])
                    ->where('amount', $financialLinePayload['amount'])
                    ->first();
            }

            if (! isset($financialLine)) {
                $financialLine = new FinancialLine();
            }

            $financialLine->fill($financialLinePayload);

            try {
                $salesOrder->financialLines()->save($financialLine);
            } catch (Throwable $e) {
                dd($e);
            }

            $existingFinancialLineIds[] = $financialLine->id;
            unset($financialLine);
        }

        // sync financial lines
        if ($sync) {
            $salesOrder->financialLines()
                ->whereNotIn('id', $existingFinancialLineIds)->delete();
        }

        // allocate financial lines
        //$this->financialLineRepository->allocateLines($salesOrder);

        return true;
    }

    public function setPayment(SalesOrder $salesOrder, $paymentLine)
    {
        if ($paymentType = PaymentType::where('name', $paymentLine['payment_type'])->first()) {
            $paymentLine['link_id'] = $salesOrder->id;
            $paymentLine['link_type'] = SalesOrder::class;
            $paymentLine['payment_type_id'] = $paymentType->id;

            Payment::updateOrCreate([
                'link_id' => $salesOrder->id,
                'link_type' => SalesOrder::class,
            ], $paymentLine);
        }
    }

    private function updateSubmittedFulfillments(SalesOrder $order): void
    {
        $salesOrderFulfillments = $order->salesOrderFulfillments()
            ->where('status', SalesOrderFulfillment::STATUS_SUBMITTED)
            ->get();

        $salesOrderFulfillments->each(/**
         * @throws StarshipitException
         * @throws Throwable
         */ function (SalesOrderFulfillment $salesOrderFulfillment) {
            switch ($salesOrderFulfillment->fulfillment_type) {
                case SalesOrderFulfillment::TYPE_SHIPSTATION:
                    $shippingProvider = new ShipStation(IntegrationInstance::shipstation()->first());
                    $shippingProvider->submitOrder($salesOrderFulfillment->toShipStationOrder());
                    break;
                case SalesOrderFulfillment::TYPE_STARSHIPIT:
                    // Starshipit does not support ship by date so nothing to do here
                    //                    $shippingProvider = new StarShipIt(IntegrationInstance::starshipit()->first());
                    //                    $shippingProviderOrder = $salesOrderFulfillment->starshipitOrder;
                    //                    $shippingProvider->updateOrder($salesOrderFulfillment->toStarshipitOrder($shippingProviderOrder));
                    break;
            }
        });
    }

    public function markAllSalesOrderFulfillmentsAsSubmittedToSalesChannel(SalesOrder $salesOrder): void
    {
        $salesOrder->salesOrderFulfillments()->update([
            'submitted_to_sales_channel' => true,
        ]);
    }

    private function saveLines(Collection $salesOrderCollection, $data): Collection
    {
        if ($salesOrderCollection->isEmpty()) {
            return collect();
        }

        $productIds = $data
            ->flatMap(function ($salesOrderDto) {
                // Extracting sales order lines and plucking product IDs
                return $salesOrderDto->sales_order_lines->toCollection()->pluck('product_id');
            })
            ->reject(fn($id) => empty($id)) // Removing empty product IDs
            ->unique() // Optional: Remove duplicate product IDs
            ->values() // Resetting the keys of the collection
            ->all(); // Converting the collection to a plain PHP array

        $bundleProducts = $this->products->getForMatchingIdsAndType($productIds, Product::TYPE_BUNDLE);

        $taxLines = SalesOrderLineTaxLinesData::collection($salesOrderCollection->flatMap(function ($salesOrder) use ($data) {
            $salesOrderDto = $data->where('sales_order_number', $salesOrder['sales_order_number'])->first();
            return $salesOrderDto->sales_order_lines->toCollection()->map(function (SalesOrderLineData $salesOrderLineData) {
                return SalesOrderLineTaxLinesData::from([
                    'taxLines' => $salesOrderLineData->tax_data instanceof Optional ? null : $salesOrderLineData->tax_data?->taxLines
                ]);
            });
        }));

        $taxLines = $taxLines->reject(fn($taxLine) => !$taxLine->taxLines);
        $taxRates = $this->taxRateManager->getOrCreateTaxRatesFromTaxData($taxLines);

        $salesOrderLineCollection = collect();
        $salesOrderCollection->each(function ($salesOrder) use (
            $salesOrderLineCollection,
            $data,
            $bundleProducts,
            $taxRates
        ) {
            $salesOrderDto = $data->where('sales_order_number', $salesOrder['sales_order_number'])->first();
            $salesOrderDto->sales_order_lines->each(function (SalesOrderLineData $salesOrderLineData) use (
                $salesOrderDto,
                $salesOrder,
                $salesOrderLineCollection,
                $bundleProducts,
                $taxRates
            )
            {
                // TODO: Functionality needed in new update method
                $salesOrderLineData->sales_order_id = $salesOrder['id'];
                if (! $salesOrderLineData->tax_data instanceof Optional) {
                    if ($taxLine = $this->taxRateManager->combineTaxLinesData($salesOrderLineData->tax_data)) {
                        if ($taxRate = $taxRates
                            ->where('name', $taxLine->rateName)
                            ->where('rate', $taxLine->rate)
                            ->first()) {
                            $salesOrderLineData->tax_rate_id = $taxRate->id;
                            $salesOrderLineData->tax_rate = $taxRate->rate;
                            $salesOrderLineData->is_taxable = true;
                        }
                    }
                }

                // Check if the product is a bundle
                $bundleProduct = $salesOrderLineData->product_id
                    ? $bundleProducts->where('id', $salesOrderLineData->product_id)->first()
                    : null;

                /** @var Product $bundleProduct */
                if ($bundleProduct)
                {
                    // If the product is a bundle, we need to create a sales order line for each of its components
                    $bundleProduct->components->each(function (Product $component) use (
                        $salesOrderLineData,
                        $salesOrderLineCollection,
                        $salesOrderDto,
                        $salesOrder,
                        $bundleProduct,
                    ) {
                        $componentSalesOrderLineData = clone $salesOrderLineData;
                        $componentSalesOrderLineData->product_id = $component->id;
                        $componentSalesOrderLineData->quantity = $salesOrderLineData->quantity * $component->pivot->quantity;
                        $componentSalesOrderLineData->amount = $salesOrderLineData->amount * $component->getBundlePriceProration($bundleProduct);
                        $componentSalesOrderLineData->bundle_id = $bundleProduct->id;
                        $componentSalesOrderLineData->bundle_quantity_cache = $salesOrderLineData->quantity;
                        $salesOrderLineCollection->add(SalesOrderLineData::from(new SalesOrderLine($componentSalesOrderLineData->toArray())));
                    });
                } else {
                    $salesOrderLineData->bundle_id = null;
                    $salesOrderLineData->bundle_quantity_cache = null;
                    $salesOrderLineCollection->add(SalesOrderLineData::from(new SalesOrderLine($salesOrderLineData->toArray())));
                }

            });
        });

        $this->saveTaxTotals($salesOrderLineCollection);

        $salesOrderLineCollection = app(SalesOrderLineRepository::class)->save($salesOrderLineCollection,
            SalesOrderLine::class);

        $salesOrderLineFinancialCollection = $salesOrderLineCollection->map(function ($line) {
            return SalesOrderLineFinancialDto::from([
                'sales_order_line_id' => $line['id']
            ]);
        });

        app(SalesOrderLineFinancialsRepository::class)->save($salesOrderLineFinancialCollection,
            SalesOrderLineFinancial::class);

        $this->salesOrderLineFinancials->invalidateForSalesOrderLineIds($salesOrderLineCollection->pluck('id')->toArray());

        return $salesOrderLineCollection;
    }

    /**
     * @param $data
     * @param  Collection  $salesOrderCollection
     * @return void
     */
    private function saveFinancialLines($data, Collection $salesOrderCollection): void
    {
        $financialLineCollection = $data->reject(fn($order) => $order->financial_lines instanceof Optional)
            ->map(function ($order) use ($salesOrderCollection) {
                return $order->financial_lines->toCollection()->map(function ($line) use (
                    $salesOrderCollection,
                    $order
                ) {
                    $line->sales_order_id = $salesOrderCollection->where('sales_order_number',
                        $order->sales_order_number)->first()['id'];
                    unset($line->sales_order_number); // No fillable on model
                    return FinancialLineData::from($line->toArray());
                });
            })->flatten();

        app(FinancialLineRepository::class)->save($financialLineCollection, FinancialLine::class);
    }

    /**
     * @param $data
     * @param  Collection  $salesOrderCollection
     * @return void
     */
    private function savePayments($data, Collection $salesOrderCollection): void
    {
        $paymentTypeCollection = app(PaymentTypeRepository::class)->getFromNames($data->pluck('payment.payment_type')->unique()->toArray());

        $paymentCollection = $data->reject(fn($order) => $order->payment instanceof Optional)
            ->map(function ($order) use ($salesOrderCollection, $paymentTypeCollection) {
                $paymentDto                  = $order->payment;
                $paymentDto->payment_type_id = $paymentTypeCollection->where('name',
                    $paymentDto->payment_type)->first()['id'];
                $paymentDto->link_id         = $salesOrderCollection->where('sales_order_number',
                    $order->sales_order_number)->first()['id'];
                $paymentDto->link_type       = SalesOrder::class;
                return PaymentDto::from(new Payment($paymentDto->toArray()));
            });

        app(PaymentRepository::class)->save($paymentCollection, Payment::class);
    }

    private function saveOrderLinks($data, Collection $salesOrderCollection): void
    {
        $salesChannelId = $data->pluck('sales_channel_id')->unique()->first();
        $orderLinks = $data
            ->filter(fn ($item) => !($item->order_link instanceof Optional))
            ->map(function (SalesOrderDto $salesOrderDto) {
                return [
                    'child_sales_order_number' => $salesOrderDto->sales_order_number,
                    'parent_sales_order_number' => $salesOrderDto->order_link->parent_order_number,
                ];
            });

        $parentOrderCollection = $this->getForSalesOrderNumbersInSalesChannel($salesChannelId, $orderLinks->pluck('parent_sales_order_number')->toArray())
            ->map(function (SalesOrder $salesOrder) {
                return [
                    'id' => $salesOrder->id,
                    'sales_order_number' => $salesOrder->sales_order_number,
                ];
            });

        $orderLinkCollection = collect();

        foreach ($orderLinks as $orderLink)
        {
            $childSalesOrder = $salesOrderCollection->where('sales_order_number', $orderLink['child_sales_order_number'])->first();
            $parentSalesOrder = $parentOrderCollection->where('sales_order_number', $orderLink['parent_sales_order_number'])->first();

            if ($childSalesOrder && $parentSalesOrder)
            {
                $orderLinkCollection->add(OrderLinkData::from([
                    'child_id' => $childSalesOrder['id'],
                    'child_type' => SalesOrder::class,
                    'parent_id' => $parentSalesOrder['id'],
                    'parent_type' => SalesOrder::class,
                    'link_type' => OrderLink::LINK_TYPE_RESEND,
                    'created_at' => now(),
                ]));
            }
        }

        $this->orderLinks->save($orderLinkCollection, OrderLink::class);
    }

    private function saveCustomer(Collection $data): Collection
    {
        $customerCollection = app(CustomerRepository::class)
            ->saveWithRelations(CustomerData::collection(
                $data
                    ->reject(fn($order) => $order->customer instanceof Optional)
                    ->pluck('customer')
            ));

        return $data->map(function (SalesOrderDto $salesOrderDto) use ($customerCollection) {
            $customer = $salesOrderDto->customer instanceof Optional
                ? null
                : $customerCollection->where('email', $salesOrderDto->customer?->email)
                    ->where('name', $salesOrderDto->customer?->name)
                    ->first();
            if ($customer) {
                $salesOrderDto->customer_id         = $customer['id'];
                $salesOrderDto->shipping_address_id = @$customer['default_shipping_address_id'];
                $salesOrderDto->billing_address_id  = @$customer['default_billing_address_id'];
            }
            return SalesOrderDto::from(new SalesOrder($salesOrderDto->toArray()));
        });
    }

    private function addNotes(SalesOrder $salesOrder, #[DataCollectionOf(NoteData::class)] DataCollection $notes): void
    {
        if ($notes->count() == 0)
        {
            return;
        }

        $notesData = NoteData::collection($notes->map(function (NoteData $noteData) use ($salesOrder) {
            $noteData->link_id = $salesOrder->id;
            $noteData->link_type = SalesOrder::class;
            return $noteData;
        }));

        $this->notes->create($notesData);
    }

    private function saveTaxTotals(Collection $data): void
    {
        $taxTotalData = $data->groupBy('sales_order_id')->map(function ($lines, $salesOrderId) {
            return [
                'id' => $salesOrderId,
                'tax_total'      => $lines->sum('tax_allocation'),
            ];
        });
        batch()->update(new SalesOrder(), $taxTotalData->toArray(), 'id');
    }

    public function getForSalesOrderNumbersInSalesChannel(int $salesChannelId, array $salesOrderNumbers): Collection
    {
        return SalesOrder::query()
            ->where('sales_channel_id', $salesChannelId)
            ->whereIn('sales_order_number', $salesOrderNumbers)
            ->get();
    }
}
