<?php

namespace App\Repositories;

use App\Helpers;
use App\Models\PurchaseOrderLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use Exception;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Spatie\QueryBuilder\AllowedFilter;
use Spatie\QueryBuilder\QueryBuilder;

class ReportingRepository
{
    /**
     * @throws Exception
     */
    public function getPurchaseSummaryByPeriod(string $period, ?int $trailing_days = null): EloquentCollection
    {
        $dateFormat = match ($period) {
            'day' => '%Y-%m-%d',
            'month' => '%Y-%m',
            'year' => '%Y',
            default => throw new Exception('Unknown period in getSummaryByPeriod()'),
        };

        $query = QueryBuilder::for(PurchaseOrderLine::class)
            ->selectRaw("DATE_FORMAT(CONVERT_TZ(purchase_order_date, 'UTC', '".Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE)."'), '".$dateFormat."') as purchase_order_date")
            ->selectRaw('SUM(`currency_rate` * `quantity` * (`amount`) - IF(`is_tax_included` = 1, `tax_allocation`, 0)) as total_purchase_volume')
            ->join('purchase_orders', 'purchase_orders.id', 'purchase_order_lines.purchase_order_id')
            ->groupBy(DB::raw("DATE_FORMAT(CONVERT_TZ(purchase_order_date, 'UTC', '".Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE)."'), '".$dateFormat."')"))
            ->allowedFilters([
                AllowedFilter::scope('start_date'),
                AllowedFilter::scope('end_date'),
                AllowedFilter::exact('supplier_id', 'supplier_products.supplier_id'),
            ]);

        if (Arr::has($_REQUEST, 'filter.supplier_id')) {
            $query
                ->join('products', 'products.id', 'purchase_order_lines.product_id')
                ->leftJoin('supplier_products', function (JoinClause $join) {
                    $join->on('supplier_products.product_id', 'purchase_order_lines.product_id');
                    $join->where('is_default', 1);
                });
        }

        if (! Arr::has($_REQUEST, 'filter.start_date') && ! Arr::has($_REQUEST, 'filter.end_date') && $trailing_days) {
            $query->trailingDays($trailing_days);
        }

        return $query->get();
    }
}
