<?php

namespace App\Repositories;

use App\Models\Amazon\FBARealTimeInventory;
use App\Models\Product;
use App\Models\ProductInventory;
use App\Models\PurchaseOrder;
use App\Models\Supplier;
use App\Models\Warehouse;
use Illuminate\Support\Collection;

class ProductInventoryRepository
{
    /**
     * ProductInventoryRepository constructor.
     */
    public function __construct(protected Product $product, protected bool $noFBA = false, protected bool $noSuppliers = false)
    {
    }

    private function getWarehouseFbaRealTimeData(): Collection
    {
        if ($this->noFBA) {
            return collect([]);
        }

        $productWarehouses = Warehouse::with(['integrationInstance.salesChannel'])
            ->where(function ($query) {
                $query->whereNull('supplier_id');
                if (! $this->noSuppliers) {
                    $query->orWhereIn('id', $this->getSupplierDropshipWarehouses());
                }
            })
            ->when($this->noFBA, function ($query) {
                return $query->where('type', '!=', Warehouse::TYPE_AMAZON_FBA);
            })
            ->archived(0)
            ->get(['id', 'name', 'supplier_id', 'type', 'integration_instance_id']);

        $integrationInstanceIds = $productWarehouses
            ->where('type', Warehouse::TYPE_AMAZON_FBA)
            ->pluck('integration_instance_id');

        return FBARealTimeInventory::with(['integrationInstance.warehouse'])
            ->whereHas('productListing', function ($query) {
                $query->where('product_id', $this->product->id);
            })
            ->whereIn('integration_instance_id', $integrationInstanceIds)
            ->get();
    }

    private function getSupplierDropshipWarehouses()
    {
        $suppliersDropshipWarehousesId = [];
        $this->product->suppliers->map(function (Supplier $supplier) use (&$suppliersDropshipWarehousesId) {
            $dropshipWarehouses = $supplier->warehouses->where('dropship_enabled', true);
            if ($dropshipWarehouses->isNotEmpty()) {
                $suppliersDropshipWarehousesId = array_merge($suppliersDropshipWarehousesId, $dropshipWarehouses->pluck('id')->toArray());
            }
        });

        return $suppliersDropshipWarehousesId;
    }

    public function total()
    {
        return $this->totalInbound() + $this->totalAvailable() + $this->totalStock() + $this->totalReserved() + $this->totalInTransit();
    }

    public function totalInbound()
    {
        return
            $this->product
                ->purchaseOrderLinesOpened
                ->sum('unreceived_quantity') +
            $this->getWarehouseFbaRealTimeData()
                ->sum('inventory_incoming');
    }

    public function totalStock()
    {
        return
            $this->product
                ->inWarehousesQuantity
                ->sum('inventory_total') +
            $this->getWarehouseFbaRealTimeData()
                ->sum('inventory_total');
    }

    public function totalReserved()
    {
        return
            $this->product
                ->inWarehousesReservedQuantity
                ->sum('inventory_reserved') +
            $this->getWarehouseFbaRealTimeData()
                ->sum('inventory_reserved');
    }

    public function totalAvailable()
    {
        return
            $this->product
                ->inWarehouseAvailableQuantity
                ->sum('inventory_available') +
            $this->getWarehouseFbaRealTimeData()
                ->sum('inventory_available');
    }

    public function totalInTransit()
    {
        return $this->product->inWarehouseTransitQuantity->sum('inventory_in_transit');
    }

    public function inbound(Warehouse $warehouse)
    {
        if ($warehouse->type == Warehouse::TYPE_AMAZON_FBA) {
            return $this->getWarehouseFbaRealTimeData()
                ->where('integration_instance_id', $warehouse->integration_instance_id)
                ->sum('inventory_incoming');
        }

        return $this->product->purchaseOrderLines()
            ->whereHas('purchaseOrder', function ($query) use ($warehouse) {
                $query->where('status', PurchaseOrder::STATUS_OPEN);
                $query->where('destination_warehouse_id', $warehouse->id);
            })
            ->sum('unreceived_quantity');
    }

    public function stock(Warehouse $warehouse)
    {
        if ($warehouse->type == Warehouse::TYPE_AMAZON_FBA) {
            return $this->getWarehouseFbaRealTimeData()
                ->where('integration_instance_id', $warehouse->integration_instance_id)
                ->sum('inventory_total');
        }
        $supplierInventory = $this->product->suppliersInventory->firstWhere('warehouse_id', $warehouse->id);

        return $warehouse->isSupplierWarehouse() ? ($supplierInventory->stock ?? 0) :
      $this->product->inWarehousesQuantity->where('warehouse_id', $warehouse->id)->sum('inventory_total');
    }

    public function reserved(Warehouse $warehouse)
    {
        if ($warehouse->type == Warehouse::TYPE_AMAZON_FBA) {
            return $this->getWarehouseFbaRealTimeData()
                ->where('integration_instance_id', $warehouse->integration_instance_id)
                ->sum('inventory_reserved');
        }

        return $this->product->inWarehousesReservedQuantity->where('warehouse_id', $warehouse->id)->sum('inventory_reserved');
    }

    public function available(Warehouse $warehouse)
    {
        if ($warehouse->type == Warehouse::TYPE_AMAZON_FBA) {
            return $this->getWarehouseFbaRealTimeData()
                ->where('integration_instance_id', $warehouse->integration_instance_id)
                ->sum('inventory_available');
        }
        $supplierInventory = $this->product->suppliersInventory->firstWhere('warehouse_id', $warehouse->id);

        return $warehouse->isSupplierWarehouse() ? ($supplierInventory->stock ?? 0) :
      $this->product->inWarehouseAvailableQuantity->where('warehouse_id', $warehouse->id)->sum('inventory_available');
    }

    public function inTransit(Warehouse $warehouse)
    {
        if ($warehouse->type == Warehouse::TYPE_AMAZON_FBA) {
            return 0;
        }

        return $this->product->inWarehouseTransitQuantity->where('warehouse_id', $warehouse->id)->sum('inventory_in_transit');
    }

    public function deleteForProductExceptWarehouseIds(array $exceptWarehouseIds): void
    {
        $query = ProductInventory::where('product_id', $this->product->id);

        if (! empty($exceptWarehouseIds)) {
            $query->whereNotIn('warehouse_id', $exceptWarehouseIds);
        }
        $query->delete();
    }
}
