<?php

namespace App\Repositories;

use App\Helpers;
use App\Jobs\InventorySnapshotJob;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\InventorySnapshot;
use App\Models\Setting;
use App\Notifications\MonitoringMessage;
use Carbon\Carbon;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\LazyCollection;
use Illuminate\Support\Str;
use PDO;
use Throwable;

class InventorySnapshotRepository
{
    public function purgeInvalidSnapshots(array $productIds = []): void
    {
        $query = InventorySnapshot::query()
            ->where('is_cache_valid', 0);

        if (! empty($productIds)) {
            $query->whereIn('product_id', $productIds);
        }

        $query->delete();
    }

    public function sanitizeValidSnapshots(array|Arrayable $productIds = []): void
    {
        InventorySnapshot::query()
            ->whereIn('product_id', $productIds)
            ->whereIn('id', function ($query) {
                $query->select('id')
                    ->from('inventory_snapshots as is1')
                    ->whereIn('product_id', function ($query) {
                        $query->select('product_id')
                            ->from('inventory_snapshots')
                            ->where('is_cache_valid', 0)
                            ->groupBy('product_id');
                    })
                    ->where('date', '>', function ($query) {
                        $query->select(DB::raw('MIN(date)'))
                            ->from('inventory_snapshots')
                            ->where('is_cache_valid', 0)
                            ->whereColumn('product_id', 'is1.product_id');
                    })
                    ->where('is_cache_valid', 1);
            })
            ->update(['is_cache_valid' => 0]);
    }

    public function wasProductInStockBeforeDate(int $productId, Carbon $date): bool
    {
        return InventorySnapshot::query()
            ->select('inventory_in_stock')
            ->whereDate('date', '<', $date)
            ->where('product_id', $productId)
            ->latest('date')
            ->first()->inventory_in_stock ?? false;
    }

    public function getProductSnapshotsOnOrAfterDate(int $productId, Carbon $date): EloquentCollection
    {
        return InventorySnapshot::query()
            ->where('product_id', $productId)
            ->whereDate('date', '>=', $date)
            ->get();
    }

    public function areThereInvalidCaches(): bool
    {
        return InventorySnapshot::query()->where('is_cache_valid', 0)->count();
    }

    public function invalidateProductDate(int $product_id, Carbon $date): void
    {
        InventorySnapshot::query()
            ->where('product_id', $product_id)
            ->whereDate('date', '>=', $date)
            ->update(['is_cache_valid' => 0]);

        dispatch(new InventorySnapshotJob());
    }

    /*
     * Shape of expected array is
     * [
     *  [3983] => '2020-01-01 07:00:00',
     *  [5489] => '2020-01-09 07:00:00',
     * ]
     *
     * MODEL_BYPASS: App\Models\InventorySnapshot
     */
    /**
     * @throws Throwable
     */
    public function invalidateProductDates(array $productDates): void
    {
        if (empty($productDates)) {
            return;
        }

        $temporaryTableName = 'temporary_product_dates_'.str_replace('-', '_', Str::uuid());

        $createTempTableQuery = <<<SQL
        CREATE TABLE IF NOT EXISTS $temporaryTableName (
                `product_id` bigint(20) unsigned,
                `date` datetime,
                PRIMARY KEY (`product_id`, `date`)
            ) ENGINE=InnoDB DEFAULT COLLATE utf8mb4_unicode_ci  
        SQL;

        DB::statement($createTempTableQuery);

        $insertTempTableQuery = "INSERT INTO $temporaryTableName VALUES ";
        foreach ($productDates as $product_id => $date) {
            $insertTempTableQuery .= '('.$product_id.', "'.$date.'"),';
        }
        $insertTempTableQuery = trim($insertTempTableQuery, ',');

        try {
            DB::statement($insertTempTableQuery);
        } catch (Throwable $e) {
            Notification::route('slack', config('slack.debugging'))->notify(new MonitoringMessage('InventorySnapshotRepository::invalidateProductDates() failed to insert into temporary table.  Query is: '.$insertTempTableQuery));
            throw $e;
        }

        $updateQuery = <<<SQL
        UPDATE inventory_snapshots AS isnap
        INNER JOIN $temporaryTableName AS tpd
            ON tpd.product_id = isnap.product_id
        SET is_cache_valid = 0
        WHERE
            isnap.date >= tpd.date
        SQL;

        DB::statement($updateQuery);

        DB::statement("DROP TABLE $temporaryTableName");

        dispatch(new InventorySnapshotJob());
    }

    public function getProductIdsWithInvalidCache(?array $productIds): LazyCollection
    {
        $inventorySnapshots = InventorySnapshot::query()
            ->select('product_id')
            ->groupBy('product_id')
            ->where('is_cache_valid', 0);

        if (! empty($productIds)) {
            $inventorySnapshots->whereIn('product_id', $productIds);
        }

        return $inventorySnapshots->cursor()->pluck('product_id');
    }

    public function getMissingProductDays(array $productIds = []): array
    {
        $missingProductDays = [];

        /*
         * The date is in UTC. The UTC date needs to be converted to local to determine what local date it belongs to...
         * then strip the timestamp, and convert back to UTC
         */
        $query = InventoryMovement::query()
            ->select('inventory_movements.product_id')
            ->selectRaw(Helpers::getSqlUtcStartOfLocalDate('inventory_movements.inventory_movement_date').' as date')
            ->leftJoin('inventory_snapshots', function (JoinClause $join) {
                $join->on('inventory_snapshots.product_id', 'inventory_movements.product_id');
                $join->on(
                    'inventory_snapshots.date',
                    DB::raw("CONVERT_TZ(
                        DATE_FORMAT(
                            CONVERT_TZ(inventory_movements.inventory_movement_date, 'UTC', 
                            '".Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE)."'), '%Y-%m-%d'
                        ), '".Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE)."', 'UTC')"
                    )
                );
            })
            ->groupBy([
                DB::raw(Helpers::getSqlUtcStartOfLocalDate('inventory_movements.inventory_movement_date')),
                'inventory_movements.product_id',
            ])
            ->whereNull('inventory_snapshots.id');

        if (! empty($productIds)) {
            $query->whereIn('inventory_movements.product_id', $productIds);
        }

        $query->cursor()->each(function ($inventoryMovement) use (&$missingProductDays) {
            $missingProductDays[] = [
                'product_id' => $inventoryMovement->product_id,
                'date' => $inventoryMovement->date,
                'is_cache_valid' => 0,
            ];
        });

        return $missingProductDays;
    }

    public function seed(array $productIds = []): void
    {
        $this->saveBulk($this->getMissingProductDays($productIds));
    }

    public function saveBulk(array $data): void
    {
        foreach (array_chunk($data, 1000) as $smallerData) {
            InventorySnapshot::query()->upsert($smallerData, ['product_id', 'date']);
        }
    }

    /**
     * Builds validated inventory snapshot data from a combination of previous validated snapshots and new
     * data built from inventory movements
     */
    public function getNewlyValidatedData(array|Arrayable $productIds): array
    {
        $inventoryMovements = InventoryMovement::query()
            ->from('inventory_movements as im')
            ->select([
                'im.product_id',
            ])
            /*
             * getSqlUtcStartOfLocalDate allows us to store a date in UTC timezone in the database that represents the
             * start time of a User's day
             */
            ->selectRaw(Helpers::getSqlUtcStartOfLocalDate('im.inventory_movement_date').' as date')
            /*
             * We use SUM(SUM(... since the outer SUM is a MySQL window function, and the inner SUM is to SUM within
             * the group.  We also use CASE WHEN THEN ELSE syntax since there are multiple inventory movement
             * statuses that correspond to a single inventory snapshot record
             */
            ->selectRaw('SUM(SUM(CASE WHEN inventory_status = "'.InventoryMovement::INVENTORY_STATUS_ACTIVE.'" THEN quantity ELSE 0 END)) OVER (PARTITION BY product_id ORDER BY '.Helpers::getSqlUtcStartOfLocalDate('im.inventory_movement_date').') AS inventory_available')
            ->selectRaw('SUM(SUM(CASE WHEN inventory_status = "'.InventoryMovement::INVENTORY_STATUS_RESERVED.'" THEN quantity ELSE 0 END)) OVER (PARTITION BY product_id ORDER BY '.Helpers::getSqlUtcStartOfLocalDate('im.inventory_movement_date').') AS inventory_reserved')
            ->selectRaw('SUM(SUM(CASE WHEN inventory_status = "'.InventoryMovement::INVENTORY_STATUS_IN_TRANSIT.'" THEN quantity ELSE 0 END)) OVER (PARTITION BY product_id ORDER BY '.Helpers::getSqlUtcStartOfLocalDate('im.inventory_movement_date').') AS inventory_in_transit')
            // Only FIFO layers will result in inventory value
            ->selectRaw('SUM(IFNULL(SUM(quantity * (CASE WHEN fifo_layers.original_quantity = 0 THEN 0 ELSE fifo_layers.total_cost / fifo_layers.original_quantity END)), 0)) OVER (PARTITION BY product_id ORDER BY '.Helpers::getSqlUtcStartOfLocalDate('im.inventory_movement_date').') AS inventory_stock_value')
            ->selectRaw('1 as is_cache_valid')
            ->leftJoin('fifo_layers', function (JoinClause $join) {
                $join->on('im.layer_id', 'fifo_layers.id')
                    ->where('layer_type', FifoLayer::class);
            })
            ->where(function (Builder $query) {
                /*
                 * For improved performance, we only get inventory movements for dates after the last validated snapshot
                 * for a product.  We later add the validated snapshot values to each cumulative total
                 */
                $query->orWhere(DB::raw(Helpers::getSqlUtcStartOfLocalDate('im.inventory_movement_date')), '>', function ($query) {
                    $query
                        ->from('inventory_snapshots as is1')
                        ->select('date')
                        ->where('is_cache_valid', 1)
                        ->where('date', function ($query) {
                            $query->selectRaw('MAX(date)')
                                ->from('inventory_snapshots')
                                ->where('is_cache_valid', 1)
                                ->whereColumn('product_id', 'is1.product_id')
                                ->groupBy('product_id');
                        })
                        ->whereColumn('is1.product_id', 'im.product_id');
                });
                // If no validated snapshot exists, we get all inventory movements for a product
                $query->orWhereNotExists(function ($query) {
                    $query
                        ->from('inventory_snapshots as is1')
                        ->where('is_cache_valid', 1)
                        ->whereColumn('is1.product_id', 'im.product_id');
                });
            })
            ->groupBy([
                DB::raw(Helpers::getSqlUtcStartOfLocalDate('im.inventory_movement_date')),
                'im.product_id',
            ])
            ->orderBy('inventory_movement_date');

        if (! empty($productIds)) {
            $inventoryMovements->whereIn('im.product_id', $productIds);
        }

        /*
        |--------------------------------------------------------------------------
        | These temporary tables add the latest validated snapshot data to each
        | cumulative total
        |--------------------------------------------------------------------------
        */

        $createTemporaryTableQuery = '
        CREATE TEMPORARY TABLE temp_snapshot_data AS ('.
            str_replace('\\', '\\\\', $inventoryMovements->toSqlWithBindings()).'
        )';

        DB::statement($createTemporaryTableQuery);

        $addIndexesQuery = <<<'SQL'
        ALTER TABLE temp_snapshot_data ADD PRIMARY KEY (`product_id`, `date`)
        SQL;

        DB::statement($addIndexesQuery);

        $latestValidatedSnapshotsQuery = InventorySnapshot::query()
            ->from('inventory_snapshots as is1')
            ->select([
                'product_id',
                'is_cache_valid',
                'date',
                'inventory_available',
                'inventory_reserved',
                'inventory_in_transit',
                'inventory_stock_value',
            ])
            ->where('is_cache_valid', 1)
            ->where('date', function ($query) {
                $query->selectRaw('MAX(date)')
                    ->from('inventory_snapshots')
                    ->where('is_cache_valid', 1)
                    ->whereColumn('product_id', 'is1.product_id')
                    ->groupBy('product_id');
            });

        if (! empty($productIds)) {
            $latestValidatedSnapshotsQuery->whereIn('product_id', $productIds);
        }

        $createLatestValidatedSnapshotsTemporaryTable = '
        CREATE TEMPORARY TABLE temp_latest_validated_snapshots AS ('.
            $latestValidatedSnapshotsQuery->toSqlWithBindings().'
        )
        ';

        DB::statement($createLatestValidatedSnapshotsTemporaryTable);

        $addIndexesQuery = <<<'SQL'
        ALTER TABLE temp_latest_validated_snapshots ADD PRIMARY KEY (`product_id`, `date`)
        SQL;

        DB::statement($addIndexesQuery);

        $updateSnapshotsQuery = <<<'SQL'
        UPDATE temp_snapshot_data AS tsd
        LEFT JOIN temp_latest_validated_snapshots AS tlvs
            ON tlvs.product_id = tsd.product_id
        SET tsd.inventory_available = tsd.inventory_available + IFNULL(tlvs.inventory_available, 0),
            tsd.inventory_reserved = tsd.inventory_reserved + IFNULL(tlvs.inventory_reserved, 0),
            tsd.inventory_in_transit = tsd.inventory_in_transit + IFNULL(tlvs.inventory_in_transit, 0),
            tsd.inventory_stock_value = tsd.inventory_stock_value + IFNULL(tlvs.inventory_stock_value, 0)
        SQL;

        DB::statement($updateSnapshotsQuery);

        $pdo = DB::getPdo();
        $statement = $pdo->prepare('SELECT * FROM temp_snapshot_data');
        $statement->setFetchMode(PDO::FETCH_ASSOC);
        $statement->execute();

        $data = $statement->fetchAll();

        // We must drop temporary tables since they persist in between chunks of products
        DB::statement('DROP TABLE temp_snapshot_data');
        DB::statement('DROP TABLE temp_latest_validated_snapshots');

        return $data;
    }
}
