<?php

namespace App\Repositories;

use App\Collections\InventoryMovementCollection;
use App\DTO\BulkImportResponseDto;
use App\DTO\InventoryMovementDto;
use App\Helpers;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\Warehouse;
use App\Models\WarehouseTransferShipmentLine;
use App\Models\WarehouseTransferShipmentReceiptLine;
use App\Services\InventoryManagement\InventoryEvent;
use Carbon\Carbon;
use Config;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Managers\AmazonLedgerManager;
use Modules\Amazon\Managers\AmazonReportManager;
use Spatie\QueryBuilder\AllowedFilter;
use Spatie\QueryBuilder\QueryBuilder;
use Str;
use Throwable;

class InventoryMovementRepository
{
    public function getProductsWithMovementsInPastDays(int $days): Collection
    {
        return InventoryMovement::query()
            ->select('product_id')
            ->whereDate('inventory_movement_date', '>=', Helpers::utcStartOfLocalDate(Carbon::now()->subDays($days)))
            ->pluck('product_id');
    }

    public function getTotalAmount(EloquentCollection $inventoryMovements): float
    {
        $amount = 0;

        $inventoryMovements->each(function (InventoryMovement $inventoryMovement) use (&$amount) {
            $amount += $inventoryMovement->quantity * $inventoryMovement->fifo_layer->avg_cost;
        });

        return abs($amount);
    }

    /**
     * @param  string  $inventoryStatus see InventoryMovement::INVENTORY_STATUS
     */
    public function getTallies(
        int $warehouse_id,
        string $inventoryStatus,
        Product $product
    ): array {
        $runningTally = 0;

        $tallies = [];

        $inventoryMovements = InventoryMovement::query()
            ->where('product_id', $product->id)
            ->where('warehouse_id', $warehouse_id)
            ->where('inventory_status', $inventoryStatus)
            ->orderBy('inventory_movement_date', 'ASC')
            ->orderBy('id', 'DESC')
            ->get();

        $inventoryMovements->each(function (InventoryMovement $inventoryMovement) use (&$runningTally, &$tallies) {
            $runningTally += $inventoryMovement->quantity;
            //print '(' . $inventoryMovement->id . ', qty ' . $inventoryMovement->quantity . '): ' . $inventoryMovement->inventory_movement_date . "\n";
            //print 'tally: ' . $runningTally . "\n";
            $tallies[$inventoryMovement->id] = $runningTally;
        });

        return $tallies;
    }

    public function getReservationsPriorToStartDate(array $ids = []): EloquentCollection
    {
        $query = InventoryMovement::query()
            ->where('link_type', SalesOrderLine::class)
            ->where('type', InventoryMovement::TYPE_SALE)
            ->whereDate('inventory_movement_date', '<', Helpers::setting(Setting::KEY_INVENTORY_START_DATE));

        if (! empty($ids)) {
            $query->whereHas('salesOrderLine', function ($query) use ($ids) {
                $query->whereIn('sales_order_id', $ids);
            });
        }

        return $query->get();
    }

    public function getReservationReleasesPriorToStartDate(array $ids = []): EloquentCollection
    {
        $query = InventoryMovement::query()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('quantity', '<', 0)
            ->where('link_type', SalesOrderFulfillmentLine::class)
            ->where('type', InventoryMovement::TYPE_SALE)
            ->whereDate('inventory_movement_date', '<', Helpers::setting(Setting::KEY_INVENTORY_START_DATE));

        if (! empty($ids)) {
            $query->whereHas('salesOrderFulfillmentLine', function ($query) use ($ids) {
                $query->whereHas('salesOrderLine', function ($query) use ($ids) {
                    $query->whereIn('sales_order_id', $ids);
                });
            });
        }

        return $query->get();
    }

    /**
     * @throws Throwable
     */
    public function save(InventoryMovementDto $data): InventoryMovement
    {
        /** @var InventoryMovement $inventoryMovement */
        $inventoryMovement = InventoryMovement::query()->updateOrCreate(['id' => $data->id], $data->toArray());

        return $inventoryMovement;
    }

    /**
     * @param  InventoryMovementCollection  $data
     * @return void
     */
    public function saveBulk(InventoryMovementCollection $data): void
    {

        $uniqueByColumns = [];

        if ($data->first()?->id) {
            $uniqueByColumns[] = 'id';
        } else {
            $data = $data->map(function ($_data) {
                $_data->uid = Str::uuid();
                return $_data;
            })->values();

            $uniqueByColumns[] = 'uid';
        }

        InventoryMovement::query()->upsert(
            $data->toArray(),
            $uniqueByColumns
        );
    }

    public function getActiveMovementUsagesForFifoLayerIds(array $fifoLayerIds): EloquentCollection
    {
        return InventoryMovement::query()
            ->whereIn('layer_id', $fifoLayerIds)
            ->where('layer_type', FifoLayer::class)
            ->where('quantity', '<', 0)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->orderBy('inventory_movement_date')
            ->get();
    }

    public function getReservedMovementUsagesForFifoLayerIds(array $fifoLayerIds): EloquentCollection
    {
        return InventoryMovement::query()
            ->where('layer_type', FifoLayer::class)
            ->whereIn('layer_id', $fifoLayerIds)
            ->where('quantity', '>', 0)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('link_type', SalesOrderLine::class)
            ->orderBy('inventory_movement_date')
            ->get();
    }

    public function deleteAllInventoryMovementsForEvent(InventoryEvent $event)
    {
        $event->inventoryMovements()->each(function (InventoryMovement $inventoryMovement) {
            $inventoryMovement->delete();
        });
    }

    public function getSalesOrderLinesFromUsages(EloquentCollection $usages): EloquentCollection
    {
        $salesOrderLineIds = $usages
            ->where('link_type', SalesOrderLine::class)
            ->pluck('link_id')->toArray();

        return SalesOrderLine::with(['product.defaultSupplierProduct', 'salesOrder'])
            ->whereIn('id', $salesOrderLineIds)
            ->get();
    }

    /**
     * @throws Exception
     */
    public function getValuationSummaryByInventoryStatus(): array
    {
        // Define the select parts for each inventory status to include both quantity and valuation
        $statusSelectParts = [];
        foreach (InventoryMovement::INVENTORY_STATUS as $status) {
            $statusSelectParts[] = "ROUND(COALESCE(SUM(CASE WHEN inventory_status = '$status' THEN inventory_movements.quantity ELSE 0 END), 0)) as {$status}_quantity";
            $statusSelectParts[] = "ROUND(COALESCE(SUM(CASE WHEN inventory_status = '$status' THEN inventory_movements.quantity * (fifo_layers.total_cost / fifo_layers.original_quantity) ELSE 0 END), 0), 2) as {$status}_valuation";
        }

        // Add select parts for total quantity and total cost
        $selectParts = array_merge($statusSelectParts, [
            "ROUND(SUM(inventory_movements.quantity)) as total_quantity",
            "ROUND(SUM(inventory_movements.quantity * (fifo_layers.total_cost / fifo_layers.original_quantity)), 2) as total_valuation"
        ]);

        // Build the query string
        $queryString = implode(', ', $selectParts);

        // Execute the query
        $result = QueryBuilder::for(InventoryMovement::class)
            ->selectRaw($queryString)
            ->allowedFilters([
                AllowedFilter::exact('warehouse_id'),
                AllowedFilter::scope('before_inventory_date')
            ])
            ->join('fifo_layers', 'inventory_movements.layer_id', '=', 'fifo_layers.id')
            ->where('inventory_movements.layer_type', '=', FifoLayer::class)
            ->first();

        // Calculate the average cost
        $average_cost = $result->total_quantity > 0 ? $result->total_valuation / $result->total_quantity : 0;

        // Prepare the final totals array including status-specific totals
        $totals = [
            'total_quantity' => $result->total_quantity ?? 0.00,
            'total_valuation' => $result->total_valuation ?? 0.00,
            'average_cost' => $average_cost
        ];

        foreach (InventoryMovement::INVENTORY_STATUS as $status) {
            $totals["{$status}_quantity"] = $result->{"{$status}_quantity"};
            $totals["{$status}_valuation"] = $result->{"{$status}_valuation"};
        }

        if($warehouseId = @request()->all()['filter']['warehouse_id'])
        {
            $warehouse = Warehouse::findOrFail($warehouseId);
            if ($integrationInstance = $warehouse->integrationInstance)
            {
                if ($integrationInstance->isAmazonInstance() && $warehouse->type == Warehouse::TYPE_AMAZON_FBA) {
                    $integrationInstance = AmazonIntegrationInstance::findOrFail($integrationInstance->id);
                    $totals['reconciliation_date'] = (new AmazonLedgerManager($integrationInstance))->getLastReconciledDate()->toDateString();
                }
            }
        }

        return $totals;
    }

    /**
     * Get a summary of valuations by inventory status and product.
     *
     * This method retrieves a summary of inventory movements and valuations,
     * grouped by product and inventory status.
     *
     */
    public function getValuationSummaryByInventoryStatusAndProduct(): Collection
    {
        // Initial select columns from the products table
        $selectParts = ['products.id', 'products.sku', 'products.name'];

        // Dynamically create select parts for each inventory status
        foreach (InventoryMovement::INVENTORY_STATUS as $status) {
            $selectParts[] = $this->createInventoryStatusSelectPart($status);
        }

        $selectParts[] = "ROUND(SUM(inventory_movements.quantity)) as total_quantity";
        $selectParts[] = "ROUND(SUM(inventory_movements.quantity * (fifo_layers.total_cost / fifo_layers.original_quantity)), 2) as total_valuation";

        // Create the final query string by concatenating all parts
        $queryString = implode(', ', $selectParts);

        // Build and execute the query
        $results = QueryBuilder::for(InventoryMovement::class)
            ->selectRaw($queryString)
            ->allowedFilters([
                AllowedFilter::exact('warehouse_id'),
                AllowedFilter::scope('before_inventory_date')
            ])
            ->join('fifo_layers', 'inventory_movements.layer_id', '=', 'fifo_layers.id')
            ->where('inventory_movements.layer_type', '=', FifoLayer::class)
            ->join('products', 'fifo_layers.product_id', '=', 'products.id')
            ->groupBy('products.id')
            ->havingRaw('total_valuation > 0')
            ->orderBy('total_valuation', 'desc') // Order by total valuation descending
            ->get();

        // Calculate the average cost for each result
        return $results->map(function ($result) {
            $result->average_cost = $result->total_quantity > 0 ? $result->total_valuation / $result->total_quantity : 0;
            return $result;
        });
    }

    /**
     * Helper function to create a select part for a given inventory status.
     * This includes the calculation of total quantity and total cost per status.
     *
     * @param string $status The inventory status.
     * @return string The select part for the SQL query.
     */
    private function createInventoryStatusSelectPart(string $status): string
    {
        return "ROUND(COALESCE(SUM(CASE WHEN inventory_status = '$status' THEN inventory_movements.quantity ELSE 0 END), 0)) as {$status}_quantity, " .
            "ROUND(COALESCE(SUM(CASE WHEN inventory_status = '$status' THEN inventory_movements.quantity * (fifo_layers.total_cost / fifo_layers.original_quantity) ELSE 0 END), 0), 2) as {$status}_valuation";
    }

    /**
     * This method updates the references of inventory movements associated with the
     * given ids and type to the given reference.
     *
     * @param  array  $ids
     * @param  string  $type
     * @param  string  $reference
     * @return void
     */
    public function updateReferences(array $ids, string $type, string $reference): void{
        InventoryMovement::query()
            ->whereIn('link_id', $ids)
            ->where('link_type', $type)
            ->update(['reference' => $reference]);
    }

    /**
     * @throws Throwable
     */
    public function getCorrespondingReservationMovementForActiveSalesOrderLineMovement(InventoryMovement $inventoryMovement): ?InventoryMovement
    {
        throw_if($inventoryMovement->link_type !== SalesOrderLine::class, new Exception('The given inventory movement is not associated with a sales order line'));
        return InventoryMovement::query()
            ->where('link_id', $inventoryMovement->link_id)
            ->where('link_type', SalesOrderLine::class)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('layer_id', $inventoryMovement->layer_id)
            ->where('layer_type', $inventoryMovement->layer_type)
            ->where('quantity', abs($inventoryMovement->quantity))
            // Closest match by created_at timestamp
            ->orderByRaw('ABS(TIMESTAMPDIFF(SECOND, created_at, ?))', [$inventoryMovement->created_at])
            ->first();
    }

    /**
     * @throws Throwable
     */
    public function getCorrespondingFulfillmentMovementsForReservationSalesOrderLineMovement(InventoryMovement $inventoryMovement): Collection
    {
        throw_if($inventoryMovement->link_type !== SalesOrderLine::class, new Exception('The given inventory movement is not associated with a sales order line'));
        $salesOrderFulfillmentLines = $inventoryMovement->link->salesOrderFulfillmentLines;
        if (!$salesOrderFulfillmentLines) {
            return collect();
        }
        return InventoryMovement::query()
            ->whereIn('link_id', $salesOrderFulfillmentLines->pluck('id'))
            ->where('link_type', SalesOrderFulfillmentLine::class)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('layer_id', $inventoryMovement->layer_id)
            ->where('layer_type', FifoLayer::class)
            ->get();
    }

    /**
     * @throws Throwable
     */
    public function getCorrespondingFulfillmentMovementForReservationSalesOrderLineMovement(InventoryMovement $inventoryMovement): ?InventoryMovement
    {
        throw_if($inventoryMovement->link_type !== SalesOrderLine::class, new Exception('The given inventory movement is not associated with a sales order line'));
        $salesOrderFulfillmentLines = $inventoryMovement->link->salesOrderFulfillmentLines;
        if (!$salesOrderFulfillmentLines) {
            return null;
        }

        try {
            $inventoryMovement = InventoryMovement::query()
                ->whereIn('link_id', $salesOrderFulfillmentLines->pluck('id'))
                ->where('link_type', SalesOrderFulfillmentLine::class)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                ->where('layer_id', $inventoryMovement->layer_id)
                ->where('layer_type', FifoLayer::class)
                ->where('quantity', -abs($inventoryMovement->quantity))
                ->sole();
        } catch (ModelNotFoundException) {
            return null;
        }

        return $inventoryMovement;
    }

    public function getCorrespondingReservationMovementForFulfillmentLineMovement(SalesOrderFulfillmentLine $salesOrderFulfillmentLine): InventoryMovement
    {
        return InventoryMovement::query()
            ->where('link_id', $salesOrderFulfillmentLine->salesOrderLine->id)
            ->where('link_type', SalesOrderLine::class)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('quantity', abs($salesOrderFulfillmentLine->quantity))
            // TODO: Consideration: Why not filter by layer_id?
            ->where('layer_type', FifoLayer::class)
            ->sole();
    }

    public function getCorrespondingAddInTransitMovementForActiveLineMovement(InventoryMovement $inventoryMovement): InventoryMovement
    {
        return InventoryMovement::query()
            ->where('link_id', $inventoryMovement->link_id)
            ->where('link_type', $inventoryMovement->link_type)
            ->where('type', InventoryMovement::TYPE_TRANSFER)
            ->where('quantity', abs($inventoryMovement->quantity))
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_IN_TRANSIT)
            ->where('layer_id', $inventoryMovement->layer_id)
            ->where('layer_type', $inventoryMovement->layer_type)
            ->sole();
    }

    /**
     * @throws Throwable
     */
    public function getCorrespondingDeductInTransitMovementForAddInTransitLineMovement(InventoryMovement $inventoryMovement): Collection
    {
        throw_if($inventoryMovement->link_type !== WarehouseTransferShipmentLine::class, new Exception('The given inventory movement is not associated with a warehouse transfer shipment line'));
        /** @var WarehouseTransferShipmentLine $warehouseTransferShipmentLine **/
        $warehouseTransferShipmentLine = $inventoryMovement->link;
        $warehouseTransferShipmentReceiptLines = $warehouseTransferShipmentLine->receiptLines;
        return InventoryMovement::query()
            ->whereIn('link_id', $warehouseTransferShipmentReceiptLines->pluck('id'))
            ->where('link_type', WarehouseTransferShipmentReceiptLine::class)
            ->where('type', InventoryMovement::TYPE_TRANSFER)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_IN_TRANSIT)
            ->where('layer_id', $inventoryMovement->layer_id)
            ->where('layer_type', $inventoryMovement->layer_type)
            ->get();
    }

    public function getActiveInventoryMovementsForUnfulfilledSalesOrderLinesForFifoLayer(FifoLayer $fifoLayer): Collection
    {
        return $fifoLayer->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('type', InventoryMovement::TYPE_SALE)
            ->whereHasMorph('link', SalesOrderLine::class, function ($query) {
                $query->whereHas('salesOrder', function ($query) {
                    $query->where('order_status', '!=', SalesOrder::STATUS_CLOSED);
                });
                $query->whereDoesntHave('salesOrderFulfillmentLines');
            })
            ->get();
    }

    public function findSalesOrderLineMatchUsingReference(InventoryMovement $inventoryMovement): SalesOrderLine
    {
        return SalesOrderLine::with([])
            ->where('product_id', $inventoryMovement->product_id)
            ->where('warehouse_id', $inventoryMovement->warehouse_id)
            ->where('quantity', abs($inventoryMovement->quantity))
            ->whereHas('salesOrder', function ($query) use ($inventoryMovement) {
                $query->where('sales_order_number', $inventoryMovement->reference);
            })
            ->sole();
    }
}
