<?php

namespace App\Repositories;

use App\Exceptions\InventoryMovementTypeException;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Warehouse;
use App\Services\StockTake\OpenStockTakeException;
use Illuminate\Database\Eloquent\Builder;

class InventoryHealthRepository
{
    public function salesOrderLinesMissingMovementsQuery(string $inventoryStatus, ?SalesOrderLine $salesOrderLine = null): Builder
    {
        $query = SalesOrderLine::with(['salesOrder', 'product'])
            ->whereHas('salesOrder', function (Builder $query) {
                $query->where('order_status', SalesOrder::STATUS_CLOSED); // Closed for now but may need to include open
                $query->whereNull('canceled_at');
                $query->where(function (Builder $query) {
                    $query->where('is_fba', false);
                    $query->orWhereNull('is_fba');
                });
                $query->where('fulfillment_status', '!=', SalesOrder::FULFILLMENT_STATUS_OVER_FULFILLED);
            })
            ->whereHas('warehouse', function (Builder $query) {
                $query->whereNot('type', Warehouse::TYPE_SUPPLIER);
                $query->whereNot('type', Warehouse::TYPE_AMAZON_FBA);
            })
            ->where('quantity', '>', 0)
            ->whereRaw('quantity - externally_fulfilled_quantity > 0')
            ->whereDoesntHave('inventoryMovements', function (Builder $query) use ($inventoryStatus) {
                $query->where('inventory_status', $inventoryStatus);
            });

        if ($salesOrderLine) {
            $query->where('id', $salesOrderLine->id);
        }

        return $query;
    }

    public function salesOrderLinesMissingFulfillmentMovementsQuery(?SalesOrderLine $salesOrderLine = null): Builder
    {
        $query = SalesOrderLine::with(['salesOrder', 'product', 'salesOrderFulfillmentLines.salesOrderFulfillment'])
            ->whereHas('salesOrder', function (Builder $query) {
                $query->where('order_status', SalesOrder::STATUS_CLOSED); // Closed for now but may need to include open
                $query->where(function (Builder $query) {
                    $query->where('is_fba', false);
                    $query->orWhereNull('is_fba');
                });
                $query->where('fulfillment_status', '!=', SalesOrder::FULFILLMENT_STATUS_OVER_FULFILLED);
            })
            ->whereHas('warehouse', function (Builder $query) {
                $query->whereNot('type', Warehouse::TYPE_SUPPLIER);
                $query->whereNot('type', Warehouse::TYPE_AMAZON_FBA);
            })
            ->where('quantity', '>', 0)
            ->whereRaw('quantity - externally_fulfilled_quantity > 0')
            ->whereHas('salesOrderFulfillmentLines', function (Builder $query) {
                $query->whereDoesntHave('inventoryMovements');
            });


        if ($salesOrderLine) {
            $query->where('id', $salesOrderLine->id);
        }

        return $query;
    }

    public function salesOrderLinesReservationButNotActiveMovementsQuery(?SalesOrderLine $salesOrderLine = null): Builder
    {
        $query = SalesOrderLine::with(['salesOrder', 'product'])
            ->whereHas('salesOrder', function (Builder $query) {
                $query->where('order_status', SalesOrder::STATUS_CLOSED); // Closed for now but may need to include open
                $query->whereNull('canceled_at');
                $query->where(function (Builder $query) {
                    $query->where('is_fba', false);
                    $query->orWhereNull('is_fba');
                });
                $query->where('fulfillment_status', '!=', SalesOrder::FULFILLMENT_STATUS_OVER_FULFILLED);
            })
            ->whereHas('warehouse', function (Builder $query) {
                $query->whereNot('type', Warehouse::TYPE_SUPPLIER);
                $query->whereNot('type', Warehouse::TYPE_AMAZON_FBA);
            })
            ->where('quantity', '>', 0)
            ->whereRaw('quantity - externally_fulfilled_quantity > 0')
            ->whereHas('inventoryMovements', function (Builder $query) {
                $query->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE);
            })
            ->whereDoesntHave('inventoryMovements', function (Builder $query) {
                $query->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED);
            });

        if ($salesOrderLine) {
            $query->where('id', $salesOrderLine->id);
        }

        return $query;
    }

    public function fulfillmentReleaseInventoryMovementsMissingFulfillmentLinesQuery(): Builder
    {
        return InventoryMovement::with(['product', 'warehouse'])
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('link_type', SalesOrderFulfillmentLine::class)
            ->whereNotNull('link_id')
            ->whereDoesntHave('link');
    }

    public function orphanSalesOrderLineReservationMovements(): Builder
    {
        return InventoryMovement::with(['product', 'warehouse'])
            ->where('link_type', SalesOrderLine::class)
//            ->whereHas('product', function (Builder $query) {
//                $query->whereIn('type', Product::TYPES_WITH_INVENTORY);
//            })
            ->whereNotNull('link_id')
            ->whereDoesntHave('link');
    }

    public function inventoryMovementsForProductsWithInvalidTypeQuery(): Builder
    {
        return InventoryMovement::with(['product', 'warehouse'])
            ->whereHas('product', function (Builder $query) {
                $query->whereNotIn('type', Product::TYPES_WITH_INVENTORY);
            });
    }

    /**
     * @throws OpenStockTakeException
     * @throws InventoryMovementTypeException
     */
    public function createActiveInventoryMovementFromReservationMovement(
        InventoryMovement $reservationMovement, ?int $overridingQuantity = null, ?FifoLayer $overridingFifoLayer = null
    ): InventoryMovement
    {
        $activeInventoryMovement = $reservationMovement->replicate();
        $activeInventoryMovement->quantity = abs($overridingQuantity ?? $reservationMovement->quantity);
        if($overridingFifoLayer) {
            $activeInventoryMovement->layer()->associate($overridingFifoLayer);
        }
        $activeInventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
        $activeInventoryMovement->save();

        return $activeInventoryMovement;
    }

    /**
     * @throws OpenStockTakeException
     * @throws InventoryMovementTypeException
     */
    public function createReservationInventoryMovementFromActiveMovement(
        InventoryMovement $activeMovement, ?int $overridingQuantity = null, ?FifoLayer $overridingFifoLayer = null
    ): InventoryMovement
    {
        $reservationMovement = $activeMovement->replicate();
        $reservationMovement->quantity = abs($overridingQuantity ?? $activeMovement->quantity);
        if($overridingFifoLayer) {
            $reservationMovement->layer()->associate($overridingFifoLayer);
        }
        $reservationMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_RESERVED;
        $reservationMovement->save();

        return $reservationMovement;
    }

    public function overSubscribedFifoLayersQuery(): Builder
    {
        return FifoLayer::with(['inventoryMovements', 'product', 'warehouse'])
            ->whereHas('inventoryMovements', function (Builder $query) {
                $query->selectRaw('SUM(quantity) as total_quantity')
                    ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                    ->having('total_quantity', '<', 0);
            });
    }

    // TODO: What about when the layers exist but are not accurate?  Not worth it to fix right now since we'll be deprecating sales order line layers
    public function salesOrderLinesWithInvalidLayerCacheQuery(): Builder
    {
        return SalesOrderLine::with([])
            ->whereHas('inventoryMovements', function (Builder $query) {
                $query->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                    ->where('layer_type', FifoLayer::class);
            })
            ->whereDoesntHave('salesOrderLineLayers');
    }

    public function externallyFulfilledSaleOrderLinesWithInventoryMovementsQuery(?SalesOrderLine $salesOrderLine = null): Builder
    {
        $query = SalesOrderLine::with(['salesOrder', 'product'])
            ->whereRaw('externally_fulfilled_quantity - quantity >= 0')
            ->whereHas('inventoryMovements');

        if ($salesOrderLine) {
            $query->where('id', $salesOrderLine->id);
        }

        return $query;
    }

    public function getNonInventoryProductsWithInventoryMovementsQuery(): Builder
    {
        return Product::with(['fifoLayers.inventoryMovements', 'salesOrderLines'])
            ->withCount(['fifoLayers', 'backorderQueues', 'inventoryMovements'])
            ->whereNotIn('type', Product::TYPES_WITH_INVENTORY)
            ->whereHas('inventoryMovements');
    }
}
