<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 10/21/20
 * Time: 11:59 AM.
 */

namespace App\Repositories;

use App\Helpers;
use App\Models\InventoryForecastItem;
use App\Models\InventoryForecast;
use App\Models\PurchaseOrderLine;
use App\Services\InventoryForecasting\ForecastManager;
use App\Services\PurchaseOrder\PurchaseOrderBuilder\PurchaseOrderBuilderRepository;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redis;

/**
 * Class InventoryForecastCacheRepository.
 */
class InventoryForecastRepository extends PurchaseOrderBuilderRepository
{

    /**
     * Clears the entire cache.
     */
    public function clear(): void
    {
        InventoryForecastItem::with([])->delete();
        InventoryForecast::with([])->delete();
    }

    /**
     * @return LengthAwarePaginator
     */
    public function getForecastsForToday(): LengthAwarePaginator
    {
        $timezone = Helpers::getAppTimezone() ?: 'UTC';


        $linkedPurchaseOrderLinesQuery = $this->getLinkedPurchaseOrderLinesQuery();

        return InventoryForecast::with([
            'supplier.defaultPricingTier',
            'inventoryForecastItems.product.primaryImage',
            'inventoryForecastItems.product.purchaseOrderLinesOpened',
            'inventoryForecastItems.warehouse.integrationInstance.salesChannel',
            'inventoryForecastItems' => function($query){
                $query->with([
                    'product.primaryImage',
                    'product.purchaseOrderLinesOpened',
                    'warehouse.integrationInstance.salesChannel',
                    'purchaseOrderLines'
                ])->leftJoinSub(
                    $this->getInventoryQuery(),
                    'inventory',
                    'inventory.product_id',
                    'inventory_forecast_items.product_id'
                )->select(
                    'inventory_forecast_items.*',
                    'inventory.inventory_available as inventory_available',
                    'inventory.inventory_inbound as inventory_inbound',
                    'inventory.inventory_in_transit as inventory_in_transit',
                );
            }
        ])
            ->whereHas('inventoryForecastItems', function (Builder $query) use ($linkedPurchaseOrderLinesQuery) {
                $query->leftJoinSub($linkedPurchaseOrderLinesQuery, 'linked_purchase_order_lines', 'linked_purchase_order_lines.id', 'inventory_forecast_items.id')
                    ->whereColumn('linked_purchase_order_lines.total_quantity_purchased', '<', 'inventory_forecast_items.suggested_purchase_quantity')
                    ->orWhereDoesntHave('purchaseOrderLines');
            })
            ->whereDate('forecast_date', now($timezone)->format('Y-m-d'))->paginate(25);
    }

    /**
     * Clears for incomplete suppliers.
     */
    public function clearForIncompleteSuppliers(array $productIds = []): bool
    {
        $query = InventoryForecastItem::with([])
            ->where('complete_for_supplier', false);
        if (! empty($query)) {
            $query = $query->whereIn('product_id', $productIds);
        }

        return $query->delete();
    }

    /**
     * @param $supplierId
     * @param  array  $productIds
     * @return void
     */
    public function clearForSupplier($supplierId, array $productIds = []): void
    {
        $query = InventoryForecastItem::with([])
            ->where('supplier_id', $supplierId);
        if (! empty($productIds)) {
            $query = $query->whereIn('product_id', $productIds);
        }
        $query->delete();
    }

    /**
     * @param  int  $supplierId
     * @param  int  $daysOfSalesHistory
     * @param  int  $maxProjectionDays
     * @param  int|null  $leadtime
     * @param  int|null  $minimumOrderQuantity
     * @return InventoryForecast|Model
     */
    public function createForecast(
        int $supplierId,
        int $daysOfSalesHistory,
        int $maxProjectionDays,
        ?int $leadtime = 0,
        ?int $minimumOrderQuantity = 0
    ): InventoryForecast|Model
    {
        return InventoryForecast::with([])->create([
            'forecast_date' => now(Helpers::getAppTimezone() ?: 'utc')->format('Y-m-d'),
            'supplier_id' => $supplierId,
            'days_of_sales_history' => $daysOfSalesHistory,
            'max_projection_days' => $maxProjectionDays,
            'leadtime' => $leadtime,
            'minimum_order_quantity' => $minimumOrderQuantity,
        ]);
    }


    /**
     * Updates purchase order lines for products.
     * This allows us to indicate that a purchase order
     * has been processed for the forecast already.
     */
    public function setPurchaseOrderLines(array $products, Collection $purchaseOrderLines)
    {
        $purchaseOrderLines->each(function (PurchaseOrderLine $line) use ($products) {
            if ($line->product_id && in_array($line->product_id, collect($products)->pluck('product_id')->toArray())) {
                InventoryForecastItem::with([])->where('product_id', $line->product_id)->update(['purchase_order_line_id' => $line->id]);
            }
        });
    }

    /**
     * @param  array  $results
     * @return bool
     */
    public function recordAll(array $results): bool
    {
        return InventoryForecastItem::with([])->insert($results);
    }

    /**
     * @return array
     */
    public function getTracking(): array
    {
        $tracking = Redis::get(ForecastManager::INVENTORY_FORECASTING_BATCH);
        if ($tracking) {
            return json_decode($tracking, true);
        }
        return [];
    }

    /**
     * @return array
     */
    public function getTotalCost(): array
    {
        $result = DB::table('inventory_forecast_items')
            ->selectRaw('
                inventory_forecast_items.tenant_currency_code as currency_code, 
                sum(inventory_forecast_items.unit_cost_in_tenant_currency * (inventory_forecast_items.suggested_purchase_quantity - coalesce(linked_purchase_order_lines.total_quantity_purchased, 0))) as total_cost_in_tenant_currency
            ')
            ->leftJoinSub($this->getLinkedPurchaseOrderLinesQuery(), 'linked_purchase_order_lines', 'linked_purchase_order_lines.id', 'inventory_forecast_items.id')
            ->join('inventory_forecasts', 'inventory_forecasts.id', '=', 'inventory_forecast_items.inventory_forecast_id')
            ->whereDate('forecast_date', now()->format('Y-m-d'))
            ->groupBy('inventory_forecast_items.tenant_currency_id', 'inventory_forecast_items.tenant_currency_code')
            ->first();

        if($result){
            return [
                'currency_code' => $result->currency_code,
                'total_cost_in_tenant_currency' => round($result->total_cost_in_tenant_currency, 2)
            ];
        }
        return [];
    }

    private function getLinkedPurchaseOrderLinesQuery(): Builder
    {
        return InventoryForecastItem::with([])
            ->selectRaw('
                inventory_forecast_items.id,
                sum(purchase_order_lines.quantity) as total_quantity_purchased
            ')
            ->join('forecast_item_po_line_links', 'forecast_item_po_line_links.forecast_item_id', '=', 'inventory_forecast_items.id')
            ->join('purchase_order_lines', 'purchase_order_lines.id', '=', 'forecast_item_po_line_links.purchase_order_line_id')
            ->groupBy('inventory_forecast_items.id');
    }

}
