<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 7/16/20
 * Time: 9:40 AM.
 */

namespace App\Repositories;

use App\Events\InventoryAdjusted;
use App\Exceptions\InsufficientStockException;
use App\Helpers;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\PurchaseOrderLine;
use App\Models\Warehouse;
use App\Models\WarehouseTransferLine;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use InvalidArgumentException;
use Throwable;

/**
 * Class InventoryAdjustmentRepository.
 */
class InventoryAdjustmentRepository
{
    /**
     * @throws Throwable
     */
    public function adjustInventory(array $data): InventoryAdjustment
    {
        return DB::transaction(function () use ($data) {
            // Get the location ID of the warehouse
            $warehouseLocationId = $this->getWarehouseLocationIdForAdjustment($data);

            // Create inventory adjustment
            $adjustment = $this->createInventoryAdjustment($data);

            if ($data['quantity'] < 0) {
                $this->reduceInventory(-$adjustment->quantity, $adjustment, $warehouseLocationId);
            } else {
                $unitCost = $data['unit_cost'] ?? null;
                $this->increaseInventory($adjustment, $warehouseLocationId, $unitCost);
            }

            // fire the event
            event(new InventoryAdjusted($adjustment));

            return $adjustment;
        });
    }

    /**
     * @throws InsufficientStockException
     */
    private function reduceInventory($requiredQuantity, InventoryAdjustment $adjustment, $warehouseLocationId)
    {
        // get the product to check its quantity
        $product = Product::with('activeFifoLayers')->findOrFail($adjustment->product_id);

        $remainQuantity = 0;
        do {
            if ($remainQuantity) {
                $product->load('activeFifoLayers');
            }

            $activeFifoLayer = $product->getCurrentFifoLayerForWarehouse($adjustment->warehouse);
            if (! $activeFifoLayer) {
                // There are no active fifo layers to fulfill the required quantity,
                // we abort
                throw new InsufficientStockException($product->id);
                break;
            } else {
                /**
                 * Add fulfilled quantity to fifo layer.
                 */
                $fifoLayerTotalFulfilled = $activeFifoLayer->fulfilled_quantity + $requiredQuantity;
                if ($activeFifoLayer->original_quantity < $fifoLayerTotalFulfilled) {
                    // The fifo layer doesn't have enough quantity,
                    // we'll use all the remaining quantity and use
                    // the next available fifo layer to fulfill the
                    // remaining quantity.
                    $remainQuantity = $fifoLayerTotalFulfilled - $activeFifoLayer->original_quantity;
                    $activeFifoLayer->fulfilled_quantity = $activeFifoLayer->original_quantity;
                } else {
                    // The fifo layer can fulfill the required quantity,
                    // we fulfill all the quantity required
                    $remainQuantity = 0;
                    $activeFifoLayer->fulfilled_quantity = $fifoLayerTotalFulfilled;
                }

                // Save the active fifo layer
                $activeFifoLayer->save();

                // create a new inventory movement for the adjustment
                $this->createInventoryMovementForAdjustment(
                    $adjustment,
                    $warehouseLocationId,
                    $product,
                    null,
                    -($remainQuantity === 0 ? $requiredQuantity : $requiredQuantity - $remainQuantity)
                );
            }

            $requiredQuantity = $remainQuantity;
        } while ($remainQuantity !== 0);
    }

    private function increaseInventory(InventoryAdjustment $adjustment, $warehouseLocationId, $unitCost = null): void
    {
        // create a new fifo layer
        // Get unit cost for the fifo layer
        $unitCost = $unitCost ?: $this->getProductAverageCost($adjustment->product_id);

        // Add a new fifo layer to the adjustment.
        $fifoLayer = $this->addFifoLayerToAdjustment($adjustment, $unitCost);

        // store the inventory movement of adjustment
        $this->createInventoryMovementForAdjustment(
            $adjustment,
            $warehouseLocationId,
            null,
            $fifoLayer
        );
        (new UpdateProductsInventoryAndAvgCost([$adjustment->product_id]))->handle();
    }

    private function getProductAverageCost($productId): float
    {
        $product = Product::with(['activeFifoLayers'])->findOrFail($productId);
        return $product->current_fifo_layer->avg_cost;
    }

    public function updateAdjustment(InventoryAdjustment $adjustment, array $data): InventoryAdjustment
    {
        // Update relations based on what's being updated
        if (isset($data['warehouse_id']) && $data['warehouse_id'] !== $adjustment->warehouse_id) {
            $this->setMovementsWarehouse($adjustment, $data);
        }

        if (isset($data['adjustment_date']) && $data['adjustment_date'] !== $adjustment->adjustment_date) {
            $this->setMovementsDate($adjustment, $data);
        }

        if (isset($data['quantity']) && $data['quantity'] !== $adjustment->quantity) {
            $unitCost = isset($data['unit_cost']) ? $data['unit_cost'] : null;
            $this->updateAdjustmentQuantity($adjustment, $data, $unitCost);
        }

        // Update the inventory adjustment
        $adjustment->fill($data);
        $adjustment->save();

        return $adjustment;
    }

    private function updateAdjustmentQuantity(InventoryAdjustment $adjustment, array $data, ?float $unitCost = null)
    {
        $changeInQuantity = $data['quantity'] - $adjustment->quantity;
        if ($changeInQuantity > 0) {
            // Increase adjustment effect by the change
            $this->increaseAdjustmentQuantity($adjustment, $changeInQuantity, $unitCost);
        } else {
            $this->reduceInventory(
                -$changeInQuantity,
                $adjustment,
                $adjustment->warehouse->defaultLocation->id
            );
        }
    }

    private function increaseAdjustmentQuantity(InventoryAdjustment $adjustment, $increaseInQuantity, ?float $unitCost = null)
    {
        // Get the fifo layer for the adjustment and update it's
        // original quantity as well as its total cost.
        if ($adjustment->quantity < 0) {
            // The adjustment was a reduction in stock, this means the
            // attached fifo layer is on the inventory movement and not owned by the adjustment.
            $fifoLayer = $adjustment->inventoryMovements()->firstOrFail()->fifo_layer;
            // The total cost is calculated based on the value of the current fifo layer.
            // Any new unit cost provided is ignored since it's not applicable.
            $totalCost = $fifoLayer->total_cost + $increaseInQuantity * $fifoLayer->avg_cost;
        } else {
            // For positive adjustments, the fifo layer is directly linked to
            // the adjustment.
            $fifoLayer = $adjustment->fifoLayers()->firstOrFail();
            // Since the adjustment owns the fifo layer, unless a new unit cost is provided in the update,
            // we'll use the average cost on the fifo layer to calculate the new total cost of the fifo layer.
            // Whatever unit cost is used, it has to apply to the entire new original quantity of the fifo layer.
            $totalCost = ($fifoLayer->original_quantity + $increaseInQuantity) * ($unitCost ?? $fifoLayer->avg_quantity);
        }

        $currentTotalCost = $fifoLayer->total_cost;

        $fifoLayer->original_quantity += $increaseInQuantity;
        $fifoLayer->total_cost = $totalCost;

        // Update product average cost before saving fifo layer
        Helpers::updateProductAverageCost(
            $fifoLayer->product_id,
            $totalCost - $currentTotalCost,
            $increaseInQuantity
        );

        // Save the fifo layer
        $fifoLayer->save();

        // Update the quantity of the inventory movement
        $movement = $adjustment->inventoryMovements()->first();
        $movement->quantity += $increaseInQuantity;
        $movement->save();
    }

    private function setMovementsWarehouse(InventoryAdjustment $adjustment, $data)
    {
        $warehouse = [
            'warehouse_id' => $data['warehouse_id'],
            'warehouse_location_id' => $this->getWarehouseLocationIdForAdjustment($data),
        ];
        $adjustment->inventoryMovements()->update($warehouse);
    }

    private function setMovementsDate(InventoryAdjustment $adjustment, array $data)
    {
        $adjustment->inventoryMovements()->update(['inventory_movement_date' => $data['adjustment_date']]);
    }

    private function addFifoLayerToAdjustment(InventoryAdjustment $adjustment, $unitCost): FifoLayer
    {
        $fifoLayer = new FifoLayer();

        $fifoLayer->fifo_layer_date = $adjustment->adjustment_date;
        $fifoLayer->product_id = $adjustment->product_id;
        $fifoLayer->original_quantity = $adjustment->quantity;
        $fifoLayer->total_cost = $adjustment->quantity * $unitCost;

        // Update the product's average cost before adding the fifo layer
        Helpers::updateProductAverageCostFifoLayer($fifoLayer->product_id, $fifoLayer);

        // Add the fifo layer to the adjustment
        $adjustment->fifoLayers()->save($fifoLayer);

        return $fifoLayer;
    }

    /**
     * @return int|mixed
     */
    private function getWarehouseLocationIdForAdjustment(array $adjustment)
    {
        if (! empty($adjustment['warehouse_location_id'])) {
            return $adjustment['warehouse_location_id'];
        }
        $warehouse = Warehouse::with(['defaultLocation'])->findOrFail($adjustment['warehouse_id']);

        return $warehouse->defaultLocation->id;
    }

    private function createInventoryMovementForAdjustment(
        InventoryAdjustment $adjustment,
        $warehouseLocationId,
        ?Product $product = null,
        ?FifoLayer $fifoLayer = null,
        ?int $quantity = null
    ): InventoryMovement {
        if (! $product && ! $fifoLayer) {
            throw new InvalidArgumentException('Product or fifo layer must be provided');
        }

        // store the inventory movement of adjustment
        $inventoryMovement = new InventoryMovement();
        $inventoryMovement->inventory_movement_date = $adjustment->adjustment_date;
        $inventoryMovement->product_id = $adjustment->product_id;
        $inventoryMovement->quantity = $quantity ?? $adjustment->quantity;
        $inventoryMovement->type = InventoryMovement::TYPE_ADJUSTMENT;
        $inventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
        $inventoryMovement->warehouse_id = $adjustment->warehouse_id;
        $inventoryMovement->warehouse_location_id = $warehouseLocationId;
        $inventoryMovement->fifo_layer = $fifoLayer ? $fifoLayer->id : $product->getCurrentFifoLayerForWarehouse($adjustment->warehouse)?->id;
        $adjustment->inventoryMovements()->save($inventoryMovement);

        return $inventoryMovement;
    }

    public function createInventoryAdjustment(array $data): InventoryAdjustment
    {
        $id = $data['id'] ?? null;

        /** @var InventoryAdjustment $inventoryAdjustment */
        $inventoryAdjustment = InventoryAdjustment::query()->updateOrCreate(['id' => $id], $data);

        return $inventoryAdjustment;
    }

    public function getNegativeAdjustmentsForReceiptLines(): Collection
    {
        return InventoryAdjustment::with('inventoryMovements')
            ->where('quantity', '<', 0)
            ->whereHasMorph('link', [PurchaseOrderLine::class, WarehouseTransferLine::class])
            ->get();
    }
}
