<?php

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Helpers;
use App\Models\FinancialAlert;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use Illuminate\Support\Collection;

class FinancialAlertRepository extends AbstractRepository
{
    public function getSalesOrderLinesWithAlerts(?Collection $salesOrderLines = null): Collection
    {
        $query = SalesOrderLine::with('salesOrderLineFinancial')
            ->whereHas('salesOrderLineFinancial', function ($query) {
                $query->where(function ($query) {
                    $query->where('profit', '<', 0);
                    if (Helpers::setting(Setting::KEY_FINANCIAL_ALERTS_LOW_MARGIN_THRESHOLD) > 0)
                    {
                        $query->where('profit_margin', '<', Helpers::setting(Setting::KEY_FINANCIAL_ALERTS_LOW_MARGIN_THRESHOLD) / 100);
                    }
                });
            });


        if ($salesOrderLines)
        {
            $query->whereIn('id', $salesOrderLines->pluck('id'));
        }

        return $query->get();
    }

    public function deleteResolvedAlerts(): void
    {
        FinancialAlert::query()
        ->whereHas('salesOrderLineFinancial', function ($query) {
            $query->where('profit', '>=', 0);
            if (Helpers::setting(Setting::KEY_FINANCIAL_ALERTS_LOW_MARGIN_THRESHOLD) > 0)
            {
                $query->where('profit_margin', '>=', Helpers::setting(Setting::KEY_FINANCIAL_ALERTS_LOW_MARGIN_THRESHOLD) / 100);
            }
        })->delete();
    }

    public function getDashboardData(): array
    {
        return FinancialAlert::query()
            ->archived(0)
            ->selectRaw("alert_type, count(id) as count")
            ->groupBy('alert_type')
            ->get()
            ->toArray();
    }

    public function getLastFinancialAlertsForNotification(string $date): Collection
    {
        return FinancialAlert::where('created_at', '>', $date)
            ->limit(6) // if the count is less than 6 will send immediately
            ->get();
    }
}