<?php

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Models\ReportingDailyFinancial;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Redis;
use Illuminate\Support\LazyCollection;

class DailyFinancialRepository extends AbstractRepository
{
    private function getInvalidatedDailyFinancialsQuery(array $days = []): Builder
    {
        $query = ReportingDailyFinancial::query()
            ->whereIn('date', Redis::connection('cache')->smembers(ReportingDailyFinancial::INVALID_FINANCIALS_KEY));

        if (! empty($days)) {
            $query->whereIn('date', $days);
        }

        return $query;
    }

    public function getInvalidatedDays(): array
    {
        $days = Redis::connection('cache')->smembers(ReportingDailyFinancial::INVALID_FINANCIALS_KEY);
        sort($days);
        return $days;
    }

    public function hasInvalidatedDailyFinancials(): bool
    {
        return $this->countInvalidatedDailyFinancials() > 0;
    }

    public function countInvalidatedDailyFinancials(): int
    {
        return $this->getInvalidatedDailyFinancialsQuery()->count();
    }

    public function getInvalidatedDailyFinancials(array $days = []): LazyCollection
    {
        return $this->getInvalidatedDailyFinancialsQuery($days)->cursor();
    }

    public function invalidateForDates(string|array $dates): void
    {
        $dates = Arr::wrap($dates);
        if (empty($dates)) {
            return;
        }
        Redis::connection('cache')->sadd(ReportingDailyFinancial::INVALID_FINANCIALS_KEY, ...$dates);
    }

    public function deleteAllDailyReportsForProducts(array $productIds): void
    {
        if (! empty($productIds)) {
            ReportingDailyFinancial::query()->whereIn('product_id', $productIds)->delete();
        }
    }

    public function purgeInvalidDailyFinancials(array $days = []): void
    {
        $query = ReportingDailyFinancial::query()
            ->whereIn('date', Redis::connection('cache')->smembers(ReportingDailyFinancial::INVALID_FINANCIALS_KEY));

        if (! empty($days)) {
            $query->whereIn('date', $days);
        }

        $query->delete();
    }

    public function clearInvalidDailyFinancialsCache(?array $dates = null): void
    {
        if (! empty($dates)) {
            customlog('daily_financials', 'clearInvalidDailyFinancialsCache for dates', $dates);
            customlog('daily_financials', 'invalid financial dates count before clearing: ' . Redis::connection('cache')->scard(ReportingDailyFinancial::INVALID_FINANCIALS_KEY));
            foreach($dates as $date) {
                Redis::connection('cache')->srem(ReportingDailyFinancial::INVALID_FINANCIALS_KEY, $date);
            }
            customlog('daily_financials', 'invalid financial dates count after clearing: ' . Redis::connection('cache')->scard(ReportingDailyFinancial::INVALID_FINANCIALS_KEY));
        } else {
            customlog('daily_financials', 'clearInvalidDailyFinancialsCache for all');
            Redis::connection('cache')->del(ReportingDailyFinancial::INVALID_FINANCIALS_KEY);
        }
    }

    public function deleteEmptyDailyFinancials(): void
    {
        ReportingDailyFinancial::query()
            ->whereQuantity(0)
            ->whereNumOrders(0)
            ->whereTotalRevenue(0)
            ->whereTotalCost(0)
            ->whereProfit(0)
            ->delete();
    }
}
