<?php

namespace App\Repositories;

use App\Abstractions\AbstractRepository;
use App\Data\CustomerData;
use App\Models\Address;
use App\Models\Customer;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Spatie\LaravelData\Attributes\DataCollectionOf;
use Spatie\LaravelData\Data;
use Spatie\LaravelData\DataCollection;
use Spatie\LaravelData\Optional;

/**
 * Class UserRepository.
 */
class CustomerRepository extends AbstractRepository
{
    protected string $modelClassName = \App\Models\Customer::class;

    public function findOneByEmail(string $email): ?Customer
    {
        $query = 'SELECT * FROM `customers` WHERE `customers`.`email` = ?

            UNION SELECT * FROM `customers` WHERE `customers`.`default_shipping_address_id` IN (SELECT `addresses`.`id` FROM `addresses` WHERE `addresses`.`email` = ?)

            UNION SELECT * FROM `customers` WHERE `customers`.`default_billing_address_id` IN (SELECT `addresses`.`id` FROM `addresses` WHERE `addresses`.`email` = ?)

            UNION SELECT * FROM `customers` WHERE `customers`.`id` IN (
                SELECT `addressables`.`addressable_id` FROM `addressables`
                WHERE `addressables`.`address_id` IN (SELECT `addresses`.`id` FROM `addresses` WHERE `addresses`.`email` = ?)
                AND `addressables`.`addressable_type` = ?
            )

            LIMIT 1;';

        $bindings = [
            $email,
            $email,
            $email,
            $email,
            $this->modelClassName,
        ];

        $results = DB::select(DB::raw($query)->getValue(DB::getQueryGrammar()), $bindings);

        if (is_array($results) && count($results)) {
            $customer = (new Customer())->setRawAttributes((array) $results[0]);
            $customer->exists = true;

            return $customer;
        }

        return null;
    }

    public function findOneByAddress(array $address): ?Customer
    {
        if (! array_key_exists('name', $address) ||
            ! array_key_exists('zip', $address) ||
            ! array_key_exists('address1', $address)) {
            return false;
        }

        $query = 'SELECT * FROM `customers`
            WHERE (`customers`.`name` = ?
            AND `customers`.`zip` = ?
            AND `customers`.`address1` = ?)
                   UNION
                  SELECT * FROM `customers`
            WHERE `customers`.`default_shipping_address_id` IN (
              SELECT `addresses`.`id` FROM `addresses`
                WHERE `name` = ?
                AND `zip` = ?
                AND `address1` = ?)
                   UNION
                  SELECT * FROM `customers`
            WHERE `customers`.`default_billing_address_id` IN (
              SELECT `addresses`.`id` FROM `addresses`
                WHERE `name` = ?
                AND `zip` = ?
                AND `address1` = ?)
                    UNION
                    SELECT * FROM `customers`
            WHERE `customers`.`id` IN (
              SELECT `addressables`.`addressable_id` FROM `addressables`
                WHERE `addressables`.`address_id` IN (
                SELECT `addresses`.`id` FROM `addresses`
                  WHERE `name` = ?
                  AND `zip` = ?
                  AND `address1` = ?)
                AND `addressables`.`addressable_type` = ?)
            LIMIT 1;';

        $bindings = [
            $address['name'],
            $address['zip'],
            $address['address1'],

            $address['name'],
            $address['zip'],
            $address['address1'],

            $address['name'],
            $address['zip'],
            $address['address1'],

            $address['name'],
            $address['zip'],
            $address['address1'],

            $this->modelClassName,
        ];

        $results = DB::select(DB::raw($query)->getValue(DB::getQueryGrammar()), $bindings);

        if (is_array($results) && count($results)) {
            $customer = (new Customer())->setRawAttributes((array) $results[0]);
            $customer->exists = true;

            return $customer;
        }

        return null;
    }

    public function saveWithRelations(#[DataCollectionOf(CustomerData::class)] DataCollection|Data $data): Collection
    {
        if ($data instanceof Data) {
            $data = Data::collection([$data]);
        }

        $data = $data->toCollection();

        $shippingAddressCollection = app(AddressRepository::class)->save($data->pluck('default_shipping_address'), Address::class);
        $billingAddressCollection = app(AddressRepository::class)->save($data->pluck('default_billing_address'), Address::class);

        $customerCollection = $data->map(function (CustomerData $customerDto) use ($shippingAddressCollection, $billingAddressCollection) {
            if (!$customerDto->default_shipping_address instanceof Optional) {
                $shippingAddress = $shippingAddressCollection
                    ->where('name', $customerDto->default_shipping_address->name)
                    ->where('address1', $customerDto->default_shipping_address->address1)
                    ->where('city', $customerDto->default_shipping_address->city)
                    ->where('province', $customerDto->default_shipping_address->province)
                    ->where('country_code', $customerDto->default_shipping_address->country_code)
                    ->where('zip', $customerDto->default_shipping_address->zip)
                    ->first();

                $customerDto->default_shipping_address_id = @$shippingAddress['id'];
            }

            if (!$customerDto->default_billing_address instanceof Optional) {
                $billingAddress = $billingAddressCollection
                    ->where('name', $customerDto->default_billing_address->name)
                    ->where('address1', $customerDto->default_billing_address->address1)
                    ->where('city', $customerDto->default_billing_address->city)
                    ->where('province', $customerDto->default_billing_address->province)
                    ->where('country_code', $customerDto->default_billing_address->country_code)
                    ->where('zip', $customerDto->default_billing_address->zip)
                    ->first();

                $customerDto->default_billing_address_id = @$billingAddress['id'];
            }

            return CustomerData::from(new Customer($customerDto->toArray()));
        });

        return $this->save($customerCollection, Customer::class)
            ->map(function ($customer) use ($customerCollection) {
                $customer = Customer::find($customer['id']);
                $customer->default_shipping_address_id = $customerCollection->where('name', $customer->name)->first()->default_shipping_address_id;
                $customer->default_billing_address_id = $customerCollection->where('name', $customer->name)->first()->default_billing_address_id;
                $customerDto = CustomerData::from($customer);
                if ($customer->salesChannelOrigin?->integrationInstance->isAmazonInstance())
                {
                    $customerDto = $customerDto->encryptPii();
                }
                return $customerDto->toArray();
            });
    }
}
