<?php

namespace App\Repositories;

use App\Collections\BackorderQueueCollection;
use App\DTO\BulkImportResponseDto;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Models\BackorderQueue;
use App\Models\BackorderQueueCoverage;
use App\Models\BackorderQueueRelease;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryAssemblyLine;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrderLine;
use App\Models\StockTakeItem;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\Warehouse;
use App\Models\WarehouseTransferShipmentReceiptLine;
use App\Services\InventoryManagement\BackorderManager;
use App\Services\InventoryManagement\InventoryReductionCause;
use Closure;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use InvalidArgumentException;

/**
 * Class BackorderQueueRepository.
 */
class BackorderQueueRepository
{
    /**
     * Gets the last priority of the queue.
     * When there is no priority, the default is 0.
     */
    public static function getLastPriority(int $productId): int
    {
        /** @var BackorderQueue $lastQueue */
        $lastQueue = BackorderQueue::with([])
            ->whereColumn('backordered_quantity', '>', 'released_quantity')
            ->whereNotNull('priority')
            ->orderBy('priority', 'desc')
            ->whereHas('salesOrderLine.product', function (Builder $builder) use ($productId) {
                return $builder->where('id', $productId);
            })->first();

        if ($lastQueue && $lastQueue->priority) {
            return $lastQueue->priority;
        }

        return 0;
    }

    public function switchSupplier(BackorderQueue $queue, int $supplierId): BackorderQueue
    {
        if ($queue->supplier_id == $supplierId) {
            return $queue;
        }

        return DB::transaction(function () use ($supplierId, $queue) {
            // Ensure that supplier sources the product
            // in the queue.
            $supplierProduct = SupplierProduct::with([])
                ->where('supplier_id', $supplierId)
                ->where('product_id', $queue->salesOrderLine->product_id)
                ->first();

            if (! $supplierProduct) {
                throw new InvalidArgumentException('Supplier id: '.
                                            $supplierId.' does not source the product on the queue id: '.
                                            $queue->id);
            }

            // Get the PO lines in the coverages to be used for the old supplier.
            /** @var EloquentCollection $coverages */
            $coverages = $queue->backorderQueueCoverages;

            $queue->supplier_id = $supplierId;
            $queue->save();

            $queue->backorderQueueCoverages->each(function (BackorderQueueCoverage $coverage) {
                // This ensures that coverages are updated for the new supplier.
                $coverage->delete();
            });
            // Attempt to move up coverages to fill in any gaps.
            // dispatch_now(new MoveBackorderCoveragesUp($queue->unallocated_backorder_quantity, $queue));

            // Finally, we create coverages for the old supplier
            $coverages->each(function (BackorderQueueCoverage $coverage) {
                $coverage->purchaseOrderLine->refresh();
            });

            dispatch(new SyncBackorderQueueCoveragesJob($coverages->pluck('purchase_order_line_id')->toArray()));

            return $queue;
        });
    }

    public function switchSuppliers(array $queues): array
    {
        $errors = [];

        foreach ($queues as $key => $payload) {
            /** @var BackorderQueue $queue */
            $queue = BackorderQueue::with([])
                ->findOrFail($payload['queue_id']);
            try {
                $this->switchSupplier(
                    $queue,
                    $payload['supplier_id']
                );
            } catch (InvalidArgumentException $e) {
                $errors[] = [
                    'key' => $key,
                    'message' => $e->getMessage(),
                ];
            }
        }

        return $errors;
    }

    public function withSupplier(
        int $quantity,
        $salesOrderLine,
        InventoryReductionCause $reductionCause = InventoryReductionCause::INCREASED_NEGATIVE_EVENT,
        ?int $supplierId = null
    ): BackorderQueue {
        if ($quantity <= 0) {
            throw new InvalidArgumentException('Backorder quantity must be positive.');
        }

        if (! ($salesOrderLine instanceof SalesOrderLine)) {
            $salesOrderLine = SalesOrderLine::with([])->findOrFail($salesOrderLine);
        }

        $salesOrderLineId = $salesOrderLine->id;
        $supplierId = $supplierId ?? ($salesOrderLine->product->defaultSupplierProduct ? $salesOrderLine->product->defaultSupplierProduct->supplier_id : null);

        /**
         * Ensure each line has unique queue.
         * Therefore, if the line already has a queue, we
         * use it and recycle it if inactive.
         */
        /** @var BackorderQueue $queue */
        $queue = BackorderQueue::with([])
            ->where('sales_order_line_id', $salesOrderLineId)
            ->first();

        if ($queue) {
            // For an existing queue, how we treat the quantity depends on
            // whether the backorder is needed because an originally positive
            // event quantity is being reduced (e.g. PO receipt qty is reducing)
            // or if the backorder was caused by a negative event (e.g. increase in sales order line qty).
            if ($reductionCause == InventoryReductionCause::REDUCED_POSITIVE_EVENT) {
                // Positive event quantity is being reduced, we first prioritize reducing
                // any released quantity before increasing the backordered quantity any further.
                if ($queue->released_quantity > 0) {
                    $remainder = $queue->released_quantity - $quantity;
                    $queue->released_quantity = max($remainder, 0);
                    $quantity = $queue->backordered_quantity;
                    if ($remainder < 0) {
                        // Release quantity was not enough for the extra quantity.
                        $quantity += abs($remainder);
                    }
                }
            } else {
                // A negative event quantity is increasing, so we simply increase the backordered quantity.
                $quantity += $queue->backordered_quantity;
            }
        } else {
            $queue = new BackorderQueue;
        }

        // We can't backorder more that the sales order line quantity.
        $quantity = min($quantity, $salesOrderLine->quantity);

        $queue->supplier_id = $supplierId;
        $queue->backordered_quantity = $quantity;
        $queue->priority = self::getLastPriority($salesOrderLine->product_id) + 1;
        $queue->backorder_date = now();
        $queue->sales_order_line_id = $salesOrderLineId;
        $queue->save();

        return $queue;
    }

    public function getAvailablePurchaseOrderLinesForCoverages(BackorderQueue $backorderQueue): EloquentCollection
    {
        return PurchaseOrderLine::query()
            ->selectRaw(
                "purchase_order_lines.id, LEAST(purchase_order_lines.unreceived_quantity, purchase_order_lines.quantity - COALESCE(SUM(bqc.unreleased_quantity), 0)) as quantity_available"
            )
            ->leftJoin("backorder_queue_coverages as bqc", function ($join) {
                $join->on("purchase_order_lines.id", "=", "bqc.purchase_order_line_id");
            })
            ->whereHas("purchaseOrder", function ($query) use ($backorderQueue) {
                $query->where('order_status', PurchaseOrder::STATUS_OPEN);
                $query->where(
                    "destination_warehouse_id",
                    $backorderQueue->salesOrderLine->warehouse_id
                );
            })
            ->where("product_id", $backorderQueue->salesOrderLine->product_id)
            ->groupBy("purchase_order_lines.id") // Ensure grouping by purchase order line
            ->having("quantity_available", ">", 0)
            ->get();
    }


    /**
     * Gets the backorder queue for the
     * given product id.
     *
     *
     * @return BackorderQueue[]
     */
    public function getActiveQueuesForProduct(int $productId, ?int $warehouseId): EloquentCollection
    {
        return BackorderQueue::with(['salesOrderLine', 'backorderQueueReleases'])
            ->whereHas('salesOrderLine', function (Builder $builder) use ($productId, $warehouseId) {
                $builder->where('product_id', $productId);
                if ($warehouseId) {
                    $builder->where('warehouse_id', $warehouseId);
                }
            })
            ->active()
            ->orderBy('priority')
            ->get();
    }

    /**
     * Fetches the queue immediately after the given priority.
     */
    public function firstAfterPriority(int $priority, int $productId): ?BackorderQueue
    {
        return BackorderQueue::with([])
            ->active()
            ->where('priority', '>', $priority)
            ->whereHas('salesOrderLine.product', function (Builder $builder) use ($productId) {
                return $builder->where('id', $productId);
            })
            ->orderBy('priority')
            ->first();
    }

    /**
     * @param  null  $supplierId
     * @return BackorderQueue|Builder
     */
    public function afterPriority(
        int $priority,
        int $productId,
        string $priorityDirection = 'asc',
        $supplierId = null
    ) {
        $query = BackorderQueue::with([])
            ->active()
            ->where('priority', '>', $priority);

        if ($supplierId) {
            $query = $query->where('supplier_id', $supplierId);
        }
        $query = $query->whereHas('salesOrderLine', function (Builder $builder) use ($productId) {
            return $builder->where('product_id', $productId);
        });

        return $query
            ->orderBy('priority', $priorityDirection)
            ->get();
    }

    /**
     * Fetches the queue immediately before the given priority.
     */
    public function firstBeforePriority(int $priority, int $productId): ?BackorderQueue
    {
        return BackorderQueue::with([])
            ->active()
            ->where('priority', '<', $priority)
            ->whereHas('salesOrderLine.product', function (Builder $builder) use ($productId) {
                return $builder->where('id', $productId);
            })
            ->latest('priority')
            ->first();
    }

    /**
     * Gets the queue at the bottom of the priority list.
     */
    public function lastPriority(int $productId): ?BackorderQueue
    {
        return BackorderQueue::with([])
            ->active()
            ->whereHas('salesOrderLine.product', function (Builder $builder) use ($productId) {
                return $builder->where('id', $productId);
            })
            ->latest('priority')
            ->first();
    }

    /**
     * Gets the queue at the beginning of the priority list.
     */
    public function firstPriority(int $productId): ?BackorderQueue
    {
        return BackorderQueue::with([])
            ->active()
            ->whereHas('salesOrderLine.product', function (Builder $builder) use ($productId) {
                return $builder->where('id', $productId);
            })
            ->orderBy('priority')
            ->first();
    }

    /**
     * Increments the priorities by the given amount.
     */
    public function incrementPriorities(int $productId, ?array $queues = [], int $value = 1, array $except = [])
    {
        $query = BackorderQueue::with([])
            ->active()
            ->whereHas('salesOrderLine.product', function (Builder $builder) use ($productId) {
                return $builder->where('id', $productId);
            })
            ->whereNotIn('id', $except);

        if (! empty($queues)) {
            $query = $query->whereIn('id', $queues);
        }
        $query->increment('priority', $value);
    }

    public function incrementPrioritiesBefore($referencePriority, int $productId, int $value = 1, array $except = [])
    {
        $beforeReferencePriority = BackorderQueue::with([])
            ->active()
            ->whereHas('salesOrderLine.product', function (Builder $builder) use ($productId) {
                return $builder->where('id', $productId);
            })
            ->where('priority', '<', $referencePriority)
            ->pluck('id');
        if ($beforeReferencePriority->count()) {
            $this->incrementPriorities(
                $productId,
                $beforeReferencePriority->toArray(),
                $value,
                $except
            );
        }
    }

    public function incrementPrioritiesBetween(int $priority1, int $priority2, int $productId, int $value = 1, array $except = [])
    {
        $affected = BackorderQueue::with([])
            ->active()
            ->whereHas('salesOrderLine.product', function (Builder $builder) use ($productId) {
                return $builder->where('id', $productId);
            })
            ->whereBetween('priority', Arr::sort([$priority1, $priority2]))
            ->pluck('id');
        if ($affected->count()) {
            $this->incrementPriorities(
                $productId,
                $affected->toArray(),
                $value,
                $except
            );
        }
    }

    public function decrementPrioritiesBetween(int $priority1, int $priority2, int $productId, int $value = 1, array $except = [])
    {
        $affected = BackorderQueue::with([])
            ->active()
            ->whereHas('salesOrderLine.product', function (Builder $builder) use ($productId) {
                return $builder->where('id', $productId);
            })
            ->whereBetween('priority', Arr::sort([$priority1, $priority2]))
            ->pluck('id');
        if ($affected->count()) {
            $this->decrementPriorities(
                $productId,
                $affected->toArray(),
                $value,
                $except
            );
        }
    }

    public function decrementPriorities(int $productId, array|Closure|null $queues = null, int $value = 1, array $except = [])
    {
        $query = BackorderQueue::with([])
            ->active()
            ->whereHas('salesOrderLine.product', function (Builder $builder) use ($productId) {
                return $builder->where('id', $productId);
            })
            ->whereNotIn('id', $except);

        if ($queues) {
            if ($queues instanceof Closure) {
                $query->where($queues);
            } elseif (is_array($queues) && ! empty($queues)) {
                $query->whereIn('id', $queues);
            }
        }
        $query->decrement('priority', $value);
    }

    public function decrementPrioritiesAfter($referencePriority, int $productId, int $value = 1, array $except = [])
    {
        if (! is_null($referencePriority)) {
            $this->decrementPriorities(
                $productId,
                function (Builder $builder) use ($referencePriority) {
                    $builder->where('priority', '>', $referencePriority);
                },
                $value,
                $except
            );
        }
    }

    public function findById(int $id, array $relations = []): ?BackorderQueue
    {
        return BackorderQueue::with($relations)
            ->find($id);
    }

    /**
     * @return array|array[]
     */
    public function supplierAvailabilityForQueues(array $ids)
    {
        $products = Product::with(['salesOrderLines.backorderQueue', 'suppliers'])
            ->whereHas('salesOrderLines.backorderQueue', function (Builder $builder) use ($ids) {
                return $builder->whereIn('id', $ids);
            })->get()->toArray();

        return array_map(function ($row) use ($ids) {
            return [
                'product_id' => $row['id'],
                'sku' => $row['sku'],
                'name' => $row['name'],
                'suppliers' => collect($row['suppliers'])->map(function ($supplier) {
                    return collect($supplier)->only(['id', 'name']);
                })->toArray(),
                'backorder_queue_ids' => array_intersect(collect($row['sales_order_lines'])->map(function ($line) {
                    return $line['backorder_queue']['id'];
                })->toArray(), $ids),
            ];
        }, $products);
    }

    public function empty(?int $productId = null): bool
    {
        $query = BackorderQueue::with([])
            ->active();

        if ($productId) {
            $query = $query->whereHas('salesOrderLine.product', function (Builder $builder) use ($productId) {
                return $builder->where('id', $productId);
            });
        }

        return $query->count() === 0;
    }

    public function getUnreleasedCoveragesForBackorderQueue(BackorderQueue $backorderQueue): Collection
    {
        return $backorderQueue->backorderQueueCoverages()
                              ->where('unreleased_quantity', '>', 0)
                              ->get();
    }

    public function createCoverage(int $quantity, BackorderQueue $queue, PurchaseOrderLine $orderLine): int
    {
        $quantity = min($quantity, max($orderLine->unreceived_quantity - $orderLine->backorder_queue_coverage_quantity, 0));

        if ($quantity == 0) {
            // For instance the PO line has exhausted its quantity with coverages.
            return 0;
        }

        // If the same order line has a coverage for the queue, we simply increase the covered quantity.
        /** @var BackorderQueueCoverage $coverage */
        $coverage = $queue->backorderQueueCoverages()
            ->where('purchase_order_line_id', $orderLine->id)
            ->first();

        if ($coverage) {
            $additionalCoverageQuantity = min($queue->unallocated_backorder_quantity, $quantity);
            $coverage->covered_quantity += $additionalCoverageQuantity;
            $coverage->save();
            $quantity = $additionalCoverageQuantity;
        } else {
            // We create a new coverage for the order line.
            $queue->backorderQueueCoverages()
                ->create([
                    'covered_quantity' => $quantity,
                    'released_quantity' => 0,
                    'purchase_order_line_id' => $orderLine->id,
                ]);
        }

        return $quantity;
    }

    protected function compatible(BackorderQueue $queue1, BackorderQueue $queue2): bool
    {
        // Must have same sku and suppliers
        return $queue1->salesOrderLine->product_id == $queue2->salesOrderLine->product_id &&
           $queue1->supplier_id == $queue2->supplier_id;
    }

    public function swapCoverages(BackorderQueue $queue1, BackorderQueue $queue2)
    {
        // Only swap for same sku
        if (! $this->compatible($queue1, $queue2)) {
            return;
        }

        /**
         * To simplify the swaping process, we essentially
         * reset the coverages and attempt to cover the
         * high-priority (low priority value) queue first.
         *
         * This accounts for quantities, releases, etc.
         */
        DB::transaction(function () use ($queue1, $queue2) {
            $queue1->backorderQueueCoverages()->delete();
            $queue2->backorderQueueCoverages()->delete();

            /** @var BackorderManager $manager */
            $manager = app(BackorderManager::class);
            if ($queue1->priority <= $queue2->priority) {
                $manager->coverBackorderQueues(backorderQueueIds: [$queue1->id]);
                $manager->coverBackorderQueues(backorderQueueIds: [$queue2->id]);
            } else {
                $manager->coverBackorderQueues(backorderQueueIds: [$queue2->id]);
                $manager->coverBackorderQueues(backorderQueueIds: [$queue1->id]);
            }
        });
    }

    /**
     * @deprecated
     */
    public function setCoverages(BackorderQueue $queue, array $coverageIds)
    {
        foreach ($coverageIds as $coverageId) {
            /** @var BackorderQueueCoverage $coverage */
            $coverage = BackorderQueueCoverage::with([])->findOrFail($coverageId);
            /** @var BackorderQueueCoverage $existing */
            $existing = BackorderQueueCoverage::with([])
                ->where('purchase_order_line_id', $coverage->purchase_order_line_id)
                ->where('backorder_queue_id', $queue->id)
                ->first();
            if ($existing) {
                $existing->covered_quantity += $coverage->covered_quantity;
                $existing->released_quantity += $coverage->released_quantity;
                $existing->save();
                $coverage->delete();
            } else {
                $coverage->backorder_queue_id = $queue->id;
                $coverage->save();
            }
        }
    }

    public function moveCoverages(BackorderQueue $from, BackorderQueue $to)
    {
        // Only move for same sku
        if (! $this->compatible($from, $to)) {
            return;
        }

        $this->setCoverages($to, $from->backorderQueueCoverages()->pluck('id')->toArray());
    }

    public function getWarehouseProductsForQueues(array $queueIds): array
    {
        return SalesOrderLine::query()
            ->selectRaw('DISTINCT sol.product_id, sol.warehouse_id')
            ->from('sales_order_lines', 'sol')
            ->join('backorder_queues as boq', 'boq.sales_order_line_id', 'sol.id')
            ->whereIn('boq.id', $queueIds)
            ->whereNotNull('sol.warehouse_id')
            ->whereNotNull('sol.product_id')
            ->get()
            ->groupBy('warehouse_id')
            ->toArray();
    }

    public function saveBulk(BackorderQueueCollection $data): void
    {

        if($data->isEmpty()){
            return;
        }

        BackorderQueue::query()->upsert(
            $data->toArray(),
            ['uid'],
            ['backordered_quantity', 'released_quantity']
        );
    }

    public function getBackorderQueuesForSalesOrderLineIds(array $salesOrderLineIds): EloquentCollection
    {
        return BackorderQueue::query()
            ->whereIn('sales_order_line_id', $salesOrderLineIds)
            ->get();
    }

    public function getBackorderQueueReleasesForFifoLayerIds(array $fifoLayerIds): EloquentCollection
    {
        // Bypassing the initial count error
        return BackorderQueueRelease::query()
            ->where(function (Builder $query) use ($fifoLayerIds) {
                $query->whereHasMorph('link',
                    [
                        PurchaseOrderShipmentReceiptLine::class,
                        InventoryAdjustment::class,
                        StockTakeItem::class,
                        WarehouseTransferShipmentReceiptLine::class,
                        SalesCreditReturnLine::class,
                        InventoryAssemblyLine::class,
                    ],
                    function (Builder $query) use ($fifoLayerIds) {
                        $query->whereHas('inventoryMovements', function (Builder $query) use ($fifoLayerIds) {
                            $query->whereIn('inventory_movements.layer_id', $fifoLayerIds);
                            $query->where('inventory_movements.layer_type', FifoLayer::class);
                        });
                    });
            })
            ->get();
    }

    public function unreleasedUncoveredBackorders(Collection $unreceivedPurchaseOrderLinesProductWarehousePairs, ?array $backorderQueueIds = null): EloquentCollection|Collection
    {
        $query = DB::table((new BackorderQueue())->getTable())
            ->select('backorder_queues.*', 'sales_order_lines.product_id', 'sales_order_lines.warehouse_id')
            ->selectRaw('IFNULL(backorder_queue_coverages.unreleased_quantity, 0) as covered_quantity')
            ->whereColumn('backordered_quantity', '>', 'backorder_queues.released_quantity')
            ->where(function ($query) {
                // Condition for backorder queues without any coverages
                $query->whereNull('backorder_queue_coverages.covered_quantity')
                    // We should find coverages with positive unreleased quantity.
                    // Note that existing coverages may be unneeded after a non-po positive event (e.g +ve adjustment).
                    ->orWhereRaw('backorder_queue_coverages.unreleased_quantity > 0');
            })
            ->join('sales_order_lines', 'backorder_queues.sales_order_line_id', '=', 'sales_order_lines.id')
            ->join('products', 'products.id', '=', 'sales_order_lines.product_id')
            ->leftJoin('backorder_queue_coverages', 'backorder_queues.id', '=', 'backorder_queue_coverages.backorder_queue_id')
            ->where(function($query) use ($unreceivedPurchaseOrderLinesProductWarehousePairs){
                $unreceivedPurchaseOrderLinesProductWarehousePairs->each(function($pair) use ($query) {
                    $query->orWhere(function($q) use ($pair){
                        $q->where('sales_order_lines.product_id', $pair->product_id)
                        ->where('sales_order_lines.warehouse_id', $pair->destination_warehouse_id);
                    });
                });
            })
            ->where('products.type', '!=', Product::TYPE_KIT) // Kit backorders cannot be covered by POs
            ->orderBy('warehouse_id')
            ->orderBy('product_id')
            ->orderBy('priority')
            ->orderBy('backorder_date');

        if (! empty($backorderQueueIds)) {
            $query->whereIn('backorder_queues.id', $backorderQueueIds);
        }

        return $query->get();
    }

    public function getUnreleasedBackordersForProductWarehousePairs(Collection $productWarehousePairs): EloquentCollection|Collection
    {
        cache()->set('test-implicit-commit', true);
        $tempTable = 'tmp_product_warehouse_pairs_'.Str::random(10);
        $createTableQuery = <<<SQL
            CREATE TEMPORARY TABLE $tempTable (
                product_id INT,
                warehouse_id INT,
                PRIMARY KEY (product_id, warehouse_id)
            )
        SQL;
        DB::statement($createTableQuery);
        DB::table($tempTable)->insert($productWarehousePairs->toArray());

        $query = BackorderQueue::with('salesOrderLine.salesOrder', 'salesOrderLine.inventoryMovements', 'salesOrderLine.salesOrderLineLayers')
            ->select('backorder_queues.*')
            ->whereColumn('backordered_quantity', '>', 'backorder_queues.released_quantity')
            ->join('sales_order_lines', 'backorder_queues.sales_order_line_id', '=', 'sales_order_lines.id')
            ->join($tempTable, function ($join) use ($tempTable) {
                $join->on('sales_order_lines.product_id', "$tempTable.product_id")
                    ->on('sales_order_lines.warehouse_id', "$tempTable.warehouse_id");
            })
            ->where(function ($query) {
                return $query->whereDoesntHave('backorderQueueCoverages')
                    ->orWhereHas('backorderQueueCoverages', function ($q) {
                        return $q->where('is_tight', false);
                    });
            })
            ->orderBy('sales_order_lines.warehouse_id')
            ->orderBy('sales_order_lines.product_id')
            ->orderBy('priority')
            ->orderBy('backorder_date');

        return $query->get();
    }

    public function unreleasedBackorderProductWarehouseGroups(Collection $unreceivedPurchaseOrderLinesProductWarehousePairs, ?array $backorderQueueIds = null): EloquentCollection|Collection
    {
        $query = DB::table((new BackorderQueue())->getTable())
            ->select('sales_order_lines.product_id', 'sales_order_lines.warehouse_id')
            ->whereColumn('backordered_quantity', '>', 'released_quantity')
            ->join('sales_order_lines', 'backorder_queues.sales_order_line_id', '=', 'sales_order_lines.id')
            ->where(function($query) use ($unreceivedPurchaseOrderLinesProductWarehousePairs){
                $unreceivedPurchaseOrderLinesProductWarehousePairs->each(function($pair) use ($query) {
                    $query->orWhere(function($q) use ($pair){
                        $q->where('sales_order_lines.product_id', $pair->product_id)
                            ->where('sales_order_lines.warehouse_id', $pair->destination_warehouse_id);
                    });
                });
            });

        if (! empty($backorderQueueIds)) {
            $query->whereIn('backorder_queues.id', $backorderQueueIds);
        }

        $query->groupBy(['product_id', 'warehouse_id']);

        return $query->get();
    }

    public function getWarehousesNeedingBackordersForSupplier(Supplier $supplier): EloquentCollection
    {
        return BackorderQueue::query()
            ->select('sales_order_lines.warehouse_id')
            ->join('sales_order_lines', 'backorder_queues.sales_order_line_id', '=', 'sales_order_lines.id')
            ->where('supplier_id', $supplier->id)
            ->whereColumn('backordered_quantity', '>', 'released_quantity')
            ->whereNotNull('sales_order_lines.warehouse_id')
            ->groupBy('sales_order_lines.warehouse_id')
            ->get();
    }

    public function getProductQuantityNeededForSupplierAndWarehouse(Supplier $supplier, Warehouse $warehouse): EloquentCollection
    {
        return BackorderQueue::query()
            ->selectRaw('sales_order_lines.product_id, products.brand_id, sum(backordered_quantity - released_quantity) as quantity')
            ->join('sales_order_lines', 'backorder_queues.sales_order_line_id', '=', 'sales_order_lines.id')
            ->join('products', 'sales_order_lines.product_id', '=', 'products.id')
            ->where('supplier_id', $supplier->id)
            ->where('sales_order_lines.warehouse_id', $warehouse->id)
            ->whereColumn('backordered_quantity', '>', 'released_quantity')
            ->groupBy('sales_order_lines.product_id', 'products.brand_id')
            ->get();
    }

    public function reduceCoverageQtyBy(SalesOrderLine $orderLine, float|int $quantity): void
    {
        if ($orderLine->active_backordered_quantity <= 0) {
            return;
        }

        $coverages = $orderLine->backorderQueue
            ->backorderQueueCoverages()
            ->where('unreleased_quantity', '>', 0)
            ->get();

        if ($coverages->isEmpty()) {
            return;
        }

        $coveragesToDelete = [];
        $coveragesToUpdate = [];

        /** @var BackorderQueueCoverage $coverage */
        foreach ($coverages as $coverage) {
            if ($coverage->unreleased_quantity >= $quantity) {
                $updateQty = max(0, $coverage->covered_quantity - $quantity);
                $quantity -= $quantity;
            } else {
                $updateQty = max(0, $coverage->covered_quantity - $coverage->unreleased_quantity);
                $quantity -= $coverage->unreleased_quantity;
            }

            if ($updateQty == 0) {
                $coveragesToDelete[] = $coverage->id;
            } else {
                $coveragesToUpdate[] = [
                    'id' => $coverage->id,
                    'covered_quantity' => $updateQty,
                ];
            }

            if ($quantity <= 0) {
                break;
            }
        }

        batch()->update(new BackorderQueueCoverage, $coveragesToUpdate, 'id');
        BackorderQueueCoverage::query()->whereIn('id', $coveragesToDelete)->delete();

    }

    public function getNextPriority(SalesOrderLine $salesOrderLine): int
    {
        return BackorderQueue::query()
            ->whereHas('salesOrderLine', function (Builder $query) use ($salesOrderLine) {
                $query->where('product_id', $salesOrderLine->product_id);
                $query->where('warehouse_id', $salesOrderLine->warehouse_id);
            })
            ->max('priority') + 1;
    }
}
