<?php

namespace App\Repositories;

use App\Models\BackorderQueue;
use App\Models\BackorderQueueCoverage;
use App\Models\PurchaseOrderShipmentReceiptLine;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Collection;

class BackorderQueueCoverageRepository
{

    public function getExistingCoverages(BackorderQueue $queue, array $purchaseOrderLineIds): EloquentCollection
    {
        return $queue
            ->backorderQueueCoverages()
            ->whereIn('purchase_order_line_id', $purchaseOrderLineIds)
            ->get();
    }

    public function deleteUnreleased(?array $productIds = null, ?int $warehouseId = null): void
    {
        $query = BackorderQueueCoverage::query()
            ->where('released_quantity', 0);

        // If a warehouse ID is provided, join with the purchase_order table via the purchase_order_line table
        if (!empty($warehouseId)) {
            $query->joinRelationship('purchaseOrderLine.purchaseOrder')
                ->where('purchase_orders.destination_warehouse_id', $warehouseId);
        }

        // If product IDs are provided, join with the purchase_order_lines table and filter
        if (!empty($productIds)) {
            if(!empty($warehouseId)){
                // Purchase order lines are already joined, so we can filter directly
                $query->whereIn('purchase_order_lines.product_id', $productIds);
            }else{
                // Join with the purchase_order_lines table and filter
                $query->joinRelationship('purchaseOrderLine')
                    ->whereIn('purchase_order_lines.product_id', $productIds);
            }
        }

        // Delete the records that match the criteria
        $query->delete();
    }

    public function getCoveragesNeedingUpdate(): EloquentCollection
    {
        return BackorderQueueCoverage::query()
            ->selectRaw(
                'backorder_queue_coverages.backorder_queue_id, 
                backorder_queue_coverages.purchase_order_line_id,
                backorder_queue_coverages.covered_quantity,
                IFNULL(backorder_queue_releases.released_quantity, 0) as released_quantity'
            )
            ->join(
                'purchase_order_lines',
                'purchase_order_lines.id',
                '=',
                'backorder_queue_coverages.purchase_order_line_id'
            )
            ->leftJoin(
                'purchase_order_shipment_lines',
                'purchase_order_shipment_lines.purchase_order_line_id',
                '=',
                'purchase_order_lines.id'
            )
            ->leftJoin(
                'purchase_order_shipment_receipt_lines',
                'purchase_order_shipment_receipt_lines.purchase_order_shipment_line_id',
                '=',
                'purchase_order_shipment_lines.id'
            )
            ->leftJoin('backorder_queue_releases', function ($join) {
                $join
                    ->on(
                        'backorder_queue_releases.link_id',
                        '=',
                        'purchase_order_shipment_receipt_lines.id'
                    )
                    ->on(
                        'backorder_queue_releases.backorder_queue_id',
                        '=',
                        'backorder_queue_coverages.backorder_queue_id'
                    );
            })
            ->where(function ($query) {
                $query->whereNull('backorder_queue_releases.id');
                $query->where('backorder_queue_coverages.released_quantity', '!=', 0);
            })
            ->orWhere(function ($query) {
                $query->where(
                    'backorder_queue_releases.link_type',
                    PurchaseOrderShipmentReceiptLine::class
                );
                $query->whereColumn(
                    'backorder_queue_coverages.released_quantity',
                    '!=',
                    'backorder_queue_releases.released_quantity'
                );
            })
            ->get();
    }

    public function saveBulk(Collection $data): void
    {
        BackorderQueueCoverage::query()->upsert(
            $data->toArray(),
            ['backorder_queue_id', 'purchase_order_line_id'],
            ['covered_quantity', 'released_quantity']
        );
    }
}
